/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.common.cloud;

import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import org.apache.solr.common.cloud.SolrZkClient;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZkConfigManager {
    private static final Logger logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    public static final String CONFIGS_ZKNODE = "/configs";
    private final SolrZkClient zkClient;

    public ZkConfigManager(SolrZkClient zkClient) {
        this.zkClient = zkClient;
    }

    private void uploadToZK(final Path rootPath, final String zkPath) throws IOException {
        if (!Files.exists(rootPath, new LinkOption[0])) {
            throw new IOException("Path " + rootPath + " does not exist");
        }
        Files.walkFileTree(rootPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                String filename = file.getFileName().toString();
                if (filename.startsWith(".")) {
                    return FileVisitResult.CONTINUE;
                }
                String zkNode = ZkConfigManager.createZkNodeName(zkPath, rootPath, file);
                try {
                    ZkConfigManager.this.zkClient.makePath(zkNode, file.toFile(), false, true);
                }
                catch (InterruptedException | KeeperException e) {
                    throw new IOException("Error uploading file " + file.toString() + " to zookeeper path " + zkNode, SolrZkClient.checkInterrupted(e));
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                return dir.getFileName().toString().startsWith(".") ? FileVisitResult.SKIP_SUBTREE : FileVisitResult.CONTINUE;
            }
        });
    }

    private static String createZkNodeName(String zkRoot, Path root, Path file) {
        String relativePath = root.relativize(file).toString();
        String separator = root.getFileSystem().getSeparator();
        if ("\\".equals(separator)) {
            relativePath = relativePath.replaceAll("\\\\", "/");
        }
        return zkRoot + "/" + relativePath;
    }

    private void downloadFromZK(String zkPath, Path dir) throws IOException {
        try {
            List<String> files = this.zkClient.getChildren(zkPath, null, true);
            Files.createDirectories(dir, new FileAttribute[0]);
            for (String file : files) {
                List<String> children = this.zkClient.getChildren(zkPath + "/" + file, null, true);
                if (children.size() == 0) {
                    byte[] data = this.zkClient.getData(zkPath + "/" + file, null, null, true);
                    Path filename = dir.resolve(file);
                    logger.info("Writing file {}", (Object)filename);
                    Files.write(filename, data, new OpenOption[0]);
                    continue;
                }
                this.downloadFromZK(zkPath + "/" + file, dir.resolve(file));
            }
        }
        catch (InterruptedException | KeeperException e) {
            throw new IOException("Error downloading files from zookeeper path " + zkPath + " to " + dir.toString(), SolrZkClient.checkInterrupted(e));
        }
    }

    public void uploadConfigDir(Path dir, String configName) throws IOException {
        this.uploadToZK(dir, "/configs/" + configName);
    }

    public void downloadConfigDir(String configName, Path dir) throws IOException {
        this.downloadFromZK("/configs/" + configName, dir);
    }

    public List<String> listConfigs() throws IOException {
        try {
            return this.zkClient.getChildren(CONFIGS_ZKNODE, null, true);
        }
        catch (KeeperException.NoNodeException e) {
            return Collections.emptyList();
        }
        catch (InterruptedException | KeeperException e) {
            throw new IOException("Error listing configs", SolrZkClient.checkInterrupted(e));
        }
    }

    public Boolean configExists(String configName) throws IOException {
        try {
            return this.zkClient.exists("/configs/" + configName, true);
        }
        catch (InterruptedException | KeeperException e) {
            throw new IOException("Error checking whether config exists", SolrZkClient.checkInterrupted(e));
        }
    }

    public void deleteConfigDir(String configName) throws IOException {
        try {
            this.zkClient.clean("/configs/" + configName);
        }
        catch (InterruptedException | KeeperException e) {
            throw new IOException("Error checking whether config exists", SolrZkClient.checkInterrupted(e));
        }
    }

    private void copyConfigDirFromZk(String fromZkPath, String toZkPath, Set<String> copiedToZkPaths) throws IOException {
        try {
            List<String> files = this.zkClient.getChildren(fromZkPath, null, true);
            for (String file : files) {
                List<String> children = this.zkClient.getChildren(fromZkPath + "/" + file, null, true);
                if (children.size() == 0) {
                    String toZkFilePath = toZkPath + "/" + file;
                    logger.info("Copying zk node {} to {}", (Object)(fromZkPath + "/" + file), (Object)toZkFilePath);
                    byte[] data = this.zkClient.getData(fromZkPath + "/" + file, null, null, true);
                    this.zkClient.makePath(toZkFilePath, data, true);
                    if (copiedToZkPaths == null) continue;
                    copiedToZkPaths.add(toZkFilePath);
                    continue;
                }
                this.copyConfigDirFromZk(fromZkPath + "/" + file, toZkPath + "/" + file, copiedToZkPaths);
            }
        }
        catch (InterruptedException | KeeperException e) {
            throw new IOException("Error copying nodes from zookeeper path " + fromZkPath + " to " + toZkPath, SolrZkClient.checkInterrupted(e));
        }
    }

    public void copyConfigDir(String fromConfig, String toConfig) throws IOException {
        this.copyConfigDir("/configs/" + fromConfig, "/configs/" + toConfig, null);
    }

    public void copyConfigDir(String fromConfig, String toConfig, Set<String> copiedToZkPaths) throws IOException {
        this.copyConfigDirFromZk("/configs/" + fromConfig, "/configs/" + toConfig, copiedToZkPaths);
    }
}

