/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.seata.rm.tcc.api;

import org.apache.seata.common.Constants;
import org.apache.seata.common.exception.FrameworkException;
import org.apache.seata.common.util.CollectionUtils;
import org.apache.seata.common.util.StringUtils;
import org.apache.seata.core.exception.TransactionException;
import org.apache.seata.core.model.BranchStatus;
import org.apache.seata.integration.tx.api.interceptor.ActionContextUtil;
import org.apache.seata.integration.tx.api.util.JsonUtil;
import org.apache.seata.rm.DefaultResourceManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * the api of sharing business action context to tcc phase 2
 *
 */
public final class BusinessActionContextUtil {

    private BusinessActionContextUtil() {
    }

    private static final Logger LOGGER = LoggerFactory.getLogger(BusinessActionContextUtil.class);

    private static final ThreadLocal<BusinessActionContext> CONTEXT_HOLDER = new ThreadLocal<>();

    /**
     * add business action context and share it to tcc phase2
     *
     * @param key   the key of new context
     * @param value new context
     * @return branch report succeed
     */
    public static boolean addContext(String key, Object value) {
        if (value == null) {
            return false;
        }

        Map<String, Object> newContext = Collections.singletonMap(key, value);
        return addContext(newContext);
    }

    /**
     * batch share new context to tcc phase 2
     *
     * @param context the new context
     * @return branch report succeed
     */
    @SuppressWarnings("deprecation")
    public static boolean addContext(Map<String, Object> context) {
        if (CollectionUtils.isEmpty(context)) {
            return false;
        }

        // put action context
        BusinessActionContext actionContext = BusinessActionContextUtil.getContext();
        if (!ActionContextUtil.putActionContext(actionContext.getActionContext(), context)) {
            // the action context is not changed, do not report
            return false;
        }
        // set updated
        actionContext.setUpdated(true);

        // if delay report, params will be finally reported after phase 1 execution
        if (Boolean.TRUE.equals(actionContext.getDelayReport())) {
            return false;
        }

        // do branch report
        return reportContext(actionContext);
    }

    /**
     * to do branch report sharing actionContext
     *
     * @param actionContext the context
     * @return branch report succeed
     */
    public static boolean reportContext(BusinessActionContext actionContext) {
        // check is updated
        if (!Boolean.TRUE.equals(actionContext.getUpdated())) {
            return false;
        }

        try {
            // branch report
            DefaultResourceManager.get().branchReport(
                    actionContext.getBranchType(),
                    actionContext.getXid(),
                    actionContext.getBranchId(),
                    BranchStatus.Registered,
                    JsonUtil.toJSONString(Collections.singletonMap(Constants.TX_ACTION_CONTEXT, actionContext.getActionContext()))
            );

            // reset to un_updated
            actionContext.setUpdated(null);
            return true;
        } catch (TransactionException e) {
            String msg = String.format("TCC branch update error, xid: %s", actionContext.getXid());
            LOGGER.error("{}, error: {}", msg, e.getMessage());
            throw new FrameworkException(e, msg);
        }
    }

    public static BusinessActionContext getContext() {
        return CONTEXT_HOLDER.get();
    }

    public static void setContext(BusinessActionContext context) {
        CONTEXT_HOLDER.set(context);
    }

    public static void clear() {
        CONTEXT_HOLDER.remove();
    }

    /**
     * transfer tcc applicationData to BusinessActionContext
     *
     * @param xid             the xid
     * @param branchId        the branch id
     * @param resourceId      the resource id
     * @param applicationData the application data
     * @return business action context
     */
    public static BusinessActionContext getBusinessActionContext(String xid, long branchId, String resourceId,
                                                                 String applicationData) {
        Map actionContextMap = null;
        if (StringUtils.isNotBlank(applicationData)) {
            Map tccContext = JsonUtil.parseObject(applicationData, Map.class);
            actionContextMap = (Map) tccContext.get(Constants.TX_ACTION_CONTEXT);
        }
        if (actionContextMap == null) {
            actionContextMap = new HashMap<>(2);
        }

        //instance the action context
        BusinessActionContext businessActionContext = new BusinessActionContext(
                xid, String.valueOf(branchId), actionContextMap);
        businessActionContext.setActionName(resourceId);
        return businessActionContext;
    }

}
