/*
 * Decompiled with CFR 0.152.
 */
package org.apache.qpid.server.management.plugin.auth;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.AccessControlException;
import java.security.SecureRandom;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import javax.security.auth.Subject;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.xml.bind.DatatypeConverter;
import org.apache.qpid.server.management.plugin.HttpManagementConfiguration;
import org.apache.qpid.server.management.plugin.HttpManagementUtil;
import org.apache.qpid.server.management.plugin.HttpRequestInteractiveAuthenticator;
import org.apache.qpid.server.model.Broker;
import org.apache.qpid.server.plugin.PluggableService;
import org.apache.qpid.server.security.SubjectCreator;
import org.apache.qpid.server.security.auth.AuthenticationResult;
import org.apache.qpid.server.security.auth.SubjectAuthenticationResult;
import org.apache.qpid.server.security.auth.manager.oauth2.OAuth2AuthenticationProvider;
import org.apache.qpid.server.security.auth.manager.oauth2.OAuth2Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@PluggableService
public class OAuth2InteractiveAuthenticator
implements HttpRequestInteractiveAuthenticator {
    private static final Logger LOGGER = LoggerFactory.getLogger(OAuth2InteractiveAuthenticator.class);
    private static final String TYPE = "OAuth2";
    private static final int STATE_NONCE_BIT_SIZE = 256;
    static final String STATE_NAME = "stateNonce";
    static final String REDIRECT_URI_SESSION_ATTRIBUTE = "redirectURI";
    static final String ORIGINAL_REQUEST_URI_SESSION_ATTRIBUTE = "originalRequestURI";
    private static final Map<String, Integer> ERROR_RESPONSES;
    private SecureRandom _random = new SecureRandom();

    public String getType() {
        return TYPE;
    }

    @Override
    public HttpRequestInteractiveAuthenticator.AuthenticationHandler getAuthenticationHandler(final HttpServletRequest request, HttpManagementConfiguration configuration) {
        if (configuration.getAuthenticationProvider(request) instanceof OAuth2AuthenticationProvider) {
            Map<String, String> requestParameters;
            final OAuth2AuthenticationProvider oauth2Provider = (OAuth2AuthenticationProvider)configuration.getAuthenticationProvider(request);
            try {
                requestParameters = this.getRequestParameters(request);
            }
            catch (IllegalArgumentException e) {
                return new FailedAuthenticationHandler(400, "Some request parameters are included more than once " + request, e);
            }
            String error = requestParameters.get("error");
            if (error != null) {
                int responseCode = this.decodeErrorAsResponseCode(error);
                String errorDescription = requestParameters.get("error_description");
                if (responseCode == 403) {
                    LOGGER.debug("Resource owner denies the access request");
                    return new FailedAuthenticationHandler(responseCode, "Resource owner denies the access request");
                }
                LOGGER.warn("Authorization endpoint failed, error : '{}', error description '{}'", (Object)error, (Object)errorDescription);
                return new FailedAuthenticationHandler(responseCode, String.format("Authorization request failed :'%s'", error));
            }
            final String authorizationCode = requestParameters.get("code");
            if (authorizationCode == null) {
                final String authorizationRedirectURL = this.buildAuthorizationRedirectURL(request, oauth2Provider);
                return new HttpRequestInteractiveAuthenticator.AuthenticationHandler(){

                    @Override
                    public void handleAuthentication(HttpServletResponse response) throws IOException {
                        LOGGER.debug("Sending redirect to authorization endpoint {}", (Object)oauth2Provider.getAuthorizationEndpointURI());
                        response.sendRedirect(authorizationRedirectURL);
                    }
                };
            }
            HttpSession httpSession = request.getSession();
            String state = requestParameters.get("state");
            if (state == null) {
                LOGGER.warn("Deny login attempt with wrong state: {}", (Object)state);
                return new FailedAuthenticationHandler(400, "No state set on request with authorization code grant: " + request);
            }
            if (!this.checkState(request, state)) {
                LOGGER.warn("Deny login attempt with wrong state: {}", (Object)state);
                return new FailedAuthenticationHandler(401, "Received request with wrong state: " + state);
            }
            final String redirectUri = (String)httpSession.getAttribute(HttpManagementUtil.getRequestSpecificAttributeName(REDIRECT_URI_SESSION_ATTRIBUTE, request));
            final String originalRequestUri = (String)httpSession.getAttribute(HttpManagementUtil.getRequestSpecificAttributeName(ORIGINAL_REQUEST_URI_SESSION_ATTRIBUTE, request));
            return new HttpRequestInteractiveAuthenticator.AuthenticationHandler(){

                @Override
                public void handleAuthentication(HttpServletResponse response) throws IOException {
                    AuthenticationResult authenticationResult = oauth2Provider.authenticateViaAuthorizationCode(authorizationCode, redirectUri);
                    try {
                        Subject subject = this.createSubject(authenticationResult);
                        this.authoriseManagement(subject);
                        HttpManagementUtil.saveAuthorisedSubject(request, subject);
                        LOGGER.debug("Successful login. Redirect to original resource {}", (Object)originalRequestUri);
                        response.sendRedirect(originalRequestUri);
                    }
                    catch (SecurityException e) {
                        if (e instanceof AccessControlException) {
                            LOGGER.info("User '{}' is not authorised for management", (Object)authenticationResult.getMainPrincipal());
                            response.sendError(403, "User is not authorised for management");
                        }
                        LOGGER.info("Authentication failed", (Throwable)authenticationResult.getCause());
                        response.sendError(401);
                    }
                }

                private Subject createSubject(AuthenticationResult authenticationResult) {
                    SubjectCreator subjectCreator = oauth2Provider.getSubjectCreator(request.isSecure());
                    SubjectAuthenticationResult result = subjectCreator.createResultWithGroups(authenticationResult);
                    Subject original = result.getSubject();
                    if (original == null) {
                        throw new SecurityException("Only authenticated users can access the management interface");
                    }
                    Subject subject = HttpManagementUtil.createServletConnectionSubject(request, original);
                    return subject;
                }

                private void authoriseManagement(Subject subject) {
                    Broker broker = (Broker)oauth2Provider.getParent(Broker.class);
                    HttpManagementUtil.assertManagementAccess(broker, subject);
                }
            };
        }
        return null;
    }

    @Override
    public HttpRequestInteractiveAuthenticator.LogoutHandler getLogoutHandler(HttpServletRequest request, HttpManagementConfiguration configuration) {
        OAuth2AuthenticationProvider oauth2Provider;
        if (configuration.getAuthenticationProvider(request) instanceof OAuth2AuthenticationProvider && (oauth2Provider = (OAuth2AuthenticationProvider)configuration.getAuthenticationProvider(request)).getPostLogoutURI() != null) {
            final String postLogoutRedirect = oauth2Provider.getPostLogoutURI().toString();
            return new HttpRequestInteractiveAuthenticator.LogoutHandler(){

                @Override
                public void handleLogout(HttpServletResponse response) throws IOException {
                    response.sendRedirect(postLogoutRedirect);
                }
            };
        }
        return null;
    }

    private String buildAuthorizationRedirectURL(HttpServletRequest request, OAuth2AuthenticationProvider oauth2Provider) {
        String redirectUri = this.getRedirectUri(request);
        String originalRequestUri = this.getOriginalRequestUri(request);
        String authorizationEndpoint = oauth2Provider.getAuthorizationEndpointURI().toString();
        HttpSession httpSession = request.getSession();
        httpSession.setAttribute(HttpManagementUtil.getRequestSpecificAttributeName(REDIRECT_URI_SESSION_ATTRIBUTE, request), (Object)redirectUri);
        httpSession.setAttribute(HttpManagementUtil.getRequestSpecificAttributeName(ORIGINAL_REQUEST_URI_SESSION_ATTRIBUTE, request), (Object)originalRequestUri);
        HashMap<String, String> queryArgs = new HashMap<String, String>();
        queryArgs.put("client_id", oauth2Provider.getClientId());
        queryArgs.put("redirect_uri", redirectUri);
        queryArgs.put("response_type", "code");
        queryArgs.put("state", this.createState(request));
        if (oauth2Provider.getScope() != null) {
            queryArgs.put("scope", oauth2Provider.getScope());
        }
        StringBuilder urlBuilder = new StringBuilder(authorizationEndpoint);
        String query = oauth2Provider.getAuthorizationEndpointURI().getQuery();
        if (query == null) {
            urlBuilder.append("?");
        } else if (query.length() > 0) {
            urlBuilder.append("&");
        }
        urlBuilder.append(OAuth2Utils.buildRequestQuery(queryArgs));
        return urlBuilder.toString();
    }

    private String getOriginalRequestUri(HttpServletRequest request) {
        StringBuffer originalRequestURL = request.getRequestURL();
        String queryString = request.getQueryString();
        if (queryString != null) {
            originalRequestURL.append("?").append(queryString);
        }
        return originalRequestURL.toString();
    }

    private Map<String, String> getRequestParameters(HttpServletRequest request) {
        HashMap<String, String> requestParameters = new HashMap<String, String>();
        Enumeration parameterNames = request.getParameterNames();
        while (parameterNames.hasMoreElements()) {
            String parameterName = (String)parameterNames.nextElement();
            String[] parameters = request.getParameterValues(parameterName);
            if (parameters == null) {
                throw new IllegalArgumentException(String.format("Request parameter '%s' is null", parameterName));
            }
            if (parameters.length != 1) {
                throw new IllegalArgumentException(String.format("Request parameter '%s' MUST NOT occur more than once", parameterName));
            }
            requestParameters.put(parameterName, parameters[0]);
        }
        return requestParameters;
    }

    private String getRedirectUri(HttpServletRequest request) {
        String servletPath = request.getServletPath() != null ? request.getServletPath() : "";
        String pathInfo = request.getPathInfo() != null ? request.getPathInfo() : "";
        String requestURL = request.getRequestURL().toString();
        try {
            URI redirectURI = new URI(requestURL);
            String redirectString = redirectURI.normalize().toString();
            if (!redirectString.endsWith(servletPath + pathInfo)) {
                throw new IllegalStateException(String.format("RequestURL has unexpected format '%s'", redirectString));
            }
            redirectString = redirectString.substring(0, redirectString.length() - (servletPath.length() + pathInfo.length()));
            return redirectString;
        }
        catch (URISyntaxException e) {
            throw new IllegalStateException(String.format("RequestURL has unexpected format '%s'", requestURL), e);
        }
    }

    private String createState(HttpServletRequest request) {
        byte[] nonceBytes = new byte[32];
        this._random.nextBytes(nonceBytes);
        String nonce = DatatypeConverter.printBase64Binary((byte[])nonceBytes);
        request.getSession().setAttribute(HttpManagementUtil.getRequestSpecificAttributeName(STATE_NAME, request), (Object)nonce);
        return nonce;
    }

    private boolean checkState(HttpServletRequest request, String state) {
        HttpSession session = request.getSession();
        String nonce = (String)session.getAttribute(HttpManagementUtil.getRequestSpecificAttributeName(STATE_NAME, request));
        session.removeAttribute(HttpManagementUtil.getRequestSpecificAttributeName(STATE_NAME, request));
        return state != null && state.equals(nonce);
    }

    private int decodeErrorAsResponseCode(String error) {
        return ERROR_RESPONSES.containsKey(error) ? ERROR_RESPONSES.get(error) : 500;
    }

    static {
        HashMap<String, Integer> errorResponses = new HashMap<String, Integer>();
        errorResponses.put("invalid_request", 400);
        errorResponses.put("unauthorized_client", 400);
        errorResponses.put("unsupported_response_type", 400);
        errorResponses.put("invalid_scope", 400);
        errorResponses.put("access_denied", 403);
        errorResponses.put("server_error", 500);
        errorResponses.put("temporarily_unavailable", 503);
        ERROR_RESPONSES = Collections.unmodifiableMap(errorResponses);
    }

    class FailedAuthenticationHandler
    implements HttpRequestInteractiveAuthenticator.AuthenticationHandler {
        private final int _errorCode;
        private final Throwable _throwable;
        private final String _message;

        FailedAuthenticationHandler(int errorCode, String message) {
            this(errorCode, message, null);
        }

        FailedAuthenticationHandler(int errorCode, String message, Throwable t) {
            this._errorCode = errorCode;
            this._message = message;
            this._throwable = t;
        }

        @Override
        public void handleAuthentication(HttpServletResponse response) throws IOException {
            if (this._throwable != null) {
                response.sendError(this._errorCode, this._message + ": " + this._throwable);
            } else {
                response.sendError(this._errorCode, this._message);
            }
        }
    }
}

