/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cxf.rs.security.jose.jwe;

import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.security.Key;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAKey;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.ECFieldFp;
import java.security.spec.ECPoint;
import java.security.spec.EllipticCurve;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import java.util.logging.Logger;
import javax.crypto.KeyAgreement;
import javax.crypto.SecretKey;
import org.apache.cxf.common.logging.LogUtils;
import org.apache.cxf.common.util.StringUtils;
import org.apache.cxf.message.Message;
import org.apache.cxf.message.MessageUtils;
import org.apache.cxf.phase.PhaseInterceptorChain;
import org.apache.cxf.rs.security.jose.common.JoseHeaders;
import org.apache.cxf.rs.security.jose.common.JoseUtils;
import org.apache.cxf.rs.security.jose.common.KeyManagementUtils;
import org.apache.cxf.rs.security.jose.jwa.AlgorithmUtils;
import org.apache.cxf.rs.security.jose.jwa.ContentAlgorithm;
import org.apache.cxf.rs.security.jose.jwa.KeyAlgorithm;
import org.apache.cxf.rs.security.jose.jwe.AesCbcHmacJweDecryption;
import org.apache.cxf.rs.security.jose.jwe.AesCbcHmacJweEncryption;
import org.apache.cxf.rs.security.jose.jwe.AesGcmContentDecryptionAlgorithm;
import org.apache.cxf.rs.security.jose.jwe.AesGcmContentEncryptionAlgorithm;
import org.apache.cxf.rs.security.jose.jwe.AesGcmWrapKeyDecryptionAlgorithm;
import org.apache.cxf.rs.security.jose.jwe.AesGcmWrapKeyEncryptionAlgorithm;
import org.apache.cxf.rs.security.jose.jwe.AesWrapKeyDecryptionAlgorithm;
import org.apache.cxf.rs.security.jose.jwe.AesWrapKeyEncryptionAlgorithm;
import org.apache.cxf.rs.security.jose.jwe.ContentDecryptionProvider;
import org.apache.cxf.rs.security.jose.jwe.ContentEncryptionProvider;
import org.apache.cxf.rs.security.jose.jwe.DirectKeyDecryptionAlgorithm;
import org.apache.cxf.rs.security.jose.jwe.DirectKeyEncryptionAlgorithm;
import org.apache.cxf.rs.security.jose.jwe.EcdhAesWrapKeyDecryptionAlgorithm;
import org.apache.cxf.rs.security.jose.jwe.EcdhAesWrapKeyEncryptionAlgorithm;
import org.apache.cxf.rs.security.jose.jwe.JweDecryption;
import org.apache.cxf.rs.security.jose.jwe.JweDecryptionProvider;
import org.apache.cxf.rs.security.jose.jwe.JweEncryption;
import org.apache.cxf.rs.security.jose.jwe.JweEncryptionProvider;
import org.apache.cxf.rs.security.jose.jwe.JweException;
import org.apache.cxf.rs.security.jose.jwe.JweHeaders;
import org.apache.cxf.rs.security.jose.jwe.KeyDecryptionProvider;
import org.apache.cxf.rs.security.jose.jwe.KeyEncryptionProvider;
import org.apache.cxf.rs.security.jose.jwe.RSAKeyDecryptionAlgorithm;
import org.apache.cxf.rs.security.jose.jwe.RSAKeyEncryptionAlgorithm;
import org.apache.cxf.rs.security.jose.jwk.JsonWebKey;
import org.apache.cxf.rs.security.jose.jwk.JsonWebKeys;
import org.apache.cxf.rs.security.jose.jwk.JwkUtils;
import org.apache.cxf.rs.security.jose.jwk.KeyOperation;
import org.apache.cxf.rs.security.jose.jwk.KeyType;
import org.apache.cxf.rt.security.crypto.MessageDigestUtils;

public final class JweUtils {
    private static final Logger LOG = LogUtils.getL7dLogger(JweUtils.class);

    private JweUtils() {
    }

    public static String encrypt(PublicKey key, KeyAlgorithm keyAlgo, ContentAlgorithm contentAlgo, byte[] content) {
        return JweUtils.encrypt(key, keyAlgo, contentAlgo, content, null);
    }

    public static String encrypt(PublicKey key, KeyAlgorithm keyAlgo, ContentAlgorithm contentAlgo, byte[] content, String ct) {
        KeyEncryptionProvider keyEncryptionProvider = JweUtils.getPublicKeyEncryptionProvider(key, keyAlgo);
        return JweUtils.encrypt(keyEncryptionProvider, contentAlgo, content, ct);
    }

    public static String encrypt(SecretKey key, KeyAlgorithm keyAlgo, ContentAlgorithm contentAlgo, byte[] content) {
        return JweUtils.encrypt(key, keyAlgo, contentAlgo, content, null);
    }

    public static String encrypt(SecretKey key, KeyAlgorithm keyAlgo, ContentAlgorithm contentAlgo, byte[] content, String ct) {
        if (keyAlgo != null) {
            KeyEncryptionProvider keyEncryptionProvider = JweUtils.getSecretKeyEncryptionAlgorithm(key, keyAlgo);
            return JweUtils.encrypt(keyEncryptionProvider, contentAlgo, content, ct);
        }
        return JweUtils.encryptDirect(key, contentAlgo, content, ct);
    }

    public static String encrypt(JsonWebKey key, ContentAlgorithm contentAlgo, byte[] content, String ct) {
        KeyEncryptionProvider keyEncryptionProvider = JweUtils.getKeyEncryptionProvider(key);
        return JweUtils.encrypt(keyEncryptionProvider, contentAlgo, content, ct);
    }

    public static String encryptDirect(SecretKey key, ContentAlgorithm contentAlgo, byte[] content) {
        return JweUtils.encryptDirect(key, contentAlgo, content, null);
    }

    public static String encryptDirect(SecretKey key, ContentAlgorithm contentAlgo, byte[] content, String ct) {
        JweEncryption jwe = JweUtils.getDirectKeyJweEncryption(key, contentAlgo);
        return jwe.encrypt(content, JweUtils.toJweHeaders(ct));
    }

    public static String encryptDirect(JsonWebKey key, byte[] content, String ct) {
        JweEncryption jwe = JweUtils.getDirectKeyJweEncryption(key);
        return jwe.encrypt(content, JweUtils.toJweHeaders(ct));
    }

    public static byte[] decrypt(PrivateKey key, KeyAlgorithm keyAlgo, ContentAlgorithm contentAlgo, String content) {
        KeyDecryptionProvider keyDecryptionProvider = JweUtils.getPrivateKeyDecryptionProvider(key, keyAlgo);
        return JweUtils.decrypt(keyDecryptionProvider, contentAlgo, content);
    }

    public static byte[] decrypt(SecretKey key, KeyAlgorithm keyAlgo, ContentAlgorithm contentAlgo, String content) {
        if (keyAlgo != null) {
            KeyDecryptionProvider keyDecryptionProvider = JweUtils.getSecretKeyDecryptionProvider(key, keyAlgo);
            return JweUtils.decrypt(keyDecryptionProvider, contentAlgo, content);
        }
        return JweUtils.decryptDirect(key, contentAlgo, content);
    }

    public static byte[] decrypt(JsonWebKey key, ContentAlgorithm contentAlgo, String content) {
        KeyDecryptionProvider keyDecryptionProvider = JweUtils.getKeyDecryptionProvider(key);
        return JweUtils.decrypt(keyDecryptionProvider, contentAlgo, content);
    }

    public static byte[] decryptDirect(SecretKey key, ContentAlgorithm contentAlgo, String content) {
        JweDecryption jwe = JweUtils.getDirectKeyJweDecryption(key, contentAlgo);
        return jwe.decrypt(content).getContent();
    }

    public static byte[] decryptDirect(JsonWebKey key, String content) {
        JweDecryption jwe = JweUtils.getDirectKeyJweDecryption(key);
        return jwe.decrypt(content).getContent();
    }

    public static KeyEncryptionProvider getKeyEncryptionProvider(JsonWebKey jwk) {
        return JweUtils.getKeyEncryptionProvider(jwk, null);
    }

    public static KeyEncryptionProvider getKeyEncryptionProvider(JsonWebKey jwk, KeyAlgorithm defaultAlgorithm) {
        KeyAlgorithm keyAlgo = jwk.getAlgorithm() == null ? defaultAlgorithm : KeyAlgorithm.getAlgorithm(jwk.getAlgorithm());
        KeyEncryptionProvider keyEncryptionProvider = null;
        KeyType keyType = jwk.getKeyType();
        if (KeyType.RSA == keyType) {
            keyEncryptionProvider = JweUtils.getPublicKeyEncryptionProvider(JwkUtils.toRSAPublicKey(jwk, true), keyAlgo);
        } else if (KeyType.OCTET == keyType) {
            keyEncryptionProvider = JweUtils.getSecretKeyEncryptionAlgorithm(JwkUtils.toSecretKey(jwk), keyAlgo);
        } else {
            ContentAlgorithm ctAlgo = null;
            Message m = PhaseInterceptorChain.getCurrentMessage();
            if (m != null) {
                ctAlgo = JweUtils.getContentAlgo((String)m.get((Object)"rs.security.encryption.content.algorithm"));
            }
            keyEncryptionProvider = new EcdhAesWrapKeyEncryptionAlgorithm(JwkUtils.toECPublicKey(jwk), jwk.getStringProperty("crv"), keyAlgo, ctAlgo == null ? ContentAlgorithm.A128GCM : ctAlgo);
        }
        return keyEncryptionProvider;
    }

    public static KeyEncryptionProvider getPublicKeyEncryptionProvider(PublicKey key, KeyAlgorithm algo) {
        return JweUtils.getPublicKeyEncryptionProvider(key, null, algo);
    }

    public static KeyEncryptionProvider getPublicKeyEncryptionProvider(PublicKey key, Properties props, KeyAlgorithm algo) {
        if (algo == null) {
            algo = JweUtils.getDefaultPublicKeyAlgorithm(key);
        }
        if (key instanceof RSAPublicKey) {
            return new RSAKeyEncryptionAlgorithm((RSAPublicKey)key, algo);
        }
        if (key instanceof ECPublicKey) {
            ContentAlgorithm ctAlgo = null;
            Message m = PhaseInterceptorChain.getCurrentMessage();
            if (m != null) {
                ctAlgo = JweUtils.getContentAlgo((String)m.get((Object)"rs.security.encryption.content.algorithm"));
            }
            String curve = props == null ? "P-256" : props.getProperty("rs.security.elliptic.curve", "P-256");
            return new EcdhAesWrapKeyEncryptionAlgorithm((ECPublicKey)key, curve, algo, ctAlgo == null ? ContentAlgorithm.A128GCM : ctAlgo);
        }
        return null;
    }

    private static KeyAlgorithm getDefaultPublicKeyAlgorithm(PublicKey key) {
        if (key instanceof RSAPublicKey) {
            return KeyAlgorithm.RSA_OAEP;
        }
        if (key instanceof ECPublicKey) {
            return KeyAlgorithm.ECDH_ES_A128KW;
        }
        return null;
    }

    private static KeyAlgorithm getDefaultPrivateKeyAlgorithm(PrivateKey key) {
        if (key instanceof RSAPrivateKey) {
            return KeyAlgorithm.RSA_OAEP;
        }
        if (key instanceof ECPrivateKey) {
            return KeyAlgorithm.ECDH_ES_A128KW;
        }
        return null;
    }

    public static KeyEncryptionProvider getSecretKeyEncryptionAlgorithm(SecretKey key, KeyAlgorithm algo) {
        if (AlgorithmUtils.isAesKeyWrap(algo.getJwaName())) {
            return new AesWrapKeyEncryptionAlgorithm(key, algo);
        }
        if (AlgorithmUtils.isAesGcmKeyWrap(algo.getJwaName())) {
            return new AesGcmWrapKeyEncryptionAlgorithm(key, algo);
        }
        return null;
    }

    public static KeyDecryptionProvider getKeyDecryptionProvider(JsonWebKey jwk) {
        return JweUtils.getKeyDecryptionProvider(jwk, null);
    }

    public static KeyDecryptionProvider getKeyDecryptionProvider(JsonWebKey jwk, KeyAlgorithm defaultAlgorithm) {
        KeyAlgorithm keyAlgo = jwk.getAlgorithm() == null ? defaultAlgorithm : KeyAlgorithm.getAlgorithm(jwk.getAlgorithm());
        KeyDecryptionProvider keyDecryptionProvider = null;
        KeyType keyType = jwk.getKeyType();
        keyDecryptionProvider = KeyType.RSA == keyType ? JweUtils.getPrivateKeyDecryptionProvider(JwkUtils.toRSAPrivateKey(jwk), keyAlgo) : (KeyType.OCTET == keyType ? JweUtils.getSecretKeyDecryptionProvider(JwkUtils.toSecretKey(jwk), keyAlgo) : JweUtils.getPrivateKeyDecryptionProvider(JwkUtils.toECPrivateKey(jwk), keyAlgo));
        return keyDecryptionProvider;
    }

    public static KeyDecryptionProvider getPrivateKeyDecryptionProvider(PrivateKey key, KeyAlgorithm algo) {
        if (key instanceof RSAPrivateKey) {
            return new RSAKeyDecryptionAlgorithm((RSAPrivateKey)key, algo);
        }
        if (key instanceof ECPrivateKey) {
            return new EcdhAesWrapKeyDecryptionAlgorithm((ECPrivateKey)key, algo);
        }
        return null;
    }

    public static KeyDecryptionProvider getSecretKeyDecryptionProvider(SecretKey key, KeyAlgorithm algo) {
        if (AlgorithmUtils.isAesKeyWrap(algo.getJwaName())) {
            return new AesWrapKeyDecryptionAlgorithm(key, algo);
        }
        if (AlgorithmUtils.isAesGcmKeyWrap(algo.getJwaName())) {
            return new AesGcmWrapKeyDecryptionAlgorithm(key, algo);
        }
        return null;
    }

    public static ContentEncryptionProvider getContentEncryptionProvider(JsonWebKey jwk) {
        return JweUtils.getContentEncryptionProvider(jwk, null);
    }

    public static ContentEncryptionProvider getContentEncryptionProvider(JsonWebKey jwk, ContentAlgorithm defaultAlgorithm) {
        ContentAlgorithm ctAlgo = jwk.getAlgorithm() == null ? defaultAlgorithm : JweUtils.getContentAlgo(jwk.getAlgorithm());
        KeyType keyType = jwk.getKeyType();
        if (KeyType.OCTET == keyType) {
            return JweUtils.getContentEncryptionProvider(JwkUtils.toSecretKey(jwk), ctAlgo);
        }
        return null;
    }

    public static ContentEncryptionProvider getContentEncryptionProvider(SecretKey key, ContentAlgorithm algorithm) {
        return JweUtils.getContentEncryptionProvider(key.getEncoded(), algorithm);
    }

    public static ContentEncryptionProvider getContentEncryptionProvider(byte[] key, ContentAlgorithm algorithm) {
        if (AlgorithmUtils.isAesGcm(algorithm.getJwaName())) {
            return new AesGcmContentEncryptionAlgorithm(key, null, algorithm);
        }
        return null;
    }

    public static ContentEncryptionProvider getContentEncryptionProvider(ContentAlgorithm algorithm) {
        return JweUtils.getContentEncryptionProvider(algorithm, false);
    }

    public static ContentEncryptionProvider getContentEncryptionProvider(ContentAlgorithm algorithm, boolean generateCekOnce) {
        if (AlgorithmUtils.isAesGcm(algorithm.getJwaName())) {
            return new AesGcmContentEncryptionAlgorithm(algorithm, generateCekOnce);
        }
        return null;
    }

    public static ContentDecryptionProvider getContentDecryptionProvider(ContentAlgorithm algorithm) {
        if (AlgorithmUtils.isAesGcm(algorithm.getJwaName())) {
            return new AesGcmContentDecryptionAlgorithm(algorithm);
        }
        return null;
    }

    public static SecretKey getContentDecryptionSecretKey(JsonWebKey jwk) {
        return JweUtils.getContentDecryptionSecretKey(jwk, null);
    }

    public static SecretKey getContentDecryptionSecretKey(JsonWebKey jwk, String defaultAlgorithm) {
        String ctEncryptionAlgo = jwk.getAlgorithm() == null ? defaultAlgorithm : jwk.getAlgorithm();
        KeyType keyType = jwk.getKeyType();
        if (KeyType.OCTET == keyType && AlgorithmUtils.isAesGcm(ctEncryptionAlgo)) {
            return JwkUtils.toSecretKey(jwk);
        }
        return null;
    }

    private static ContentAlgorithm getContentAlgo(String algo) {
        return ContentAlgorithm.getAlgorithm(algo);
    }

    public static JweEncryption getDirectKeyJweEncryption(JsonWebKey key) {
        return JweUtils.getDirectKeyJweEncryption(JwkUtils.toSecretKey(key), JweUtils.getContentAlgo(key.getAlgorithm()));
    }

    public static JweEncryption getDirectKeyJweEncryption(SecretKey key, ContentAlgorithm algo) {
        return JweUtils.getDirectKeyJweEncryption(key.getEncoded(), algo);
    }

    public static JweEncryption getDirectKeyJweEncryption(byte[] key, ContentAlgorithm algo) {
        if (AlgorithmUtils.isAesCbcHmac(algo.getJwaName())) {
            return new AesCbcHmacJweEncryption(algo, key, null, new DirectKeyEncryptionAlgorithm());
        }
        return new JweEncryption(new DirectKeyEncryptionAlgorithm(), JweUtils.getContentEncryptionProvider(key, algo));
    }

    public static JweDecryption getDirectKeyJweDecryption(JsonWebKey key) {
        return JweUtils.getDirectKeyJweDecryption(JwkUtils.toSecretKey(key), JweUtils.getContentAlgo(key.getAlgorithm()));
    }

    public static JweDecryption getDirectKeyJweDecryption(SecretKey key, ContentAlgorithm algorithm) {
        return JweUtils.getDirectKeyJweDecryption(key.getEncoded(), algorithm);
    }

    public static JweDecryption getDirectKeyJweDecryption(byte[] key, ContentAlgorithm algorithm) {
        if (AlgorithmUtils.isAesCbcHmac(algorithm.getJwaName())) {
            return new AesCbcHmacJweDecryption((KeyDecryptionProvider)new DirectKeyDecryptionAlgorithm(key), algorithm);
        }
        return new JweDecryption(new DirectKeyDecryptionAlgorithm(key), JweUtils.getContentDecryptionProvider(algorithm));
    }

    public static JweEncryptionProvider loadEncryptionProvider(boolean required) {
        return JweUtils.loadEncryptionProvider(null, required);
    }

    public static JweEncryptionProvider loadEncryptionProvider(JweHeaders headers, boolean required) {
        Properties props = JweUtils.loadEncryptionOutProperties(required);
        if (props == null) {
            return null;
        }
        return JweUtils.loadEncryptionProvider(props, headers, required);
    }

    public static JweEncryptionProvider loadEncryptionProvider(Properties props, JweHeaders headers, boolean required) {
        Message m = PhaseInterceptorChain.getCurrentMessage();
        boolean includeCert = headers != null && MessageUtils.getContextualBoolean((Message)m, (String)"rs.security.encryption.include.cert", (boolean)false);
        boolean includeCertSha1 = headers != null && MessageUtils.getContextualBoolean((Message)m, (String)"rs.security.encryption.include.cert.sha1", (boolean)false);
        boolean includeCertSha256 = !includeCertSha1 && headers != null && MessageUtils.getContextualBoolean((Message)m, (String)"rs.security.encryption.include.cert.sha256", (boolean)false);
        KeyEncryptionProvider keyEncryptionProvider = null;
        KeyAlgorithm keyAlgo = JweUtils.getKeyEncryptionAlgorithm(m, props, null, null);
        ContentAlgorithm contentAlgo = JweUtils.getContentEncryptionAlgorithm(m, props, null, ContentAlgorithm.A128GCM);
        if (m != null) {
            m.put((Object)"rs.security.encryption.content.algorithm", (Object)contentAlgo.getJwaName());
        }
        ContentEncryptionProvider ctEncryptionProvider = null;
        if ("jwk".equals(props.get("rs.security.keystore.type"))) {
            JsonWebKey jwk = JwkUtils.loadJsonWebKey(m, props, KeyOperation.ENCRYPT);
            if (KeyAlgorithm.DIRECT == keyAlgo) {
                contentAlgo = JweUtils.getContentEncryptionAlgorithm(m, props, ContentAlgorithm.getAlgorithm(jwk.getAlgorithm()), ContentAlgorithm.A128GCM);
                ctEncryptionProvider = JweUtils.getContentEncryptionProvider(jwk, contentAlgo);
            } else {
                String digest;
                boolean includeKeyId;
                keyAlgo = JweUtils.getKeyEncryptionAlgorithm(m, props, KeyAlgorithm.getAlgorithm(jwk.getAlgorithm()), JweUtils.getDefaultKeyAlgorithm(jwk));
                keyEncryptionProvider = JweUtils.getKeyEncryptionProvider(jwk, keyAlgo);
                boolean includePublicKey = headers != null && MessageUtils.getContextualBoolean((Message)m, (String)"rs.security.encryption.include.public.key", (boolean)false);
                boolean bl = includeKeyId = headers != null && MessageUtils.getContextualBoolean((Message)m, (String)"rs.security.encryption.include.key.id", (boolean)false);
                if (includeCert) {
                    JwkUtils.includeCertChain(jwk, headers, keyAlgo.getJwaName());
                }
                if (includeCertSha1) {
                    String digest2 = KeyManagementUtils.loadDigestAndEncodeX509Certificate(m, props, "SHA-1");
                    if (digest2 != null) {
                        headers.setX509Thumbprint(digest2);
                    }
                } else if (includeCertSha256 && (digest = KeyManagementUtils.loadDigestAndEncodeX509Certificate(m, props, "SHA-256")) != null) {
                    headers.setX509ThumbprintSHA256(digest);
                }
                if (includePublicKey) {
                    JwkUtils.includePublicKey(jwk, headers, keyAlgo.getJwaName());
                }
                if (includeKeyId && jwk.getKeyId() != null && headers != null) {
                    headers.setKeyId(jwk.getKeyId());
                }
            }
        } else {
            String digest;
            keyEncryptionProvider = JweUtils.getPublicKeyEncryptionProvider(KeyManagementUtils.loadPublicKey(m, props), props, keyAlgo);
            if (includeCert) {
                headers.setX509Chain(KeyManagementUtils.loadAndEncodeX509CertificateOrChain(m, props));
            }
            if (includeCertSha1) {
                digest = KeyManagementUtils.loadDigestAndEncodeX509Certificate(m, props, "SHA-1");
                if (digest != null) {
                    headers.setX509Thumbprint(digest);
                }
            } else if (includeCertSha256 && (digest = KeyManagementUtils.loadDigestAndEncodeX509Certificate(m, props, "SHA-256")) != null) {
                headers.setX509ThumbprintSHA256(digest);
            }
        }
        String compression = props.getProperty("rs.security.encryption.zip.algorithm");
        return JweUtils.createJweEncryptionProvider(keyEncryptionProvider, ctEncryptionProvider, contentAlgo.getJwaName(), compression);
    }

    public static JweDecryptionProvider loadDecryptionProvider(boolean required) {
        return JweUtils.loadDecryptionProvider(null, required);
    }

    public static JweDecryptionProvider loadDecryptionProvider(JweHeaders inHeaders, boolean required) {
        Properties props = JweUtils.loadEncryptionInProperties(required);
        if (props == null) {
            return null;
        }
        return JweUtils.loadDecryptionProvider(props, inHeaders, required);
    }

    public static JweDecryptionProvider loadDecryptionProvider(Properties props, JweHeaders inHeaders, boolean required) {
        Message m = PhaseInterceptorChain.getCurrentMessage();
        KeyDecryptionProvider keyDecryptionProvider = null;
        ContentAlgorithm contentAlgo = JweUtils.getContentEncryptionAlgorithm(m, props, null, ContentAlgorithm.A128GCM);
        SecretKey ctDecryptionKey = null;
        KeyAlgorithm keyAlgo = JweUtils.getKeyEncryptionAlgorithm(m, props, null, null);
        if (inHeaders != null && inHeaders.getHeader("x5c") != null) {
            List<X509Certificate> chain = KeyManagementUtils.toX509CertificateChain(inHeaders.getX509Chain());
            KeyManagementUtils.validateCertificateChain(props, chain);
            X509Certificate cert = chain == null ? null : chain.get(0);
            PrivateKey privateKey = KeyManagementUtils.loadPrivateKey(m, props, cert, KeyOperation.DECRYPT);
            if (keyAlgo == null) {
                keyAlgo = JweUtils.getDefaultPrivateKeyAlgorithm(privateKey);
            }
            contentAlgo = inHeaders.getContentEncryptionAlgorithm();
            keyDecryptionProvider = JweUtils.getPrivateKeyDecryptionProvider(privateKey, keyAlgo);
        } else if (inHeaders != null && inHeaders.getHeader("x5t") != null) {
            X509Certificate foundCert = KeyManagementUtils.getCertificateFromThumbprint(inHeaders.getX509Thumbprint(), "SHA-1", m, props);
            if (foundCert != null) {
                PrivateKey privateKey = KeyManagementUtils.loadPrivateKey(m, props, foundCert, KeyOperation.DECRYPT);
                if (keyAlgo == null) {
                    keyAlgo = JweUtils.getDefaultPrivateKeyAlgorithm(privateKey);
                }
                contentAlgo = inHeaders.getContentEncryptionAlgorithm();
                keyDecryptionProvider = JweUtils.getPrivateKeyDecryptionProvider(privateKey, keyAlgo);
            }
        } else if (inHeaders != null && inHeaders.getHeader("x5t#S256") != null) {
            X509Certificate foundCert = KeyManagementUtils.getCertificateFromThumbprint(inHeaders.getX509ThumbprintSHA256(), "SHA-256", m, props);
            if (foundCert != null) {
                PrivateKey privateKey = KeyManagementUtils.loadPrivateKey(m, props, foundCert, KeyOperation.DECRYPT);
                if (keyAlgo == null) {
                    keyAlgo = JweUtils.getDefaultPrivateKeyAlgorithm(privateKey);
                }
                contentAlgo = inHeaders.getContentEncryptionAlgorithm();
                keyDecryptionProvider = JweUtils.getPrivateKeyDecryptionProvider(privateKey, keyAlgo);
            }
        } else if ("jwk".equals(props.get("rs.security.keystore.type"))) {
            JsonWebKey jwk = JwkUtils.loadJsonWebKey(m, props, KeyOperation.DECRYPT);
            if (jwk == null) {
                LOG.warning("Extracting the JsonWebKey failed");
                throw new JweException(JweException.Error.KEY_DECRYPTION_FAILURE);
            }
            if (KeyAlgorithm.DIRECT == keyAlgo) {
                contentAlgo = JweUtils.getContentEncryptionAlgorithm(m, props, ContentAlgorithm.getAlgorithm(jwk.getAlgorithm()), ContentAlgorithm.A128GCM);
                ctDecryptionKey = JweUtils.getContentDecryptionSecretKey(jwk, contentAlgo.getJwaName());
            } else {
                keyAlgo = JweUtils.getKeyEncryptionAlgorithm(m, props, KeyAlgorithm.getAlgorithm(jwk.getAlgorithm()), JweUtils.getDefaultKeyAlgorithm(jwk));
                keyDecryptionProvider = JweUtils.getKeyDecryptionProvider(jwk, keyAlgo);
            }
        } else {
            PrivateKey privateKey = KeyManagementUtils.loadPrivateKey(m, props, KeyOperation.DECRYPT);
            if (keyAlgo == null) {
                keyAlgo = JweUtils.getDefaultPrivateKeyAlgorithm(privateKey);
            }
            keyDecryptionProvider = JweUtils.getPrivateKeyDecryptionProvider(privateKey, keyAlgo);
        }
        return JweUtils.createJweDecryptionProvider(keyDecryptionProvider, ctDecryptionKey, contentAlgo);
    }

    public static List<JweEncryptionProvider> loadJweEncryptionProviders(String propLoc, Message m) {
        List<JsonWebKey> jwks;
        Properties props = JweUtils.loadJweProperties(m, propLoc);
        JweEncryptionProvider theEncProvider = JweUtils.loadEncryptionProvider(props, null, false);
        if (theEncProvider != null) {
            return Collections.singletonList(theEncProvider);
        }
        ArrayList<JweEncryption> theEncProviders = null;
        if ("jwk".equals(props.get("rs.security.keystore.type")) && (jwks = JwkUtils.loadJsonWebKeys(m, props, KeyOperation.ENCRYPT)) != null) {
            theEncProviders = new ArrayList<JweEncryption>(jwks.size());
            for (JsonWebKey jwk : jwks) {
                theEncProviders.add(JweUtils.getDirectKeyJweEncryption(jwk));
            }
        }
        if (theEncProviders == null) {
            LOG.warning("Providers are not available");
            throw new JweException(JweException.Error.NO_ENCRYPTOR);
        }
        return theEncProviders;
    }

    public static List<JweDecryptionProvider> loadJweDecryptionProviders(String propLoc, Message m) {
        List<JsonWebKey> jwks;
        Properties props = JweUtils.loadJweProperties(m, propLoc);
        JweDecryptionProvider theDecProvider = JweUtils.loadDecryptionProvider(props, null, false);
        if (theDecProvider != null) {
            return Collections.singletonList(theDecProvider);
        }
        ArrayList<JweDecryption> theDecProviders = null;
        if ("jwk".equals(props.get("rs.security.keystore.type")) && (jwks = JwkUtils.loadJsonWebKeys(m, props, KeyOperation.DECRYPT)) != null) {
            theDecProviders = new ArrayList<JweDecryption>(jwks.size());
            for (JsonWebKey jwk : jwks) {
                theDecProviders.add(JweUtils.getDirectKeyJweDecryption(jwk));
            }
        }
        if (theDecProviders == null) {
            LOG.warning("Providers are not available");
            throw new JweException(JweException.Error.NO_ENCRYPTOR);
        }
        return theDecProviders;
    }

    public static JweEncryptionProvider createJweEncryptionProvider(PublicKey key, KeyAlgorithm keyAlgo, ContentAlgorithm contentEncryptionAlgo) {
        return JweUtils.createJweEncryptionProvider(key, keyAlgo, contentEncryptionAlgo, null);
    }

    public static JweEncryptionProvider createJweEncryptionProvider(PublicKey key, KeyAlgorithm keyAlgo, ContentAlgorithm contentEncryptionAlgo, String compression) {
        KeyEncryptionProvider keyEncryptionProvider = JweUtils.getPublicKeyEncryptionProvider(key, keyAlgo);
        return JweUtils.createJweEncryptionProvider(keyEncryptionProvider, contentEncryptionAlgo, compression);
    }

    public static JweEncryptionProvider createJweEncryptionProvider(PublicKey key, JweHeaders headers) {
        KeyEncryptionProvider keyEncryptionProvider = JweUtils.getPublicKeyEncryptionProvider(key, headers.getKeyEncryptionAlgorithm());
        return JweUtils.createJweEncryptionProvider(keyEncryptionProvider, headers);
    }

    public static JweEncryptionProvider createJweEncryptionProvider(SecretKey key, KeyAlgorithm keyAlgo, ContentAlgorithm contentEncryptionAlgo) {
        return JweUtils.createJweEncryptionProvider(key, keyAlgo, contentEncryptionAlgo, null);
    }

    public static JweEncryptionProvider createJweEncryptionProvider(SecretKey key, KeyAlgorithm keyAlgo, ContentAlgorithm contentEncryptionAlgo, String compression) {
        KeyEncryptionProvider keyEncryptionProvider = JweUtils.getSecretKeyEncryptionAlgorithm(key, keyAlgo);
        return JweUtils.createJweEncryptionProvider(keyEncryptionProvider, contentEncryptionAlgo, compression);
    }

    public static JweEncryptionProvider createJweEncryptionProvider(SecretKey key, JweHeaders headers) {
        KeyEncryptionProvider keyEncryptionProvider = JweUtils.getSecretKeyEncryptionAlgorithm(key, headers.getKeyEncryptionAlgorithm());
        return JweUtils.createJweEncryptionProvider(keyEncryptionProvider, headers);
    }

    public static JweEncryptionProvider createJweEncryptionProvider(JsonWebKey key, ContentAlgorithm contentEncryptionAlgo) {
        return JweUtils.createJweEncryptionProvider(key, contentEncryptionAlgo, null);
    }

    public static JweEncryptionProvider createJweEncryptionProvider(JsonWebKey key, ContentAlgorithm contentEncryptionAlgo, String compression) {
        KeyEncryptionProvider keyEncryptionProvider = JweUtils.getKeyEncryptionProvider(key);
        return JweUtils.createJweEncryptionProvider(keyEncryptionProvider, contentEncryptionAlgo, compression);
    }

    public static JweEncryptionProvider createJweEncryptionProvider(JsonWebKey key, JweHeaders headers) {
        KeyEncryptionProvider keyEncryptionProvider = JweUtils.getKeyEncryptionProvider(key);
        return JweUtils.createJweEncryptionProvider(keyEncryptionProvider, headers);
    }

    public static JweEncryptionProvider createJweEncryptionProvider(KeyEncryptionProvider keyEncryptionProvider, ContentAlgorithm contentEncryptionAlgo, String compression) {
        JweHeaders headers = JweUtils.prepareJweHeaders(keyEncryptionProvider != null ? keyEncryptionProvider.getAlgorithm().getJwaName() : null, contentEncryptionAlgo.getJwaName(), compression);
        return JweUtils.createJweEncryptionProvider(keyEncryptionProvider, headers);
    }

    public static JweEncryptionProvider createJweEncryptionProvider(KeyEncryptionProvider keyEncryptionProvider, JweHeaders headers) {
        ContentAlgorithm contentEncryptionAlgo = headers.getContentEncryptionAlgorithm();
        if (AlgorithmUtils.isAesCbcHmac(contentEncryptionAlgo.getJwaName())) {
            return new AesCbcHmacJweEncryption(contentEncryptionAlgo, keyEncryptionProvider);
        }
        return new JweEncryption(keyEncryptionProvider, JweUtils.getContentEncryptionProvider(contentEncryptionAlgo));
    }

    public static JweDecryptionProvider createJweDecryptionProvider(PrivateKey key, KeyAlgorithm keyAlgo, ContentAlgorithm contentDecryptionAlgo) {
        return JweUtils.createJweDecryptionProvider(JweUtils.getPrivateKeyDecryptionProvider(key, keyAlgo), contentDecryptionAlgo);
    }

    public static JweDecryptionProvider createJweDecryptionProvider(SecretKey key, KeyAlgorithm keyAlgo, ContentAlgorithm contentDecryptionAlgo) {
        return JweUtils.createJweDecryptionProvider(JweUtils.getSecretKeyDecryptionProvider(key, keyAlgo), contentDecryptionAlgo);
    }

    public static JweDecryptionProvider createJweDecryptionProvider(JsonWebKey key, ContentAlgorithm contentDecryptionAlgo) {
        return JweUtils.createJweDecryptionProvider(JweUtils.getKeyDecryptionProvider(key), contentDecryptionAlgo);
    }

    public static JweDecryptionProvider createJweDecryptionProvider(KeyDecryptionProvider keyDecryptionProvider, ContentAlgorithm contentDecryptionAlgo) {
        if (AlgorithmUtils.isAesCbcHmac(contentDecryptionAlgo.getJwaName())) {
            return new AesCbcHmacJweDecryption(keyDecryptionProvider, contentDecryptionAlgo);
        }
        return new JweDecryption(keyDecryptionProvider, JweUtils.getContentDecryptionProvider(contentDecryptionAlgo));
    }

    public static boolean validateCriticalHeaders(JoseHeaders headers) {
        return JoseUtils.validateCriticalHeaders(headers);
    }

    public static byte[] getECDHKey(JsonWebKey privateKey, JsonWebKey peerPublicKey, byte[] partyUInfo, byte[] partyVInfo, String algoName, int algoKeyBitLen) {
        return JweUtils.getECDHKey(JwkUtils.toECPrivateKey(privateKey), JwkUtils.toECPublicKey(peerPublicKey), partyUInfo, partyVInfo, algoName, algoKeyBitLen);
    }

    public static byte[] getECDHKey(ECPrivateKey privateKey, ECPublicKey peerPublicKey, byte[] partyUInfo, byte[] partyVInfo, String algoName, int algoKeyBitLen) {
        BigInteger xCubedPlusAXPlusB;
        if (ECPoint.POINT_INFINITY.equals(peerPublicKey.getW())) {
            throw new JweException(JweException.Error.KEY_ENCRYPTION_FAILURE);
        }
        EllipticCurve curve = peerPublicKey.getParams().getCurve();
        BigInteger x = peerPublicKey.getW().getAffineX();
        BigInteger y = peerPublicKey.getW().getAffineY();
        BigInteger p = ((ECFieldFp)curve.getField()).getP();
        if (x.compareTo(BigInteger.ZERO) < 0 || x.compareTo(p) >= 0 || y.compareTo(BigInteger.ZERO) < 0 || y.compareTo(p) >= 0) {
            throw new JweException(JweException.Error.KEY_ENCRYPTION_FAILURE);
        }
        BigInteger a = curve.getA();
        BigInteger b = curve.getB();
        BigInteger ySquared = y.modPow(BigInteger.valueOf(2L), p);
        if (!ySquared.equals(xCubedPlusAXPlusB = x.modPow(BigInteger.valueOf(3L), p).add(a.multiply(x)).add(b).mod(p))) {
            throw new JweException(JweException.Error.KEY_ENCRYPTION_FAILURE);
        }
        if (peerPublicKey.getParams().getCofactor() != 1) {
            throw new JweException(JweException.Error.KEY_ENCRYPTION_FAILURE);
        }
        byte[] keyZ = JweUtils.generateKeyZ(privateKey, peerPublicKey);
        return JweUtils.calculateDerivedKey(keyZ, algoName, partyUInfo, partyVInfo, algoKeyBitLen);
    }

    public static byte[] getAdditionalAuthenticationData(String headersJson, byte[] aad) {
        byte[] headersAAD = JweHeaders.toCipherAdditionalAuthData(headersJson);
        if (aad != null) {
            byte[] newAAD = Arrays.copyOf(headersAAD, headersAAD.length + 1 + aad.length);
            newAAD[headersAAD.length] = 46;
            System.arraycopy(aad, 0, newAAD, headersAAD.length + 1, aad.length);
            return newAAD;
        }
        return headersAAD;
    }

    private static byte[] calculateDerivedKey(byte[] keyZ, String algoName, byte[] apuBytes, byte[] apvBytes, int algoKeyBitLen) {
        byte[] emptyPartyInfo = new byte[4];
        if (apuBytes != null && apvBytes != null && Arrays.equals(apuBytes, apvBytes)) {
            LOG.warning("Derived key calculation problem: apu equals to apv");
            throw new JweException(JweException.Error.KEY_ENCRYPTION_FAILURE);
        }
        byte[] algorithmId = JweUtils.concatenateDatalenAndData(StringUtils.toBytesASCII((String)algoName));
        byte[] partyUInfo = apuBytes == null ? emptyPartyInfo : JweUtils.concatenateDatalenAndData(apuBytes);
        byte[] partyVInfo = apvBytes == null ? emptyPartyInfo : JweUtils.concatenateDatalenAndData(apvBytes);
        byte[] suppPubInfo = JweUtils.datalenToBytes(algoKeyBitLen);
        byte[] otherInfo = new byte[algorithmId.length + partyUInfo.length + partyVInfo.length + suppPubInfo.length];
        System.arraycopy(algorithmId, 0, otherInfo, 0, algorithmId.length);
        System.arraycopy(partyUInfo, 0, otherInfo, algorithmId.length, partyUInfo.length);
        System.arraycopy(partyVInfo, 0, otherInfo, algorithmId.length + partyUInfo.length, partyVInfo.length);
        System.arraycopy(suppPubInfo, 0, otherInfo, algorithmId.length + partyUInfo.length + partyVInfo.length, suppPubInfo.length);
        byte[] concatKDF = new byte[36 + otherInfo.length];
        concatKDF[3] = 1;
        System.arraycopy(keyZ, 0, concatKDF, 4, keyZ.length);
        System.arraycopy(otherInfo, 0, concatKDF, 36, otherInfo.length);
        try {
            byte[] round1Hash = MessageDigestUtils.createDigest((byte[])concatKDF, (String)"SHA-256");
            return Arrays.copyOf(round1Hash, algoKeyBitLen / 8);
        }
        catch (Exception ex) {
            LOG.warning("Derived key calculation problem: round hash1 error");
            throw new JweException(JweException.Error.KEY_ENCRYPTION_FAILURE);
        }
    }

    private static byte[] generateKeyZ(ECPrivateKey privateKey, ECPublicKey publicKey) {
        try {
            KeyAgreement ka = KeyAgreement.getInstance("ECDH");
            ka.init(privateKey);
            ka.doPhase(publicKey, true);
            return ka.generateSecret();
        }
        catch (Exception ex) {
            LOG.warning("Derived key calculation problem");
            throw new JweException(JweException.Error.KEY_ENCRYPTION_FAILURE);
        }
    }

    private static byte[] concatenateDatalenAndData(byte[] bytesASCII) {
        byte[] datalen = JweUtils.datalenToBytes(bytesASCII.length);
        byte[] all = new byte[4 + bytesASCII.length];
        System.arraycopy(datalen, 0, all, 0, 4);
        System.arraycopy(bytesASCII, 0, all, 4, bytesASCII.length);
        return all;
    }

    private static byte[] datalenToBytes(int len) {
        ByteBuffer buf = ByteBuffer.allocate(4);
        return buf.putInt(len).array();
    }

    private static JweHeaders prepareJweHeaders(String keyEncryptionAlgo, String contentEncryptionAlgo, String compression) {
        JweHeaders headers = new JweHeaders();
        if (keyEncryptionAlgo != null) {
            headers.setKeyEncryptionAlgorithm(KeyAlgorithm.getAlgorithm(keyEncryptionAlgo));
        }
        headers.setContentEncryptionAlgorithm(ContentAlgorithm.getAlgorithm(contentEncryptionAlgo));
        if (compression != null) {
            headers.setZipAlgorithm(compression);
        }
        return headers;
    }

    private static JweEncryptionProvider createJweEncryptionProvider(KeyEncryptionProvider keyEncryptionProvider, ContentEncryptionProvider ctEncryptionProvider, String contentEncryptionAlgo, String compression) {
        if (keyEncryptionProvider == null && ctEncryptionProvider == null) {
            LOG.warning("Key or content encryptor is not available");
            throw new JweException(JweException.Error.NO_ENCRYPTOR);
        }
        JweHeaders headers = JweUtils.prepareJweHeaders(keyEncryptionProvider != null ? keyEncryptionProvider.getAlgorithm().getJwaName() : null, contentEncryptionAlgo, compression);
        if (keyEncryptionProvider != null) {
            return JweUtils.createJweEncryptionProvider(keyEncryptionProvider, headers);
        }
        return new JweEncryption(new DirectKeyEncryptionAlgorithm(), ctEncryptionProvider);
    }

    private static JweDecryptionProvider createJweDecryptionProvider(KeyDecryptionProvider keyDecryptionProvider, SecretKey ctDecryptionKey, ContentAlgorithm contentDecryptionAlgo) {
        if (keyDecryptionProvider == null && ctDecryptionKey == null) {
            LOG.warning("Key or content encryptor is not available");
            throw new JweException(JweException.Error.NO_ENCRYPTOR);
        }
        if (keyDecryptionProvider != null) {
            return JweUtils.createJweDecryptionProvider(keyDecryptionProvider, contentDecryptionAlgo);
        }
        return JweUtils.getDirectKeyJweDecryption(ctDecryptionKey, contentDecryptionAlgo);
    }

    public static KeyAlgorithm getKeyEncryptionAlgorithm(Message m, Properties props, KeyAlgorithm algo, KeyAlgorithm defaultAlgo) {
        if (algo == null) {
            algo = JweUtils.getKeyEncryptionAlgorithm(m, props, defaultAlgo);
        }
        return algo;
    }

    public static KeyAlgorithm getKeyEncryptionAlgorithm(Properties props, KeyAlgorithm defaultAlgo) {
        return JweUtils.getKeyEncryptionAlgorithm(PhaseInterceptorChain.getCurrentMessage(), props, defaultAlgo);
    }

    public static KeyAlgorithm getKeyEncryptionAlgorithm(Message m, Properties props, KeyAlgorithm defaultAlgo) {
        String algo = KeyManagementUtils.getKeyAlgorithm(m, props, "rs.security.encryption.key.algorithm", defaultAlgo == null ? null : defaultAlgo.getJwaName());
        return algo == null ? null : KeyAlgorithm.getAlgorithm(algo);
    }

    private static KeyAlgorithm getDefaultKeyAlgorithm(JsonWebKey jwk) {
        KeyType keyType = jwk.getKeyType();
        if (KeyType.OCTET == keyType) {
            return KeyAlgorithm.A128GCMKW;
        }
        if (KeyType.RSA == keyType) {
            return KeyAlgorithm.RSA_OAEP;
        }
        return KeyAlgorithm.ECDH_ES_A128KW;
    }

    public static ContentAlgorithm getContentEncryptionAlgorithm(Message m, Properties props, ContentAlgorithm algo, ContentAlgorithm defaultAlgo) {
        if (algo == null) {
            algo = JweUtils.getContentEncryptionAlgorithm(m, props, defaultAlgo);
        }
        return algo;
    }

    public static ContentAlgorithm getContentEncryptionAlgorithm(Properties props, ContentAlgorithm defaultAlgo) {
        return JweUtils.getContentEncryptionAlgorithm(PhaseInterceptorChain.getCurrentMessage(), props, defaultAlgo);
    }

    public static ContentAlgorithm getContentEncryptionAlgorithm(Message m, Properties props, ContentAlgorithm defaultAlgo) {
        String algo = KeyManagementUtils.getKeyAlgorithm(m, props, "rs.security.encryption.content.algorithm", defaultAlgo == null ? null : defaultAlgo.getJwaName());
        return ContentAlgorithm.getAlgorithm(algo);
    }

    private static String encrypt(KeyEncryptionProvider keyEncryptionProvider, ContentAlgorithm contentAlgo, byte[] content, String ct) {
        JweEncryptionProvider jwe = JweUtils.createJweEncryptionProvider(keyEncryptionProvider, contentAlgo, null);
        return jwe.encrypt(content, JweUtils.toJweHeaders(ct));
    }

    private static byte[] decrypt(KeyDecryptionProvider keyDecryptionProvider, ContentAlgorithm contentAlgo, String content) {
        JweDecryptionProvider jwe = JweUtils.createJweDecryptionProvider(keyDecryptionProvider, contentAlgo);
        return jwe.decrypt(content).getContent();
    }

    private static JweHeaders toJweHeaders(String ct) {
        return new JweHeaders(Collections.singletonMap("cty", ct));
    }

    public static void validateJweCertificateChain(List<X509Certificate> certs) {
        Properties props = JweUtils.loadEncryptionInProperties(true);
        KeyManagementUtils.validateCertificateChain(props, certs);
    }

    public static Properties loadEncryptionInProperties(boolean required) {
        Message m = PhaseInterceptorChain.getCurrentMessage();
        return KeyManagementUtils.loadStoreProperties(m, required, "rs.security.encryption.in.properties", "rs.security.encryption.properties");
    }

    public static Properties loadEncryptionOutProperties(boolean required) {
        Message m = PhaseInterceptorChain.getCurrentMessage();
        return KeyManagementUtils.loadStoreProperties(m, required, "rs.security.encryption.out.properties", "rs.security.encryption.properties");
    }

    public static Properties loadEncryptionProperties(String propertiesName, boolean required) {
        Message m = PhaseInterceptorChain.getCurrentMessage();
        return KeyManagementUtils.loadStoreProperties(m, required, propertiesName, null);
    }

    public static void checkEncryptionKeySize(Key key) {
        if (key instanceof RSAKey && ((RSAKey)((Object)key)).getModulus().bitLength() < 2048) {
            LOG.fine("A key of size: " + ((RSAKey)((Object)key)).getModulus().bitLength() + " was used with an RSA encryption algorithm. 2048 is the minimum size that is accepted");
            throw new JweException(JweException.Error.KEY_DECRYPTION_FAILURE);
        }
    }

    public static JsonWebKeys loadPublicKeyEncryptionKeys(Message m, Properties props) {
        String storeType = props.getProperty("rs.security.keystore.type");
        if ("jwk".equals(storeType)) {
            return JwkUtils.loadPublicJwkSet(m, props);
        }
        PublicKey key = KeyManagementUtils.loadPublicKey(m, props);
        JsonWebKey jwk = JwkUtils.fromPublicKey(key, props, "rs.security.encryption.key.algorithm");
        return new JsonWebKeys(jwk);
    }

    private static Properties loadJweProperties(Message m, String propLoc) {
        try {
            return JoseUtils.loadProperties(propLoc, m.getExchange().getBus());
        }
        catch (Exception ex) {
            LOG.warning("JWS init properties are not available");
            throw new JweException(JweException.Error.NO_INIT_PROPERTIES);
        }
    }
}

