/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import com.qcloud.cos.COSClient;
import com.qcloud.cos.ClientConfig;
import com.qcloud.cos.auth.COSCredentialsProvider;
import com.qcloud.cos.endpoint.EndpointBuilder;
import com.qcloud.cos.exception.CosClientException;
import com.qcloud.cos.exception.CosServiceException;
import com.qcloud.cos.exception.ResponseNotCompleteException;
import com.qcloud.cos.http.HttpProtocol;
import com.qcloud.cos.model.AbortMultipartUploadRequest;
import com.qcloud.cos.model.COSObject;
import com.qcloud.cos.model.CompleteMultipartUploadRequest;
import com.qcloud.cos.model.CompleteMultipartUploadResult;
import com.qcloud.cos.model.CopyObjectRequest;
import com.qcloud.cos.model.DeleteObjectRequest;
import com.qcloud.cos.model.GetObjectMetadataRequest;
import com.qcloud.cos.model.GetObjectRequest;
import com.qcloud.cos.model.HeadBucketRequest;
import com.qcloud.cos.model.HeadBucketResult;
import com.qcloud.cos.model.InitiateMultipartUploadRequest;
import com.qcloud.cos.model.InitiateMultipartUploadResult;
import com.qcloud.cos.model.ListObjectsRequest;
import com.qcloud.cos.model.ObjectListing;
import com.qcloud.cos.model.ObjectMetadata;
import com.qcloud.cos.model.PartETag;
import com.qcloud.cos.model.PutObjectRequest;
import com.qcloud.cos.model.PutObjectResult;
import com.qcloud.cos.model.RenameRequest;
import com.qcloud.cos.model.SSEAlgorithm;
import com.qcloud.cos.model.SSECOSKeyManagementParams;
import com.qcloud.cos.model.SSECustomerKey;
import com.qcloud.cos.model.StorageClass;
import com.qcloud.cos.model.UploadPartRequest;
import com.qcloud.cos.model.UploadPartResult;
import com.qcloud.cos.region.Region;
import com.qcloud.cos.utils.Base64;
import com.qcloud.cos.utils.IOUtils;
import com.qcloud.cos.utils.Jackson;
import com.qcloud.cos.utils.StringUtils;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ThreadLocalRandom;
import org.apache.commons.codec.binary.Hex;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CosFileSystem;
import org.apache.hadoop.fs.CosNEncryptionMethods;
import org.apache.hadoop.fs.CosNEncryptionSecrets;
import org.apache.hadoop.fs.CosNPartialListing;
import org.apache.hadoop.fs.CosNResultInfo;
import org.apache.hadoop.fs.CosNUtils;
import org.apache.hadoop.fs.CosNXAttr;
import org.apache.hadoop.fs.FileMetadata;
import org.apache.hadoop.fs.NativeFileSystemStore;
import org.apache.hadoop.fs.auth.COSCredentialProviderList;
import org.apache.hadoop.fs.cosn.CustomerDomainEndpointResolver;
import org.apache.hadoop.fs.cosn.ResettableFileInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class CosNativeFileSystemStore
implements NativeFileSystemStore {
    public static final Logger LOG = LoggerFactory.getLogger(CosNativeFileSystemStore.class);
    private static final String XATTR_PREFIX = "cosn-xattr-";
    private COSClient cosClient;
    private COSCredentialProviderList cosCredentialProviderList;
    private String bucketName;
    private StorageClass storageClass;
    private int maxRetryTimes;
    private int trafficLimit;
    private boolean crc32cEnabled;
    private boolean completeMPUCheckEnabled;
    private CosNEncryptionSecrets encryptionSecrets;
    private boolean isMergeBucket;
    private CustomerDomainEndpointResolver customerDomainEndpointResolver;

    private void initCOSClient(URI uri, Configuration conf) throws IOException {
        String versionNum;
        ClientConfig config;
        this.cosCredentialProviderList = CosNUtils.createCosCredentialsProviderSet(uri, conf);
        String customerDomain = conf.get("fs.cosn.customer.domain");
        if (null == customerDomain) {
            String endpointSuffix;
            String region = conf.get("fs.cosn.bucket.region");
            if (null == region) {
                region = conf.get("fs.cosn.userinfo.region");
            }
            if (null == (endpointSuffix = conf.get("fs.cosn.bucket.endpoint_suffix"))) {
                endpointSuffix = conf.get("fs.cosn.userinfo.endpoint_suffix");
            }
            if (null == region && null == endpointSuffix) {
                String exceptionMsg = String.format("config '%s' and '%s' specify at least one.", "fs.cosn.bucket.region", "fs.cosn.bucket.endpoint_suffix");
                throw new IOException(exceptionMsg);
            }
            if (null == region) {
                config = new ClientConfig(new Region(""));
                config.setEndPointSuffix(endpointSuffix);
            } else {
                config = new ClientConfig(new Region(region));
            }
        } else {
            config = new ClientConfig(new Region(""));
            LOG.info("Use Customer Domain is {}", (Object)customerDomain);
            this.customerDomainEndpointResolver = new CustomerDomainEndpointResolver();
            this.customerDomainEndpointResolver.setEndpoint(customerDomain);
            config.setEndpointBuilder((EndpointBuilder)this.customerDomainEndpointResolver);
        }
        boolean useHttps = conf.getBoolean("fs.cosn.useHttps", false);
        if (useHttps) {
            config.setHttpProtocol(HttpProtocol.https);
        }
        int socketTimeoutSec = conf.getInt("fs.cosn.client.socket.timeoutsec", 30);
        config.setSocketTimeout(socketTimeoutSec * 1000);
        this.crc32cEnabled = conf.getBoolean("fs.cosn.crc32c.checksum.enabled", false);
        this.completeMPUCheckEnabled = conf.getBoolean("fs.cosn.complete.mpu.check", true);
        String httpProxyIp = conf.getTrimmed("fs.cosn.http.proxy.ip");
        int httpProxyPort = conf.getInt("fs.cosn.http.proxy.port", -1);
        if (null != httpProxyIp && !httpProxyIp.isEmpty()) {
            config.setHttpProxyIp(httpProxyIp);
            if (httpProxyPort >= 0) {
                config.setHttpProxyPort(httpProxyPort);
            } else if (useHttps) {
                LOG.warn("Proxy IP set without port. Using HTTPS default port 443");
                config.setHttpProxyPort(443);
            } else {
                LOG.warn("Proxy IP set without port, Using HTTP default port 80");
                config.setHttpProxyPort(80);
            }
            String proxyUserName = conf.get("fs.cosn.http.proxy.username");
            String proxyPassword = conf.get("fs.cosn.http.proxy.password");
            if ((null == proxyUserName || proxyUserName.isEmpty()) != (null == proxyPassword || proxyPassword.isEmpty())) {
                String exceptionMessage = String.format("Proxy error: '%s' or '%s' set without the other.", "fs.cosn.http.proxy.username", "fs.cosn.http.proxy.password");
                throw new IllegalArgumentException(exceptionMessage);
            }
            config.setProxyUsername(proxyUserName);
            config.setProxyPassword(proxyPassword);
        }
        String versionInfo = (versionNum = this.getPluginVersionInfo()).equals("unknown") ? "cos-hadoop-plugin-v8.0.2" : "cos-hadoop-plugin-v" + versionNum;
        String userAgent = conf.get("fs.cosn.user.agent", versionInfo);
        String emrVersion = conf.get("fs.emr.version");
        if (null != emrVersion && !emrVersion.isEmpty()) {
            userAgent = String.format("%s on %s", userAgent, emrVersion);
        }
        config.setUserAgent(userAgent);
        this.maxRetryTimes = conf.getInt("fs.cosn.maxRetries", 200);
        int clientMaxRetryTimes = conf.getInt("fs.cosn.client.maxRetries", 5);
        config.setMaxErrorRetry(clientMaxRetryTimes);
        LOG.info("hadoop cos retry times: {}, cos client retry times: {}", (Object)this.maxRetryTimes, (Object)clientMaxRetryTimes);
        config.setMaxConnectionsCount(conf.getInt("fs.cosn.max.connection.num", 2048));
        String serverSideEncryptionAlgorithm = conf.get("fs.cosn.server-side-encryption.algorithm", "");
        CosNEncryptionMethods cosSSE = CosNEncryptionMethods.getMethod(serverSideEncryptionAlgorithm);
        String sseKey = conf.get("fs.cosn.server-side-encryption.key", "");
        String sseContext = conf.get("fs.cosn.server-side-encryption.context", "");
        this.checkEncryptionMethod(config, cosSSE, sseKey);
        this.encryptionSecrets = new CosNEncryptionSecrets(cosSSE, sseKey, sseContext);
        this.trafficLimit = conf.getInt("fs.cosn.traffic.limit", -1);
        if (this.trafficLimit >= 0 && (this.trafficLimit < 819200 || this.trafficLimit > 0x32000000)) {
            String exceptionMessage = String.format("The '%s' needs to be between %d and %d. but %d.", "fs.cosn.traffic.limit", 819200, 0x32000000, this.trafficLimit);
            throw new IllegalArgumentException(exceptionMessage);
        }
        this.cosClient = new COSClient((COSCredentialsProvider)this.cosCredentialProviderList, config);
    }

    @Override
    public void initialize(URI uri, Configuration conf) throws IOException {
        try {
            this.initCOSClient(uri, conf);
            this.bucketName = uri.getHost();
            this.isMergeBucket = false;
            String storageClass = conf.get("fs.cosn.storage.class");
            if (null != storageClass && !storageClass.isEmpty()) {
                try {
                    this.storageClass = StorageClass.fromValue((String)storageClass);
                }
                catch (IllegalArgumentException e) {
                    String exceptionMessage = String.format("The specified storage class [%s] is invalid. The supported storage classes are: %s, %s, %s, %s and %s.", storageClass, StorageClass.Standard.toString(), StorageClass.Standard_IA.toString(), StorageClass.Maz_Standard.toString(), StorageClass.Maz_Standard_IA.toString(), StorageClass.Archive.toString());
                    throw new IllegalArgumentException(exceptionMessage);
                }
            }
            if (null != this.storageClass && StorageClass.Archive == this.storageClass) {
                LOG.warn("The storage class of the CosN FileSystem is set to {}. Some file operations may not be supported.", (Object)this.storageClass);
            }
        }
        catch (Exception e) {
            this.handleException(e, "");
        }
    }

    @Override
    public void setMergeBucket(boolean isMergeBucket) {
        this.isMergeBucket = isMergeBucket;
    }

    private void storeFileWithRetry(String key, InputStream inputStream, byte[] md5Hash, long length) throws IOException {
        try {
            ObjectMetadata objectMetadata = new ObjectMetadata();
            if (null != md5Hash) {
                objectMetadata.setContentMD5(Base64.encodeAsString((byte[])md5Hash));
            }
            objectMetadata.setContentLength(length);
            if (this.crc32cEnabled) {
                objectMetadata.setHeader("x-cos-crc32c-flag", (Object)"cosn");
            }
            PutObjectRequest putObjectRequest = new PutObjectRequest(this.bucketName, key, inputStream, objectMetadata);
            if (null != this.storageClass) {
                putObjectRequest.setStorageClass(this.storageClass);
            }
            if (this.trafficLimit >= 0) {
                putObjectRequest.setTrafficLimit(this.trafficLimit);
            }
            this.setEncryptionMetadata(putObjectRequest, objectMetadata);
            PutObjectResult putObjectResult = (PutObjectResult)this.callCOSClientWithRetry(putObjectRequest);
            LOG.debug("Store the file successfully. cos key: {}, ETag: {}.", (Object)key, (Object)putObjectResult.getETag());
        }
        catch (CosServiceException cse) {
            int statusCode = cse.getStatusCode();
            if (statusCode == 409) {
                FileMetadata fileMetadata = this.queryObjectMetadata(key);
                if (null == fileMetadata) {
                    this.handleException((Exception)((Object)cse), key);
                }
                LOG.warn("Upload the cos key [{}] concurrently", (Object)key);
            } else {
                this.handleException((Exception)((Object)cse), key);
            }
        }
        catch (Exception e) {
            String errMsg = String.format("Store the file failed, cos key: %s, exception: %s.", key, e.toString());
            this.handleException(new Exception(errMsg), key);
        }
    }

    @Override
    public HeadBucketResult headBucket(String bucketName) throws IOException {
        HeadBucketRequest headBucketRequest = new HeadBucketRequest(bucketName);
        try {
            HeadBucketResult result = (HeadBucketResult)this.callCOSClientWithRetry(headBucketRequest);
            return result;
        }
        catch (Exception e) {
            String errMsg = String.format("head bucket [%s] occurs an exception", bucketName, e.toString());
            this.handleException(new Exception(errMsg), bucketName);
            return null;
        }
    }

    @Override
    public void storeFile(String key, File file, byte[] md5Hash) throws IOException {
        if (null != md5Hash) {
            LOG.debug("Store the file, local path: {}, length: {}, md5hash: {}.", new Object[]{file.getCanonicalPath(), file.length(), Hex.encodeHexString((byte[])md5Hash)});
        }
        this.storeFileWithRetry(key, new ResettableFileInputStream(file), md5Hash, file.length());
    }

    @Override
    public void storeFile(String key, InputStream inputStream, byte[] md5Hash, long contentLength) throws IOException {
        if (null != md5Hash) {
            LOG.debug("Store the file to the cos key: {}, input stream md5 hash: {}, content length: {}.", new Object[]{key, Hex.encodeHexString((byte[])md5Hash), contentLength});
        }
        this.storeFileWithRetry(key, inputStream, md5Hash, contentLength);
    }

    @Override
    public void storeEmptyFile(String key) throws IOException {
        LOG.debug("Store an empty file to the key: {}.", (Object)key);
        ObjectMetadata objectMetadata = new ObjectMetadata();
        objectMetadata.setContentLength(0L);
        if (this.crc32cEnabled) {
            objectMetadata.setHeader("x-cos-crc32c-flag", (Object)"cosn");
        }
        ByteArrayInputStream input = new ByteArrayInputStream(new byte[0]);
        PutObjectRequest putObjectRequest = new PutObjectRequest(this.bucketName, key, (InputStream)input, objectMetadata);
        if (null != this.storageClass) {
            putObjectRequest.setStorageClass(this.storageClass);
        }
        try {
            PutObjectResult putObjectResult = (PutObjectResult)this.callCOSClientWithRetry(putObjectRequest);
            LOG.debug("Store the empty file successfully, cos key: {}, ETag: {}.", (Object)key, (Object)putObjectResult.getETag());
        }
        catch (CosServiceException cse) {
            int statusCode = cse.getStatusCode();
            if (statusCode == 409) {
                FileMetadata fileMetadata = this.queryObjectMetadata(key);
                if (null == fileMetadata) {
                    this.handleException((Exception)((Object)cse), key);
                }
                LOG.warn("Upload the file [{}] concurrently.", (Object)key);
            } else {
                this.handleException((Exception)((Object)cse), key);
            }
        }
        catch (Exception e) {
            String errMsg = String.format("Store the empty file failed, cos key: %s, exception: %s", key, e.toString());
            this.handleException(new Exception(errMsg), key);
        }
    }

    @Override
    public PartETag uploadPart(File file, String key, String uploadId, int partNum, byte[] md5Hash) throws IOException {
        ResettableFileInputStream inputStream = new ResettableFileInputStream(file);
        return this.uploadPart(inputStream, key, uploadId, partNum, file.length(), md5Hash);
    }

    @Override
    public PartETag uploadPart(InputStream inputStream, String key, String uploadId, int partNum, long partSize, byte[] md5Hash) throws IOException {
        LOG.debug("Upload the part to the cos key [{}]. upload id: {}, part number: {}, part size: {}", new Object[]{key, uploadId, partNum, partSize});
        ObjectMetadata objectMetadata = new ObjectMetadata();
        if (this.crc32cEnabled) {
            objectMetadata.setHeader("x-cos-crc32c-flag", (Object)"cosn");
        }
        UploadPartRequest uploadPartRequest = new UploadPartRequest();
        uploadPartRequest.setBucketName(this.bucketName);
        uploadPartRequest.setUploadId(uploadId);
        uploadPartRequest.setInputStream(inputStream);
        uploadPartRequest.setPartNumber(partNum);
        uploadPartRequest.setPartSize(partSize);
        uploadPartRequest.setObjectMetadata(objectMetadata);
        if (null != md5Hash) {
            uploadPartRequest.setMd5Digest(Base64.encodeAsString((byte[])md5Hash));
        }
        uploadPartRequest.setKey(key);
        if (this.trafficLimit >= 0) {
            uploadPartRequest.setTrafficLimit(this.trafficLimit);
        }
        this.setEncryptionMetadata(uploadPartRequest, objectMetadata);
        try {
            UploadPartResult uploadPartResult = (UploadPartResult)this.callCOSClientWithRetry(uploadPartRequest);
            return uploadPartResult.getPartETag();
        }
        catch (Exception e) {
            String errMsg = String.format("The current thread:%d, cos key: %s, upload id: %s, part num: %d, exception: %s", Thread.currentThread().getId(), key, uploadId, partNum, e.toString());
            this.handleException(new Exception(errMsg), key);
            return null;
        }
    }

    @Override
    public void abortMultipartUpload(String key, String uploadId) throws IOException {
        LOG.info("Ready to abort the multipart upload. cos key: {}, upload id: {}.", (Object)key, (Object)uploadId);
        try {
            AbortMultipartUploadRequest abortMultipartUploadRequest = new AbortMultipartUploadRequest(this.bucketName, key, uploadId);
            this.callCOSClientWithRetry(abortMultipartUploadRequest);
        }
        catch (Exception e) {
            String errMsg = String.format("Aborting the multipart upload failed. cos key: %s, upload id: %s. exception: %s.", key, uploadId, e.toString());
            this.handleException(new Exception(errMsg), key);
        }
    }

    @Override
    public String getUploadId(String key) throws IOException {
        if (null == key || key.length() == 0) {
            return "";
        }
        ObjectMetadata objectMetadata = new ObjectMetadata();
        if (this.crc32cEnabled) {
            objectMetadata.setHeader("x-cos-crc32c-flag", (Object)"cosn");
        }
        InitiateMultipartUploadRequest initiateMultipartUploadRequest = new InitiateMultipartUploadRequest(this.bucketName, key);
        if (null != this.storageClass) {
            initiateMultipartUploadRequest.setStorageClass(this.storageClass);
        }
        initiateMultipartUploadRequest.setObjectMetadata(objectMetadata);
        this.setEncryptionMetadata(initiateMultipartUploadRequest, objectMetadata);
        try {
            InitiateMultipartUploadResult initiateMultipartUploadResult = (InitiateMultipartUploadResult)this.callCOSClientWithRetry(initiateMultipartUploadRequest);
            return initiateMultipartUploadResult.getUploadId();
        }
        catch (Exception e) {
            String errMsg = String.format("Get the upload id failed, cos key: %s, exception: %s", key, e.toString());
            this.handleException(new Exception(errMsg), key);
            return null;
        }
    }

    @Override
    public CompleteMultipartUploadResult completeMultipartUpload(String key, String uploadId, List<PartETag> partETagList) throws IOException {
        Collections.sort(partETagList, new Comparator<PartETag>(){

            @Override
            public int compare(PartETag o1, PartETag o2) {
                return o1.getPartNumber() - o2.getPartNumber();
            }
        });
        try {
            ObjectMetadata objectMetadata = new ObjectMetadata();
            if (this.crc32cEnabled) {
                objectMetadata.setHeader("x-cos-crc32c-flag", (Object)"cosn");
            }
            CompleteMultipartUploadRequest completeMultipartUploadRequest = new CompleteMultipartUploadRequest(this.bucketName, key, uploadId, partETagList);
            completeMultipartUploadRequest.setObjectMetadata(objectMetadata);
            return (CompleteMultipartUploadResult)this.callCOSClientWithRetry(completeMultipartUploadRequest);
        }
        catch (Exception e) {
            String errMsg = String.format("Complete the multipart upload failed. cos key: %s, upload id: %s, exception: %s", key, uploadId, e.toString());
            this.handleException(new Exception(errMsg), key);
            return null;
        }
    }

    private FileMetadata queryObjectMetadata(String key) throws IOException {
        return this.queryObjectMetadata(key, null);
    }

    private FileMetadata queryObjectMetadata(String key, CosNResultInfo info) throws IOException {
        LOG.debug("Query Object metadata. cos key: {}.", (Object)key);
        GetObjectMetadataRequest getObjectMetadataRequest = new GetObjectMetadataRequest(this.bucketName, key);
        this.setEncryptionMetadata(getObjectMetadataRequest, new ObjectMetadata());
        try {
            ObjectMetadata objectMetadata = (ObjectMetadata)this.callCOSClientWithRetry(getObjectMetadataRequest);
            long mtime = 0L;
            long fileSize = 0L;
            if (objectMetadata.getLastModified() != null) {
                mtime = objectMetadata.getLastModified().getTime();
            }
            fileSize = objectMetadata.getContentLength();
            String ETag = objectMetadata.getETag();
            String crc64ecm = objectMetadata.getCrc64Ecma();
            String crc32cm = (String)objectMetadata.getRawMetadataValue("x-cos-hash-crc32c");
            String versionId = objectMetadata.getVersionId();
            HashMap<String, byte[]> userMetadata = null;
            if (objectMetadata.getUserMetadata() != null) {
                userMetadata = new HashMap<String, byte[]>();
                for (Map.Entry userMetadataEntry : objectMetadata.getUserMetadata().entrySet()) {
                    if (!((String)userMetadataEntry.getKey()).startsWith(CosNativeFileSystemStore.ensureValidAttributeName(XATTR_PREFIX))) continue;
                    String xAttrJsonStr = new String(Base64.decode((String)((String)userMetadataEntry.getValue())), StandardCharsets.UTF_8);
                    CosNXAttr cosNXAttr = null;
                    try {
                        cosNXAttr = (CosNXAttr)Jackson.fromJsonString((String)xAttrJsonStr, CosNXAttr.class);
                    }
                    catch (CosClientException e) {
                        LOG.warn("Parse the xAttr failed. name: {}, XAttJsonStr: {}.", userMetadataEntry.getKey(), (Object)xAttrJsonStr);
                        continue;
                    }
                    if (null == cosNXAttr) continue;
                    userMetadata.put(cosNXAttr.getName(), cosNXAttr.getValue().getBytes(CosFileSystem.METADATA_ENCODING));
                }
            }
            boolean isFile = true;
            if (this.isMergeBucket) {
                if (objectMetadata.isFileModeDir() || key.equals("/")) {
                    isFile = false;
                }
            } else {
                isFile = !key.endsWith("/");
            }
            FileMetadata fileMetadata = new FileMetadata(key, fileSize, mtime, isFile, ETag, crc64ecm, crc32cm, versionId, objectMetadata.getStorageClass(), userMetadata);
            if (info != null) {
                info.setRequestID(objectMetadata.getRequestId());
            }
            LOG.debug("Retrieve the file metadata. cos key: {}, ETag:{}, length:{}, crc64ecm: {}.", new Object[]{key, objectMetadata.getETag(), objectMetadata.getContentLength(), objectMetadata.getCrc64Ecma()});
            return fileMetadata;
        }
        catch (CosServiceException e) {
            if (info != null) {
                info.setRequestID(e.getRequestId());
            }
            if (e.getStatusCode() != 404) {
                String errorMsg = String.format("Retrieve the file metadata file failure. cos key: %s, exception: %s", key, e.toString());
                this.handleException(new Exception(errorMsg), key);
            }
            return null;
        }
    }

    @Override
    public FileMetadata retrieveMetadata(String key) throws IOException {
        return this.retrieveMetadata(key, null);
    }

    @Override
    public FileMetadata retrieveMetadata(String key, CosNResultInfo info) throws IOException {
        FileMetadata fileMetadata;
        if (key.endsWith("/")) {
            key = key.substring(0, key.length() - 1);
        }
        if (!key.isEmpty() && (fileMetadata = this.queryObjectMetadata(key, info)) != null) {
            return fileMetadata;
        }
        key = key + "/";
        return this.queryObjectMetadata(key, info);
    }

    @Override
    public byte[] retrieveAttribute(String key, String attribute) throws IOException {
        LOG.debug("Get the extended attribute. cos key: {}, attribute: {}.", (Object)key, (Object)attribute);
        FileMetadata fileMetadata = this.retrieveMetadata(key);
        if (null != fileMetadata && null != fileMetadata.getUserAttributes()) {
            return fileMetadata.getUserAttributes().get(attribute);
        }
        return null;
    }

    @Override
    public void storeDirAttribute(String key, String attribute, byte[] value) throws IOException {
        LOG.debug("Store a attribute to the specified directory. cos key: {}, attribute: {}, value: {}.", new Object[]{key, attribute, new String(value, CosFileSystem.METADATA_ENCODING)});
        if (!key.endsWith("/")) {
            key = key + "/";
        }
        this.storeAttribute(key, attribute, value, false);
    }

    @Override
    public void storeFileAttribute(String key, String attribute, byte[] value) throws IOException {
        LOG.debug("Store a attribute to the specified file. cos key: {}, attribute: {}, value: {}.", new Object[]{key, attribute, new String(value, CosFileSystem.METADATA_ENCODING)});
        this.storeAttribute(key, attribute, value, false);
    }

    @Override
    public void removeDirAttribute(String key, String attribute) throws IOException {
        LOG.debug("Remove the attribute from the specified directory. cos key: {}, attribute: {}.", (Object)key, (Object)attribute);
        if (!key.endsWith("/")) {
            key = key + "/";
        }
        this.storeAttribute(key, attribute, null, true);
    }

    @Override
    public void removeFileAttribute(String key, String attribute) throws IOException {
        LOG.debug("Remove the attribute from the specified file. cos key: {}, attribute: {}.", (Object)key, (Object)attribute);
        this.storeAttribute(key, attribute, null, true);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void storeAttribute(String key, String attribute, byte[] value, boolean deleted) throws IOException {
        ObjectMetadata objectMetadata;
        block13: {
            if (deleted) {
                LOG.debug("Delete the extended attribute. cos key: {}, attribute: {}.", (Object)key, (Object)attribute);
            }
            if (null != value && !deleted) {
                LOG.debug("Store the extended attribute. cos key: {}, attribute: {}, value: {}.", new Object[]{key, attribute, new String(value, StandardCharsets.UTF_8)});
            }
            if (null == value && !deleted) {
                throw new IOException("The attribute value to be set can not be null.");
            }
            GetObjectMetadataRequest getObjectMetadataRequest = new GetObjectMetadataRequest(this.bucketName, key);
            this.setEncryptionMetadata(getObjectMetadataRequest, new ObjectMetadata());
            objectMetadata = null;
            try {
                objectMetadata = (ObjectMetadata)this.callCOSClientWithRetry(getObjectMetadataRequest);
            }
            catch (CosServiceException e) {
                if (e.getStatusCode() == 404) break block13;
                String errorMessage = String.format("Retrieve the file metadata failed. cos key: %s, exception: %s.", key, e.toString());
                this.handleException(new Exception(errorMessage), key);
            }
        }
        if (null == objectMetadata) return;
        HashMap<String, String> userMetadata = objectMetadata.getUserMetadata();
        if (deleted) {
            if (null == userMetadata) return;
            userMetadata.remove(CosNativeFileSystemStore.ensureValidAttributeName(attribute));
        } else {
            if (null == userMetadata) {
                userMetadata = new HashMap<String, String>();
            }
            CosNXAttr cosNXAttr = new CosNXAttr();
            cosNXAttr.setName(attribute);
            cosNXAttr.setValue(new String(value, CosFileSystem.METADATA_ENCODING));
            String xAttrJsonStr = Jackson.toJsonString((Object)cosNXAttr);
            userMetadata.put(CosNativeFileSystemStore.ensureValidAttributeName(XATTR_PREFIX + attribute), Base64.encodeAsString((byte[])xAttrJsonStr.getBytes(StandardCharsets.UTF_8)));
        }
        objectMetadata.setUserMetadata(userMetadata);
        if (this.crc32cEnabled) {
            objectMetadata.setHeader("x-cos-crc32c-flag", (Object)"cosn");
        }
        CopyObjectRequest copyObjectRequest = new CopyObjectRequest(this.bucketName, key, this.bucketName, key);
        if (null != objectMetadata.getStorageClass()) {
            copyObjectRequest.setStorageClass(objectMetadata.getStorageClass());
        }
        copyObjectRequest.setNewObjectMetadata(objectMetadata);
        copyObjectRequest.setRedirectLocation("Replaced");
        this.setEncryptionMetadata(copyObjectRequest, objectMetadata);
        if (null != this.customerDomainEndpointResolver && null != this.customerDomainEndpointResolver.getEndpoint()) {
            copyObjectRequest.setSourceEndpointBuilder((EndpointBuilder)this.customerDomainEndpointResolver);
        }
        try {
            this.callCOSClientWithRetry(copyObjectRequest);
            return;
        }
        catch (Exception e) {
            String errMsg = String.format("Failed to modify the user-defined attributes. cos key: %s, attribute: %s, exception: %s.", key, attribute, e.toString());
            this.handleException(new Exception(errMsg), key);
        }
    }

    @Override
    public InputStream retrieve(String key) throws IOException {
        LOG.debug("Retrieve the key: {}.", (Object)key);
        GetObjectRequest getObjectRequest = new GetObjectRequest(this.bucketName, key);
        if (this.trafficLimit >= 0) {
            getObjectRequest.setTrafficLimit(this.trafficLimit);
        }
        this.setEncryptionMetadata(getObjectRequest, new ObjectMetadata());
        try {
            COSObject cosObject = (COSObject)this.callCOSClientWithRetry(getObjectRequest);
            return cosObject.getObjectContent();
        }
        catch (Exception e) {
            String errMsg = String.format("Retrieving the cos key [%s] occurs an exception: %s", key, e.toString());
            this.handleException(new Exception(errMsg), key);
            return null;
        }
    }

    @Override
    public InputStream retrieve(String key, long byteRangeStart) throws IOException {
        try {
            LOG.debug("Retrieve the cos key: {}, byte range start: {}.", (Object)key, (Object)byteRangeStart);
            long fileSize = this.getFileLength(key);
            long byteRangeEnd = fileSize - 1L;
            GetObjectRequest getObjectRequest = new GetObjectRequest(this.bucketName, key);
            if (this.trafficLimit >= 0) {
                getObjectRequest.setTrafficLimit(this.trafficLimit);
            }
            this.setEncryptionMetadata(getObjectRequest, new ObjectMetadata());
            if (byteRangeEnd >= byteRangeStart) {
                getObjectRequest.setRange(byteRangeStart, fileSize - 1L);
            }
            COSObject cosObject = (COSObject)this.callCOSClientWithRetry(getObjectRequest);
            return cosObject.getObjectContent();
        }
        catch (Exception e) {
            String errMsg = String.format("Retrieving key [%s] with byteRangeStart [%d] occurs an exception: %s.", key, byteRangeStart, e.toString());
            this.handleException(new Exception(errMsg), key);
            return null;
        }
    }

    @Override
    public InputStream retrieveBlock(String key, long byteRangeStart, long byteRangeEnd) throws IOException {
        LOG.debug("Retrieve the cos key: {}, byte range start: {}, byte range end: {}.", new Object[]{key, byteRangeStart, byteRangeEnd});
        try {
            GetObjectRequest request = new GetObjectRequest(this.bucketName, key);
            request.setRange(byteRangeStart, byteRangeEnd);
            if (this.trafficLimit >= 0) {
                request.setTrafficLimit(this.trafficLimit);
            }
            this.setEncryptionMetadata(request, new ObjectMetadata());
            COSObject cosObject = (COSObject)this.callCOSClientWithRetry(request);
            return cosObject.getObjectContent();
        }
        catch (CosServiceException e) {
            String errMsg = String.format("Retrieving the key %s with the byteRangeStart [%d] occurs an exception: %s.", key, byteRangeStart, e.toString());
            this.handleException(new Exception(errMsg), key);
        }
        catch (CosClientException e) {
            String errMsg = String.format("Retrieving key %s with the byteRangeStart [%d] and the byteRangeEnd [%d] occurs an exception: %s.", key, byteRangeStart, byteRangeEnd, e.toString());
            this.handleException(new Exception(errMsg), key);
        }
        return null;
    }

    @Override
    public boolean retrieveBlock(String key, long byteRangeStart, long blockSize, String localBlockPath) throws IOException {
        long fileSize = this.getFileLength(key);
        long byteRangeEnd = 0L;
        try {
            GetObjectRequest request = new GetObjectRequest(this.bucketName, key);
            if (this.trafficLimit >= 0) {
                request.setTrafficLimit(this.trafficLimit);
            }
            this.setEncryptionMetadata(request, new ObjectMetadata());
            if (fileSize > 0L) {
                byteRangeEnd = Math.min(fileSize - 1L, byteRangeStart + blockSize - 1L);
                request.setRange(byteRangeStart, byteRangeEnd);
            }
            this.cosClient.getObject(request, new File(localBlockPath));
            return true;
        }
        catch (Exception e) {
            String errMsg = String.format("Retrieving block key [%s] with range [%d - %d] occurs an exception: %s", key, byteRangeStart, byteRangeEnd, e.getMessage());
            this.handleException(new Exception(errMsg), key);
            return false;
        }
    }

    @Override
    public CosNPartialListing list(String prefix, int maxListingLength) throws IOException {
        return this.list(prefix, maxListingLength, null);
    }

    @Override
    public CosNPartialListing list(String prefix, int maxListingLength, CosNResultInfo info) throws IOException {
        return this.list(prefix, maxListingLength, null, false, info);
    }

    @Override
    public CosNPartialListing list(String prefix, int maxListingLength, String priorLastKey, boolean recurse) throws IOException {
        return this.list(prefix, maxListingLength, priorLastKey, recurse, null);
    }

    @Override
    public CosNPartialListing list(String prefix, int maxListingLength, String priorLastKey, boolean recurse, CosNResultInfo info) throws IOException {
        return this.list(prefix, recurse ? null : "/", maxListingLength, priorLastKey, info);
    }

    private CosNPartialListing list(String prefix, String delimiter, int maxListingLength, String priorLastKey, CosNResultInfo info) throws IOException {
        LOG.debug("List the cos key prefix: {}, max listing length: {}, delimiter: {}, prior last key: {}.", new Object[]{prefix, delimiter, maxListingLength, priorLastKey});
        if (!prefix.startsWith("/")) {
            prefix = prefix + "/";
        }
        ListObjectsRequest listObjectsRequest = new ListObjectsRequest();
        listObjectsRequest.setBucketName(this.bucketName);
        listObjectsRequest.setPrefix(prefix);
        listObjectsRequest.setDelimiter(delimiter);
        listObjectsRequest.setMarker(priorLastKey);
        listObjectsRequest.setMaxKeys(Integer.valueOf(maxListingLength));
        ObjectListing objectListing = null;
        try {
            objectListing = (ObjectListing)this.callCOSClientWithRetry(listObjectsRequest);
        }
        catch (Exception e) {
            String errMsg = String.format("prefix: %s, delimiter: %s, maxListingLength: %d, priorLastKey: %s. list occur a exception: %s", prefix, delimiter == null ? "" : delimiter, maxListingLength, priorLastKey, e.toString());
            LOG.error(errMsg);
            this.handleException(new Exception(errMsg), prefix);
        }
        if (null == objectListing) {
            String errMessage = String.format("List objects failed for the prefix: %s, delimiter: %s, maxListingLength:%d, priorLastKey: %s.", prefix, delimiter == null ? "" : delimiter, maxListingLength, priorLastKey);
            this.handleException(new Exception(errMessage), prefix);
        }
        ArrayList<FileMetadata> fileMetadataArray = new ArrayList<FileMetadata>();
        ArrayList<FileMetadata> commonPrefixArray = new ArrayList<FileMetadata>();
        List summaries = objectListing.getObjectSummaries();
        boolean isKeySamePrefix = false;
        for (Object cosObjectSummary : summaries) {
            String filePath = cosObjectSummary.getKey();
            if (!filePath.startsWith("/")) {
                filePath = "/" + filePath;
            }
            if (filePath.equals(prefix)) {
                isKeySamePrefix = true;
                continue;
            }
            long mtime = 0L;
            if (cosObjectSummary.getLastModified() != null) {
                mtime = cosObjectSummary.getLastModified().getTime();
            }
            long fileLen = cosObjectSummary.getSize();
            String fileEtag = cosObjectSummary.getETag();
            if (cosObjectSummary.getKey().endsWith("/") && cosObjectSummary.getSize() == 0L) {
                fileMetadataArray.add(new FileMetadata(filePath, fileLen, mtime, false, fileEtag, null, null, null, cosObjectSummary.getStorageClass()));
                continue;
            }
            fileMetadataArray.add(new FileMetadata(filePath, fileLen, mtime, true, fileEtag, null, null, null, cosObjectSummary.getStorageClass()));
        }
        List commonPrefixes = objectListing.getCommonPrefixes();
        for (String commonPrefix : commonPrefixes) {
            if (!commonPrefix.startsWith("/")) {
                commonPrefix = "/" + commonPrefix;
            }
            commonPrefixArray.add(new FileMetadata(commonPrefix, 0L, 0L, false));
        }
        FileMetadata[] fileMetadata = new FileMetadata[fileMetadataArray.size()];
        for (int i = 0; i < fileMetadataArray.size(); ++i) {
            fileMetadata[i] = (FileMetadata)fileMetadataArray.get(i);
        }
        FileMetadata[] commonPrefixMetaData = new FileMetadata[commonPrefixArray.size()];
        for (int i = 0; i < commonPrefixArray.size(); ++i) {
            commonPrefixMetaData[i] = (FileMetadata)commonPrefixArray.get(i);
        }
        if (!objectListing.isTruncated()) {
            CosNPartialListing ret = new CosNPartialListing(null, fileMetadata, commonPrefixMetaData);
            if (info != null) {
                info.setRequestID(objectListing.getRequestId());
                info.setKeySameToPrefix(isKeySamePrefix);
            }
            return ret;
        }
        CosNPartialListing ret = new CosNPartialListing(objectListing.getNextMarker(), fileMetadata, commonPrefixMetaData);
        if (info != null) {
            info.setRequestID(objectListing.getRequestId());
            info.setKeySameToPrefix(isKeySamePrefix);
        }
        return ret;
    }

    @Override
    public void delete(String key) throws IOException {
        LOG.debug("Delete the cos key: {} from bucket: {}.", (Object)key, (Object)this.bucketName);
        try {
            DeleteObjectRequest deleteObjectRequest = new DeleteObjectRequest(this.bucketName, key);
            this.callCOSClientWithRetry(deleteObjectRequest);
        }
        catch (Exception e) {
            String errMsg = String.format("Deleting the cos key [%s] occurs an exception: %s", key, e.toString());
            this.handleException(new Exception(errMsg), key);
        }
    }

    @Override
    public void deleteRecursive(String key) throws IOException {
        LOG.debug("Delete the cos key recursive: {} from bucket: {}.", (Object)key, (Object)this.bucketName);
        try {
            DeleteObjectRequest deleteObjectRequest = new DeleteObjectRequest(this.bucketName, key);
            deleteObjectRequest.setRecursive(true);
            this.callCOSClientWithRetry(deleteObjectRequest);
        }
        catch (Exception e) {
            String errMsg = String.format("Deleting the cos key recursive [%s] occurs an exception: %s", key, e.toString());
            this.handleException(new Exception(errMsg), key);
        }
    }

    @Override
    public void copy(String srcKey, String dstKey) throws IOException {
        try {
            ObjectMetadata objectMetadata = new ObjectMetadata();
            if (this.crc32cEnabled) {
                objectMetadata.setHeader("x-cos-crc32c-flag", (Object)"cosn");
            }
            CopyObjectRequest copyObjectRequest = new CopyObjectRequest(this.bucketName, srcKey, this.bucketName, dstKey);
            FileMetadata sourceFileMetadata = this.retrieveMetadata(srcKey);
            if (null != sourceFileMetadata.getStorageClass()) {
                copyObjectRequest.setStorageClass(sourceFileMetadata.getStorageClass());
            }
            copyObjectRequest.setNewObjectMetadata(objectMetadata);
            this.setEncryptionMetadata(copyObjectRequest, objectMetadata);
            if (null != this.customerDomainEndpointResolver && null != this.customerDomainEndpointResolver.getEndpoint()) {
                copyObjectRequest.setSourceEndpointBuilder((EndpointBuilder)this.customerDomainEndpointResolver);
            }
            this.callCOSClientWithRetry(copyObjectRequest);
        }
        catch (Exception e) {
            String errMsg = String.format("Copy the object failed, src cos key: %s, dst cos key: %s, exception: %s", srcKey, dstKey, e.toString());
            this.handleException(new Exception(errMsg), srcKey);
        }
    }

    @Override
    public void rename(String srcKey, String dstKey) throws IOException {
        if (!this.isMergeBucket) {
            this.normalBucketRename(srcKey, dstKey);
        } else {
            this.mergeBucketRename(srcKey, dstKey);
        }
    }

    public void normalBucketRename(String srcKey, String dstKey) throws IOException {
        LOG.debug("Rename normal bucket key, the source cos key [{}] to the dest cos key [{}].", (Object)srcKey, (Object)dstKey);
        try {
            ObjectMetadata objectMetadata = new ObjectMetadata();
            if (this.crc32cEnabled) {
                objectMetadata.setHeader("x-cos-crc32c-flag", (Object)"cosn");
            }
            CopyObjectRequest copyObjectRequest = new CopyObjectRequest(this.bucketName, srcKey, this.bucketName, dstKey);
            FileMetadata sourceFileMetadata = this.retrieveMetadata(srcKey);
            if (null != sourceFileMetadata.getStorageClass()) {
                copyObjectRequest.setStorageClass(sourceFileMetadata.getStorageClass());
            }
            copyObjectRequest.setNewObjectMetadata(objectMetadata);
            this.setEncryptionMetadata(copyObjectRequest, objectMetadata);
            if (null != this.customerDomainEndpointResolver && null != this.customerDomainEndpointResolver.getEndpoint()) {
                copyObjectRequest.setSourceEndpointBuilder((EndpointBuilder)this.customerDomainEndpointResolver);
            }
            this.callCOSClientWithRetry(copyObjectRequest);
            DeleteObjectRequest deleteObjectRequest = new DeleteObjectRequest(this.bucketName, srcKey);
            this.callCOSClientWithRetry(deleteObjectRequest);
        }
        catch (Exception e) {
            String errMsg = String.format("Rename object failed, normal bucket, source cos key: %s, dest cos key: %s, exception: %s", srcKey, dstKey, e.toString());
            this.handleException(new Exception(errMsg), srcKey);
        }
    }

    public void mergeBucketRename(String srcKey, String dstKey) throws IOException {
        LOG.debug("Rename merge bucket key, the source cos key [{}] to the dest cos key [{}].", (Object)srcKey, (Object)dstKey);
        try {
            RenameRequest renameRequest = new RenameRequest(this.bucketName, srcKey, dstKey);
            this.callCOSClientWithRetry(renameRequest);
        }
        catch (Exception e) {
            String errMsg = String.format("Rename object failed, merge bucket, source cos key: %s, dest cos key: %s, exception: %s", srcKey, dstKey, e.toString());
            this.handleException(new Exception(errMsg), srcKey);
        }
    }

    @Override
    public void purge(String prefix) throws IOException {
        throw new IOException("purge not supported");
    }

    @Override
    public void dump() throws IOException {
        throw new IOException("dump not supported");
    }

    @Override
    public void close() {
        if (null != this.cosClient) {
            this.cosClient.shutdown();
        }
    }

    private void handleException(Exception e, String key) throws IOException {
        String cosPath = "cosn://" + this.bucketName + key;
        String exceptInfo = String.format("%s : %s", cosPath, e.toString());
        throw new IOException(exceptInfo);
    }

    @Override
    public long getFileLength(String key) throws IOException {
        GetObjectMetadataRequest getObjectMetadataRequest = new GetObjectMetadataRequest(this.bucketName, key);
        this.setEncryptionMetadata(getObjectMetadataRequest, new ObjectMetadata());
        try {
            ObjectMetadata objectMetadata = (ObjectMetadata)this.callCOSClientWithRetry(getObjectMetadataRequest);
            return objectMetadata.getContentLength();
        }
        catch (Exception e) {
            String errMsg = String.format("Getting the file length occurs an exception, cos key: %s, exception: %s", key, e.getMessage());
            this.handleException(new Exception(errMsg), key);
            return 0L;
        }
    }

    private <X> void callCOSClientWithSSEKMS(X request, SSECOSKeyManagementParams managementParams) {
        try {
            if (request instanceof PutObjectRequest) {
                ((PutObjectRequest)request).setSSECOSKeyManagementParams(managementParams);
            } else if (request instanceof CopyObjectRequest) {
                ((CopyObjectRequest)request).setSSECOSKeyManagementParams(managementParams);
            } else if (request instanceof InitiateMultipartUploadRequest) {
                ((InitiateMultipartUploadRequest)request).setSSECOSKeyManagementParams(managementParams);
            }
        }
        catch (Exception e) {
            String errMsg = String.format("callCOSClientWithSSEKMS failed: %s", e.toString());
            LOG.error(errMsg);
        }
    }

    private <X> void callCOSClientWithSSECOS(X request, ObjectMetadata objectMetadata) {
        try {
            objectMetadata.setServerSideEncryption(SSEAlgorithm.AES256.getAlgorithm());
            if (request instanceof PutObjectRequest) {
                ((PutObjectRequest)request).setMetadata(objectMetadata);
            } else if (request instanceof UploadPartRequest) {
                ((UploadPartRequest)request).setObjectMetadata(objectMetadata);
            } else if (request instanceof CopyObjectRequest) {
                ((CopyObjectRequest)request).setNewObjectMetadata(objectMetadata);
            } else if (request instanceof InitiateMultipartUploadRequest) {
                ((InitiateMultipartUploadRequest)request).setObjectMetadata(objectMetadata);
            }
        }
        catch (Exception e) {
            String errMsg = String.format("callCOSClientWithSSECOS failed: %s", e.toString());
            LOG.error(errMsg);
        }
    }

    private <X> void callCOSClientWithSSEC(X request, SSECustomerKey sseKey) {
        try {
            if (request instanceof PutObjectRequest) {
                ((PutObjectRequest)request).setSSECustomerKey(sseKey);
            } else if (request instanceof UploadPartRequest) {
                ((UploadPartRequest)request).setSSECustomerKey(sseKey);
            } else if (request instanceof GetObjectMetadataRequest) {
                ((GetObjectMetadataRequest)request).setSSECustomerKey(sseKey);
            } else if (request instanceof CopyObjectRequest) {
                ((CopyObjectRequest)request).setDestinationSSECustomerKey(sseKey);
                ((CopyObjectRequest)request).setSourceSSECustomerKey(sseKey);
            } else if (request instanceof GetObjectRequest) {
                ((GetObjectRequest)request).setSSECustomerKey(sseKey);
            } else if (request instanceof InitiateMultipartUploadRequest) {
                ((InitiateMultipartUploadRequest)request).setSSECustomerKey(sseKey);
            }
        }
        catch (Exception e) {
            String errMsg = String.format("callCOSClientWithSSEC failed: %s", e.toString());
            LOG.error(errMsg);
        }
    }

    private <X> void setEncryptionMetadata(X request, ObjectMetadata objectMetadata) {
        switch (this.encryptionSecrets.getEncryptionMethod()) {
            case SSE_C: {
                this.callCOSClientWithSSEC(request, new SSECustomerKey(this.encryptionSecrets.getEncryptionKey()));
                break;
            }
            case SSE_COS: {
                this.callCOSClientWithSSECOS(request, objectMetadata);
                break;
            }
            case SSE_KMS: {
                String key = this.encryptionSecrets.getEncryptionKey();
                String context = Base64.encodeAsString((byte[])this.encryptionSecrets.getEncryptionContext().getBytes());
                SSECOSKeyManagementParams ssecosKeyManagementParams = new SSECOSKeyManagementParams(key, context);
                this.callCOSClientWithSSEKMS(request, ssecosKeyManagementParams);
                break;
            }
        }
    }

    private void checkEncryptionMethod(ClientConfig config, CosNEncryptionMethods cosSSE, String sseKey) throws IOException {
        int sseKeyLen = StringUtils.isNullOrEmpty((String)sseKey) ? 0 : sseKey.length();
        String description = "Encryption key:";
        if (sseKey == null) {
            description = description + "null ";
        } else {
            switch (sseKeyLen) {
                case 0: {
                    description = description + " empty";
                    break;
                }
                case 1: {
                    description = description + " of length 1";
                    break;
                }
                default: {
                    description = description + " of length " + sseKeyLen + " ending with " + sseKey.charAt(sseKeyLen - 1);
                }
            }
        }
        switch (cosSSE) {
            case SSE_C: {
                LOG.debug("Using SSE_C with {}", (Object)description);
                config.setHttpProtocol(HttpProtocol.https);
                if (sseKeyLen == 0) {
                    throw new IOException("missing encryption key for SSE_C ");
                }
                if (sseKey.matches("^([A-Za-z0-9+/]{4})*([A-Za-z0-9+/]{4}|[A-Za-z0-9+/]{3}=|[A-Za-z0-9+/]{2}==)$")) break;
                throw new IOException("encryption key need to Base64 encoding for SSE_C ");
            }
            case SSE_COS: {
                if (sseKeyLen == 0) break;
                LOG.debug("Using SSE_COS");
                throw new IOException("SSE_COS to encryption with key error:  (" + description + ")");
            }
            case SSE_KMS: {
                System.setProperty("com.Qcloud.services.cos.disablePutObjectMD5Validation", "true");
                config.setHttpProtocol(HttpProtocol.https);
            }
            default: {
                LOG.debug("Data is unencrypted");
            }
        }
    }

    /*
     * Loose catch block
     */
    private <X> Object callCOSClientWithRetry(X request) throws CosServiceException, IOException {
        String sdkMethod = "";
        int retryIndex = 1;
        while (true) {
            try {
                if (request instanceof PutObjectRequest) {
                    sdkMethod = "putObject";
                    if (((PutObjectRequest)request).getInputStream().markSupported()) {
                        ((PutObjectRequest)request).getInputStream().mark((int)((PutObjectRequest)request).getMetadata().getContentLength());
                    }
                    return this.cosClient.putObject((PutObjectRequest)request);
                }
                if (request instanceof UploadPartRequest) {
                    sdkMethod = "uploadPart";
                    if (((UploadPartRequest)request).getInputStream().markSupported()) {
                        ((UploadPartRequest)request).getInputStream().mark((int)((UploadPartRequest)request).getPartSize());
                    }
                    return this.cosClient.uploadPart((UploadPartRequest)request);
                }
                if (request instanceof HeadBucketRequest) {
                    sdkMethod = "headBucket";
                    return this.cosClient.headBucket((HeadBucketRequest)request);
                }
                if (request instanceof RenameRequest) {
                    sdkMethod = "rename";
                    this.cosClient.rename((RenameRequest)request);
                    return new Object();
                }
                if (request instanceof GetObjectMetadataRequest) {
                    sdkMethod = "queryObjectMeta";
                    return this.cosClient.getObjectMetadata((GetObjectMetadataRequest)request);
                }
                if (request instanceof DeleteObjectRequest) {
                    sdkMethod = "deleteObject";
                    this.cosClient.deleteObject((DeleteObjectRequest)request);
                    return new Object();
                }
                if (request instanceof CopyObjectRequest) {
                    sdkMethod = "copyFile";
                    return this.cosClient.copyObject((CopyObjectRequest)request);
                }
                if (request instanceof GetObjectRequest) {
                    sdkMethod = "getObject";
                    return this.cosClient.getObject((GetObjectRequest)request);
                }
                if (request instanceof ListObjectsRequest) {
                    sdkMethod = "listObjects";
                    return this.cosClient.listObjects((ListObjectsRequest)request);
                }
                if (request instanceof InitiateMultipartUploadRequest) {
                    sdkMethod = "initiateMultipartUpload";
                    return this.cosClient.initiateMultipartUpload((InitiateMultipartUploadRequest)request);
                }
                if (request instanceof CompleteMultipartUploadRequest) {
                    sdkMethod = "completeMultipartUpload";
                    return this.cosClient.completeMultipartUpload((CompleteMultipartUploadRequest)request);
                }
                if (request instanceof AbortMultipartUploadRequest) {
                    sdkMethod = "abortMultipartUpload";
                    this.cosClient.abortMultipartUpload((AbortMultipartUploadRequest)request);
                    return new Object();
                }
                throw new IOException("no such method");
            }
            catch (ResponseNotCompleteException nce) {
                if (this.completeMPUCheckEnabled && request instanceof CompleteMultipartUploadRequest) {
                    String key = ((CompleteMultipartUploadRequest)request).getKey();
                    FileMetadata fileMetadata = this.queryObjectMetadata(key);
                    if (null == fileMetadata) {
                        this.handleException((Exception)((Object)nce), key);
                    }
                    LOG.warn("Complete mpu resp not complete key [{}]", (Object)key);
                    return new CompleteMultipartUploadResult();
                }
                throw new IOException(nce);
            }
            catch (CosServiceException cse) {
                String errMsg = String.format("all cos sdk failed, retryIndex: [%d / %d], call method: %s, exception: %s", retryIndex, this.maxRetryTimes, sdkMethod, cse.toString());
                int statusCode = cse.getStatusCode();
                String errorCode = cse.getErrorCode();
                LOG.debug("fail to retry statusCode {}, errorCode {}", (Object)statusCode, (Object)errorCode);
                if (request instanceof CopyObjectRequest && statusCode / 100 == 2 && errorCode != null && !errorCode.isEmpty()) {
                    if (retryIndex <= this.maxRetryTimes) {
                        LOG.info(errMsg, (Throwable)cse);
                        ++retryIndex;
                        continue;
                    }
                    LOG.error(errMsg, (Throwable)cse);
                    throw new IOException(errMsg);
                }
                if (statusCode / 100 == 5) {
                    if (retryIndex <= this.maxRetryTimes) {
                        LOG.info(errMsg, (Throwable)cse);
                        long sleepLeast = (long)retryIndex * 300L;
                        long sleepBound = (long)retryIndex * 500L;
                        try {
                            if (request instanceof PutObjectRequest) {
                                LOG.info("Try to reset the put object request input stream.");
                                if (((PutObjectRequest)request).getInputStream().markSupported()) {
                                    ((PutObjectRequest)request).getInputStream().reset();
                                } else {
                                    LOG.error("The put object request input stream can not be reset, so it can not be retried.");
                                    throw cse;
                                }
                            }
                            if (request instanceof UploadPartRequest) {
                                LOG.info("Try to reset the upload part request input stream.");
                                if (((UploadPartRequest)request).getInputStream().markSupported()) {
                                    ((UploadPartRequest)request).getInputStream().reset();
                                } else {
                                    LOG.error("The upload part request input stream can not be reset, so it can not be retried.");
                                    throw cse;
                                }
                            }
                            Thread.sleep(ThreadLocalRandom.current().nextLong(sleepLeast, sleepBound));
                            ++retryIndex;
                        }
                        catch (InterruptedException e) {
                            throw new IOException(e.toString());
                        }
                        continue;
                    }
                    LOG.error(errMsg, (Throwable)cse);
                    throw new IOException(errMsg);
                }
                throw cse;
            }
            break;
        }
        catch (Exception e) {
            throw new IOException(e);
        }
    }

    private static String ensureValidAttributeName(String attributeName) {
        return attributeName.replace('.', '-').toLowerCase();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String getPluginVersionInfo() {
        String versionStr;
        InputStream inputStream;
        block5: {
            Properties versionProperties = new Properties();
            inputStream = null;
            versionStr = "unknown";
            String versionFile = "hadoopCosPluginVersionInfo.properties";
            try {
                inputStream = this.getClass().getClassLoader().getResourceAsStream("hadoopCosPluginVersionInfo.properties");
                if (inputStream != null) {
                    versionProperties.load(inputStream);
                    versionStr = versionProperties.getProperty("plugin_version");
                    break block5;
                }
                LOG.error("load versionInfo properties failed, propName: {} ", (Object)"hadoopCosPluginVersionInfo.properties");
            }
            catch (IOException e) {
                try {
                    LOG.error("load versionInfo properties exception, propName: {} ", (Object)"hadoopCosPluginVersionInfo.properties");
                }
                catch (Throwable throwable) {
                    IOUtils.closeQuietly(inputStream, (Logger)LOG);
                    throw throwable;
                }
                IOUtils.closeQuietly((Closeable)inputStream, (Logger)LOG);
            }
        }
        IOUtils.closeQuietly((Closeable)inputStream, (Logger)LOG);
        return versionStr;
    }
}

