/*
 * Decompiled with CFR 0.152.
 */
package com.jxdinfo.hussar.migration.utils;

import com.jxdinfo.hussar.common.utils.IdempotentJsonUtils;
import com.jxdinfo.hussar.migration.constants.MigrationExceptionEnum;
import com.jxdinfo.hussar.migration.enums.EncryptionType;
import com.jxdinfo.hussar.migration.enums.SignatureType;
import com.jxdinfo.hussar.migration.exceptions.MigrationArchiveException;
import com.jxdinfo.hussar.migration.exceptions.MigrationCheckException;
import com.jxdinfo.hussar.migration.exceptions.MigrationCryptoException;
import com.jxdinfo.hussar.migration.exceptions.MigrationException;
import com.jxdinfo.hussar.migration.manifest.MigrationArchiveItem;
import com.jxdinfo.hussar.migration.manifest.MigrationArchiveManifest;
import com.jxdinfo.hussar.migration.utils.MigrationArchiveOptions;
import com.jxdinfo.hussar.migration.utils.MigrationCryptoUtils;
import com.jxdinfo.hussar.migration.utils.MigrationUnarchiveOptions;
import com.jxdinfo.hussar.platform.core.utils.id.UUID;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorOutputStream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;
import org.apache.commons.io.input.BoundedInputStream;
import org.apache.commons.io.input.CloseShieldInputStream;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class MigrationArchiveUtils {
    private static final Logger logger = LoggerFactory.getLogger(MigrationArchiveUtils.class);
    public static final String MANIFEST_FILE = "manifest.json";
    public static final String MANIFEST_CHECKSUM_FILE = "manifest.sum";
    public static final String MANIFEST_SIGNATURE_FILE = "manifest.sig";
    public static final String PAYLOAD_FILE = "payload.bin";
    private static final String TEMP_PAYLOAD_PREFIX = "hussar-payload-";
    private static final String TEMP_ENCRYPT_PREFIX = "hussar-encrypt-";
    private static final String TEMP_MANIFEST_PREFIX = "hussar-manifest-";
    private static final String TEMP_DECRYPT_PREFIX = "hussar-decrypt-";
    private static final String TEMP_SUFFIX = ".tmp";
    private static final byte[] GZIP_MAGIC_NUMBER = new byte[]{31, -117};

    private MigrationArchiveUtils() {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void archive(File archive, File directory, MigrationArchiveOptions options) throws MigrationArchiveException {
        byte[] manifestSignature;
        byte[] manifestChecksum;
        byte[] manifestData;
        File originPayload;
        MigrationArchiveManifest manifest;
        if (archive == null || directory == null) {
            throw new NullPointerException();
        }
        if (options == null) {
            options = new MigrationArchiveOptions();
        }
        try {
            String manifestData2 = FileUtils.readFileToString((File)new File(directory, MANIFEST_FILE), (Charset)StandardCharsets.UTF_8);
            manifest = (MigrationArchiveManifest)IdempotentJsonUtils.parse((String)manifestData2, MigrationArchiveManifest.class);
            if (manifest == null) {
                throw new MigrationArchiveException("failed to parse manifest");
            }
        }
        catch (MigrationException ex) {
            throw ex;
        }
        catch (Exception ex) {
            throw new MigrationArchiveException("failed to read manifest", (Throwable)ex);
        }
        if (manifest.getUuid() == null || options.isResetUuid()) {
            manifest.setUuid(UUID.randomUUID().toString(true));
        }
        if (manifest.getVersion() == 0L) {
            manifest.setVersion(1L);
        } else if (!Objects.equals(manifest.getVersion(), 1L)) {
            throw new MigrationArchiveException("incompatible manifest version");
        }
        try {
            originPayload = Files.createTempFile(TEMP_PAYLOAD_PREFIX, TEMP_SUFFIX, new FileAttribute[0]).toFile();
            HashSet<String> uuids = new HashSet<String>(MigrationArchiveUtils.getUuidList(manifest));
            MigrationArchiveUtils.buildTarball(directory, originPayload, true, name -> {
                int index = name.indexOf(47);
                return index >= 0 && uuids.contains(name.substring(0, index));
            });
        }
        catch (IOException ex) {
            throw new MigrationArchiveException((Throwable)ex);
        }
        File encryptedPayload = null;
        try {
            if (options.getEncryptionType() != null) {
                encryptedPayload = Files.createTempFile(TEMP_ENCRYPT_PREFIX, TEMP_SUFFIX, new FileAttribute[0]).toFile();
                MigrationCryptoUtils.encrypt(options.getEncryptionType(), options.getEncryptionKey(), originPayload, encryptedPayload);
            } else {
                encryptedPayload = originPayload;
            }
        }
        catch (IOException ex) {
            throw new MigrationArchiveException((Throwable)ex);
        }
        finally {
            if (!originPayload.equals(encryptedPayload)) {
                try {
                    FileUtils.forceDelete((File)originPayload);
                }
                catch (IOException ex) {
                    logger.warn("failed to remove temporary payload", (Throwable)ex);
                }
            }
        }
        manifest.setEncryption(Optional.ofNullable(options.getEncryptionType()).map(EncryptionType::toString).orElse(null));
        manifest.setChecksum(Optional.ofNullable(MigrationCryptoUtils.computeChecksum(options.getChecksumType(), encryptedPayload)).map(MigrationCryptoUtils.ChecksumTypedValue::toString).orElse(null));
        manifest.setSignature(Optional.ofNullable(MigrationCryptoUtils.computeSignature(options.getSignatureType(), options.getSignaturePrivateKey(), encryptedPayload)).map(MigrationCryptoUtils.SignatureTypedValue::toString).orElse(null));
        File manifestFile = null;
        try {
            manifestData = IdempotentJsonUtils.toByteArray((Object)manifest);
            manifestFile = Files.createTempFile(TEMP_MANIFEST_PREFIX, TEMP_SUFFIX, new FileAttribute[0]).toFile();
            FileUtils.writeByteArrayToFile((File)manifestFile, (byte[])manifestData);
            manifestChecksum = Optional.ofNullable(MigrationCryptoUtils.computeChecksum(options.getChecksumType(), manifestFile)).map(MigrationCryptoUtils.ChecksumTypedValue::toString).map(str -> str.getBytes(StandardCharsets.UTF_8)).orElse(null);
            manifestSignature = Optional.ofNullable(MigrationCryptoUtils.computeSignature(options.getSignatureType(), options.getSignaturePrivateKey(), manifestFile)).map(MigrationCryptoUtils.SignatureTypedValue::toString).map(str -> str.getBytes(StandardCharsets.UTF_8)).orElse(null);
        }
        catch (IOException ex) {
            throw new MigrationArchiveException((Throwable)ex);
        }
        finally {
            if (manifestFile != null && manifestFile.exists()) {
                try {
                    FileUtils.forceDelete((File)manifestFile);
                }
                catch (IOException ex) {
                    logger.warn("failed to remove temporary manifest", (Throwable)ex);
                }
            }
        }
        try {
            File payload = encryptedPayload;
            MigrationArchiveUtils.buildTarball(archive, false, archiveTarball -> {
                try {
                    TarArchiveEntry manifestEntry = new TarArchiveEntry(MANIFEST_FILE);
                    manifestEntry.setSize((long)manifestData.length);
                    archiveTarball.putArchiveEntry((ArchiveEntry)manifestEntry);
                    IOUtils.write((byte[])manifestData, (OutputStream)archiveTarball);
                    archiveTarball.closeArchiveEntry();
                    if (manifestChecksum != null) {
                        TarArchiveEntry manifestChecksumEntry = new TarArchiveEntry(MANIFEST_CHECKSUM_FILE);
                        manifestChecksumEntry.setSize((long)manifestChecksum.length);
                        archiveTarball.putArchiveEntry((ArchiveEntry)manifestChecksumEntry);
                        IOUtils.write((byte[])manifestChecksum, (OutputStream)archiveTarball);
                        archiveTarball.closeArchiveEntry();
                    }
                    if (manifestSignature != null) {
                        TarArchiveEntry manifestSignatureEntry = new TarArchiveEntry(MANIFEST_SIGNATURE_FILE);
                        manifestSignatureEntry.setSize((long)manifestSignature.length);
                        archiveTarball.putArchiveEntry((ArchiveEntry)manifestSignatureEntry);
                        IOUtils.write((byte[])manifestSignature, (OutputStream)archiveTarball);
                        archiveTarball.closeArchiveEntry();
                    }
                    TarArchiveEntry payloadEntry = new TarArchiveEntry(PAYLOAD_FILE);
                    payloadEntry.setSize(FileUtils.sizeOf((File)payload));
                    archiveTarball.putArchiveEntry((ArchiveEntry)payloadEntry);
                    FileUtils.copyFile((File)payload, (OutputStream)archiveTarball);
                    archiveTarball.closeArchiveEntry();
                }
                catch (IOException ex) {
                    throw new MigrationArchiveException((Throwable)ex);
                }
            });
        }
        finally {
            try {
                FileUtils.forceDelete((File)encryptedPayload);
            }
            catch (IOException ex) {
                logger.warn("failed to remove temporary payload", (Throwable)ex);
            }
        }
    }

    public static void unarchive(File archive, File directory, MigrationUnarchiveOptions options) throws MigrationArchiveException {
        MigrationArchiveManifest manifest;
        File unarchiveDirectory;
        if (archive == null || directory == null) {
            throw new NullPointerException();
        }
        if (options == null) {
            options = new MigrationUnarchiveOptions();
        }
        try {
            unarchiveDirectory = Files.createTempDirectory("hussar-unarchive-", new FileAttribute[0]).toFile();
            MigrationArchiveUtils.extractTarball(archive, false, unarchiveDirectory, null);
        }
        catch (IOException ex) {
            throw new MigrationArchiveException((Throwable)ex);
        }
        try {
            File manifestFile = new File(unarchiveDirectory, MANIFEST_FILE);
            String manifestData = FileUtils.readFileToString((File)manifestFile, (Charset)StandardCharsets.UTF_8);
            manifest = (MigrationArchiveManifest)IdempotentJsonUtils.parse((String)manifestData, MigrationArchiveManifest.class);
            if (manifest == null) {
                throw new MigrationArchiveException("failed to parse manifest");
            }
            File manifestChecksumFile = new File(unarchiveDirectory, MANIFEST_CHECKSUM_FILE);
            if (options.isRequireChecksum() && !manifestChecksumFile.isFile()) {
                throw new MigrationCheckException("manifest checksum is required");
            }
            if (options.isVerifyChecksum() && manifestChecksumFile.isFile()) {
                String manifestChecksum = FileUtils.readFileToString((File)manifestChecksumFile, (Charset)StandardCharsets.UTF_8).trim();
                MigrationCryptoUtils.ChecksumTypedValue parsedManifestChecksum = MigrationCryptoUtils.ChecksumTypedValue.parse(manifestChecksum);
                if (parsedManifestChecksum != null) {
                    if (!MigrationCryptoUtils.verifyChecksum(parsedManifestChecksum, manifestFile)) {
                        throw new MigrationCheckException("failed to verify manifest checksum: " + manifestChecksum);
                    }
                } else if (options.isRequireChecksum()) {
                    throw new MigrationCheckException("manifest checksum is required");
                }
            }
            File manifestSignatureFile = new File(unarchiveDirectory, MANIFEST_SIGNATURE_FILE);
            if (options.isRequireSignature() && !manifestSignatureFile.isFile()) {
                throw new MigrationCheckException("manifest signature is required");
            }
            if (options.isVerifySignature() && manifestSignatureFile.isFile()) {
                String manifestSignature = FileUtils.readFileToString((File)manifestSignatureFile, (Charset)StandardCharsets.UTF_8).trim();
                MigrationCryptoUtils.SignatureTypedValue parsedManifestSignature = MigrationCryptoUtils.SignatureTypedValue.parse(manifestSignature);
                if (parsedManifestSignature != null) {
                    Set<String> publicKeys = Optional.ofNullable(options.getSignaturePublicKeys()).map(signaturePublicKeys -> MigrationArchiveUtils.lookupSignaturePublicKeys(signaturePublicKeys, parsedManifestSignature.getType())).orElse(Collections.emptySet());
                    if (!MigrationCryptoUtils.verifySignature(parsedManifestSignature, publicKeys, manifestFile)) {
                        throw new MigrationCheckException("failed to verify manifest signature: " + manifest.getSignature());
                    }
                } else if (options.isRequireSignature()) {
                    throw new MigrationCheckException("manifest signature is required");
                }
            }
        }
        catch (MigrationException ex) {
            throw new MigrationCheckException(MigrationExceptionEnum.CHECK_EXCEPTION.getMessage());
        }
        catch (Exception ex) {
            throw new MigrationCheckException("failed to parse manifest or its checksum/signature", (Throwable)ex);
        }
        if (options.isVerifyVersion() && manifest.getVersion() != 1L) {
            throw new MigrationCheckException("archive manifest version not supported");
        }
        File encryptedPayload = new File(unarchiveDirectory, PAYLOAD_FILE);
        if (options.isVerifyChecksum()) {
            MigrationCryptoUtils.ChecksumTypedValue parsedChecksum = MigrationCryptoUtils.ChecksumTypedValue.parse(manifest.getChecksum());
            if (parsedChecksum != null) {
                if (!MigrationCryptoUtils.verifyChecksum(parsedChecksum, encryptedPayload)) {
                    throw new MigrationCheckException("failed to verify payload checksum: " + manifest.getChecksum());
                }
            } else if (options.isRequireChecksum()) {
                throw new MigrationCheckException("payload checksum is required");
            }
        }
        if (options.isVerifySignature()) {
            MigrationCryptoUtils.SignatureTypedValue parsedSignature = MigrationCryptoUtils.SignatureTypedValue.parse(manifest.getSignature());
            if (parsedSignature != null) {
                Set<String> publicKeys = Optional.ofNullable(options.getSignaturePublicKeys()).map(signaturePublicKeys -> MigrationArchiveUtils.lookupSignaturePublicKeys(signaturePublicKeys, parsedSignature.getType())).orElse(Collections.emptySet());
                if (!MigrationCryptoUtils.verifySignature(parsedSignature, publicKeys, encryptedPayload)) {
                    throw new MigrationCheckException("failed to verify payload signature: " + manifest.getSignature());
                }
            } else if (options.isRequireSignature()) {
                throw new MigrationCheckException("payload signature is required");
            }
        }
        File originPayload = null;
        try {
            String decryptionKey;
            EncryptionType encryptionType = MigrationArchiveUtils.parseEncryptionType(manifest.getEncryption());
            String string = decryptionKey = encryptionType != null && options.getDecryptionKeys() != null ? MigrationArchiveUtils.lookupDecryptionKey(options.getDecryptionKeys(), encryptionType) : null;
            if (encryptedPayload.exists()) {
                if (encryptionType != null) {
                    originPayload = Files.createTempFile(TEMP_DECRYPT_PREFIX, TEMP_SUFFIX, new FileAttribute[0]).toFile();
                    MigrationCryptoUtils.decrypt(encryptionType, decryptionKey, encryptedPayload, originPayload);
                    MigrationArchiveUtils.assertPayloadCorrectlyDecrypted(originPayload);
                } else {
                    originPayload = encryptedPayload;
                }
            }
        }
        catch (IOException ex) {
            throw new MigrationArchiveException((Throwable)ex);
        }
        finally {
            if (encryptedPayload.exists() && !encryptedPayload.equals(originPayload)) {
                try {
                    FileUtils.forceDelete((File)encryptedPayload);
                }
                catch (IOException ex) {
                    logger.warn("failed to remove temporary payload", (Throwable)ex);
                }
            }
        }
        try {
            FileUtils.copyFile((File)new File(unarchiveDirectory, MANIFEST_FILE), (File)new File(directory, MANIFEST_FILE));
            if (originPayload != null && originPayload.exists()) {
                MigrationArchiveUtils.extractTarball(originPayload, true, directory, null);
            }
            for (String uuid : MigrationArchiveUtils.getUuidList(manifest)) {
                FileUtils.forceMkdirParent((File)new File(directory, uuid));
            }
        }
        catch (IOException ex) {
            throw new MigrationArchiveException((Throwable)ex);
        }
        finally {
            try {
                FileUtils.forceDelete((File)unarchiveDirectory);
            }
            catch (IOException ex) {
                logger.warn("failed to remove temporary unarchive directory: {}", (Object)unarchiveDirectory, (Object)ex);
            }
            if (originPayload != null && originPayload.exists()) {
                try {
                    FileUtils.forceDelete((File)originPayload);
                }
                catch (IOException ex) {
                    logger.warn("failed to remove temporary decrypted payload: {}", (Object)originPayload, (Object)ex);
                }
            }
        }
    }

    private static void assertPayloadCorrectlyDecrypted(File payload) {
        try (InputStream input = Files.newInputStream(payload.toPath(), new OpenOption[0]);){
            byte[] header = IOUtils.toByteArray((InputStream)new BoundedInputStream(input, 2L));
            if (!Arrays.equals(header, GZIP_MAGIC_NUMBER)) {
                throw new MigrationCryptoException("archive payload is incorrectly decrypted");
            }
        }
        catch (IOException ex) {
            throw new MigrationArchiveException((Throwable)ex);
        }
    }

    private static String lookupDecryptionKey(Map<String, String> decryptionKeys, EncryptionType type) {
        if (type == null) {
            return null;
        }
        switch (type) {
            case AES: {
                return Optional.ofNullable(decryptionKeys).map(keys -> {
                    String key = (String)keys.get(type.name());
                    if (key == null) {
                        key = (String)keys.get(type.name().toLowerCase());
                    }
                    return key;
                }).orElse(null);
            }
        }
        throw new MigrationArchiveException("archive encryption algorithm " + type + " not implemented yet");
    }

    private static Set<String> lookupSignaturePublicKeys(Map<String, Set<String>> signaturePublicKeys, SignatureType type) {
        if (type == null) {
            return null;
        }
        switch (type) {
            case RSA: {
                return Optional.ofNullable(signaturePublicKeys).map(keys -> {
                    Set key = (Set)keys.get(type.name());
                    if (key == null) {
                        key = (Set)keys.get(type.name().toLowerCase());
                    }
                    return key;
                }).orElse(null);
            }
        }
        throw new MigrationArchiveException("archive encryption algorithm " + type + " not implemented yet");
    }

    private static EncryptionType parseEncryptionType(String literal) {
        if (StringUtils.isNoneEmpty((CharSequence[])new CharSequence[]{literal})) {
            try {
                return EncryptionType.valueOf((String)literal.toUpperCase());
            }
            catch (IllegalArgumentException ex) {
                throw new MigrationArchiveException("unsupported EncryptionType value: " + literal, (Throwable)ex);
            }
        }
        return null;
    }

    private static List<String> getUuidList(MigrationArchiveManifest manifest) {
        if (manifest == null) {
            return Collections.emptyList();
        }
        return Optional.ofNullable(manifest.getItems()).orElse(Collections.emptyList()).stream().map(MigrationArchiveItem::getUuid).filter(xva$0 -> StringUtils.isNoneEmpty((CharSequence[])new CharSequence[]{xva$0})).collect(Collectors.toList());
    }

    private static void buildTarball(File tarball, boolean gzip, Consumer<TarArchiveOutputStream> callback) {
        try (OutputStream output = Files.newOutputStream(tarball.toPath(), new OpenOption[0]);
             TarArchiveOutputStream archive = new TarArchiveOutputStream((OutputStream)(gzip ? new GzipCompressorOutputStream(output) : output), StandardCharsets.UTF_8.name());){
            archive.setLongFileMode(2);
            archive.setBigNumberMode(1);
            callback.accept(archive);
        }
        catch (IOException ex) {
            throw new MigrationArchiveException((Throwable)ex);
        }
    }

    private static void buildTarball(File directory, File tarball, boolean gzip, Predicate<String> filter) {
        MigrationArchiveUtils.buildTarball(tarball, gzip, archive -> {
            try {
                for (TarEntryMapping mapping : TarEntryMapping.list(directory)) {
                    String name = mapping.getName();
                    File file = mapping.getFile();
                    if (filter != null && !filter.test(name)) continue;
                    TarArchiveEntry entry = new TarArchiveEntry(name);
                    if (!file.isDirectory()) {
                        entry.setSize(FileUtils.sizeOf((File)file));
                        archive.putArchiveEntry((ArchiveEntry)entry);
                        FileUtils.copyFile((File)file, (OutputStream)archive);
                    } else {
                        archive.putArchiveEntry((ArchiveEntry)entry);
                    }
                    archive.closeArchiveEntry();
                }
            }
            catch (IOException ex) {
                throw new MigrationArchiveException((Throwable)ex);
            }
        });
    }

    private static void extractTarball(File tarball, boolean gzip, File directory, Predicate<String> filter) {
        try (InputStream input = Files.newInputStream(tarball.toPath(), new OpenOption[0]);
             TarArchiveInputStream archive = new TarArchiveInputStream((InputStream)(gzip ? new GzipCompressorInputStream(input) : input), StandardCharsets.UTF_8.name());){
            TarArchiveEntry entry;
            while ((entry = archive.getNextTarEntry()) != null) {
                String name = entry.getName();
                if (filter != null && !filter.test(name)) continue;
                File destination = new File(directory, FilenameUtils.normalize((String)name));
                if (entry.isDirectory()) {
                    FileUtils.forceMkdirParent((File)destination);
                    continue;
                }
                FileUtils.copyInputStreamToFile((InputStream)new CloseShieldInputStream((InputStream)archive), (File)destination);
            }
        }
        catch (IOException ex) {
            throw new MigrationArchiveException((Throwable)ex);
        }
    }

    private static final class TarEntryMapping {
        private final String name;
        private final File file;

        private TarEntryMapping(String name, File file) {
            this.name = name;
            this.file = file;
        }

        public static TarEntryMapping of(String name, File file) {
            if (name == null || file == null) {
                throw new NullPointerException();
            }
            name = FilenameUtils.normalize((String)name, (boolean)true);
            if (file.isDirectory() && !StringUtils.endsWith((CharSequence)name, (CharSequence)"/")) {
                name = name + "/";
            }
            return new TarEntryMapping(name, file);
        }

        public static TarEntryMapping from(File root, File file) {
            String prefix;
            String name = file.toString();
            if (!name.startsWith(prefix = root.toString()) || name.length() <= prefix.length() || name.charAt(prefix.length()) != '/' && name.charAt(prefix.length()) != '\\') {
                throw new IllegalArgumentException("file " + file + " is not based on root " + root);
            }
            return TarEntryMapping.of(name.substring(prefix.length() + 1), file);
        }

        public static List<TarEntryMapping> list(File root) {
            return FileUtils.listFilesAndDirs((File)root, (IOFileFilter)TrueFileFilter.INSTANCE, (IOFileFilter)TrueFileFilter.INSTANCE).stream().filter(file -> !Objects.equals(root, file)).map(file -> TarEntryMapping.from(root, file)).collect(Collectors.toList());
        }

        public String getName() {
            return this.name;
        }

        public File getFile() {
            return this.file;
        }
    }
}

