/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.waf.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is <b>AWS WAF Classic</b> documentation. For more information, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/classic-waf-chapter.html">AWS WAF Classic</a> in the
 * developer guide.
 * </p>
 * <p>
 * <b>For the latest version of AWS WAF</b>, use the AWS WAFV2 API and see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>. With the
 * latest version, AWS WAF has a single set of endpoints for regional and global use.
 * </p>
 * </note>
 * <p>
 * The <code>RegexPatternSet</code> specifies the regular expression (regex) pattern that you want AWS WAF to search
 * for, such as <code>B[a@]dB[o0]t</code>. You can then configure AWS WAF to reject those requests.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RegexPatternSet implements SdkPojo, Serializable, ToCopyableBuilder<RegexPatternSet.Builder, RegexPatternSet> {
    private static final SdkField<String> REGEX_PATTERN_SET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RegexPatternSetId").getter(getter(RegexPatternSet::regexPatternSetId))
            .setter(setter(Builder::regexPatternSetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RegexPatternSetId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(RegexPatternSet::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> REGEX_PATTERN_STRINGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RegexPatternStrings")
            .getter(getter(RegexPatternSet::regexPatternStrings))
            .setter(setter(Builder::regexPatternStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RegexPatternStrings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REGEX_PATTERN_SET_ID_FIELD,
            NAME_FIELD, REGEX_PATTERN_STRINGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String regexPatternSetId;

    private final String name;

    private final List<String> regexPatternStrings;

    private RegexPatternSet(BuilderImpl builder) {
        this.regexPatternSetId = builder.regexPatternSetId;
        this.name = builder.name;
        this.regexPatternStrings = builder.regexPatternStrings;
    }

    /**
     * <p>
     * The identifier for the <code>RegexPatternSet</code>. You use <code>RegexPatternSetId</code> to get information
     * about a <code>RegexPatternSet</code>, update a <code>RegexPatternSet</code>, remove a
     * <code>RegexPatternSet</code> from a <code>RegexMatchSet</code>, and delete a <code>RegexPatternSet</code> from
     * AWS WAF.
     * </p>
     * <p>
     * <code>RegexMatchSetId</code> is returned by <a>CreateRegexPatternSet</a> and by <a>ListRegexPatternSets</a>.
     * </p>
     * 
     * @return The identifier for the <code>RegexPatternSet</code>. You use <code>RegexPatternSetId</code> to get
     *         information about a <code>RegexPatternSet</code>, update a <code>RegexPatternSet</code>, remove a
     *         <code>RegexPatternSet</code> from a <code>RegexMatchSet</code>, and delete a <code>RegexPatternSet</code>
     *         from AWS WAF.</p>
     *         <p>
     *         <code>RegexMatchSetId</code> is returned by <a>CreateRegexPatternSet</a> and by
     *         <a>ListRegexPatternSets</a>.
     */
    public final String regexPatternSetId() {
        return regexPatternSetId;
    }

    /**
     * <p>
     * A friendly name or description of the <a>RegexPatternSet</a>. You can't change <code>Name</code> after you create
     * a <code>RegexPatternSet</code>.
     * </p>
     * 
     * @return A friendly name or description of the <a>RegexPatternSet</a>. You can't change <code>Name</code> after
     *         you create a <code>RegexPatternSet</code>.
     */
    public final String name() {
        return name;
    }

    /**
     * Returns true if the RegexPatternStrings property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasRegexPatternStrings() {
        return regexPatternStrings != null && !(regexPatternStrings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the regular expression (regex) patterns that you want AWS WAF to search for, such as
     * <code>B[a@]dB[o0]t</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRegexPatternStrings()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the regular expression (regex) patterns that you want AWS WAF to search for, such as
     *         <code>B[a@]dB[o0]t</code>.
     */
    public final List<String> regexPatternStrings() {
        return regexPatternStrings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(regexPatternSetId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasRegexPatternStrings() ? regexPatternStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RegexPatternSet)) {
            return false;
        }
        RegexPatternSet other = (RegexPatternSet) obj;
        return Objects.equals(regexPatternSetId(), other.regexPatternSetId()) && Objects.equals(name(), other.name())
                && hasRegexPatternStrings() == other.hasRegexPatternStrings()
                && Objects.equals(regexPatternStrings(), other.regexPatternStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RegexPatternSet").add("RegexPatternSetId", regexPatternSetId()).add("Name", name())
                .add("RegexPatternStrings", hasRegexPatternStrings() ? regexPatternStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RegexPatternSetId":
            return Optional.ofNullable(clazz.cast(regexPatternSetId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "RegexPatternStrings":
            return Optional.ofNullable(clazz.cast(regexPatternStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RegexPatternSet, T> g) {
        return obj -> g.apply((RegexPatternSet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RegexPatternSet> {
        /**
         * <p>
         * The identifier for the <code>RegexPatternSet</code>. You use <code>RegexPatternSetId</code> to get
         * information about a <code>RegexPatternSet</code>, update a <code>RegexPatternSet</code>, remove a
         * <code>RegexPatternSet</code> from a <code>RegexMatchSet</code>, and delete a <code>RegexPatternSet</code>
         * from AWS WAF.
         * </p>
         * <p>
         * <code>RegexMatchSetId</code> is returned by <a>CreateRegexPatternSet</a> and by <a>ListRegexPatternSets</a>.
         * </p>
         * 
         * @param regexPatternSetId
         *        The identifier for the <code>RegexPatternSet</code>. You use <code>RegexPatternSetId</code> to get
         *        information about a <code>RegexPatternSet</code>, update a <code>RegexPatternSet</code>, remove a
         *        <code>RegexPatternSet</code> from a <code>RegexMatchSet</code>, and delete a
         *        <code>RegexPatternSet</code> from AWS WAF.</p>
         *        <p>
         *        <code>RegexMatchSetId</code> is returned by <a>CreateRegexPatternSet</a> and by
         *        <a>ListRegexPatternSets</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regexPatternSetId(String regexPatternSetId);

        /**
         * <p>
         * A friendly name or description of the <a>RegexPatternSet</a>. You can't change <code>Name</code> after you
         * create a <code>RegexPatternSet</code>.
         * </p>
         * 
         * @param name
         *        A friendly name or description of the <a>RegexPatternSet</a>. You can't change <code>Name</code> after
         *        you create a <code>RegexPatternSet</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Specifies the regular expression (regex) patterns that you want AWS WAF to search for, such as
         * <code>B[a@]dB[o0]t</code>.
         * </p>
         * 
         * @param regexPatternStrings
         *        Specifies the regular expression (regex) patterns that you want AWS WAF to search for, such as
         *        <code>B[a@]dB[o0]t</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regexPatternStrings(Collection<String> regexPatternStrings);

        /**
         * <p>
         * Specifies the regular expression (regex) patterns that you want AWS WAF to search for, such as
         * <code>B[a@]dB[o0]t</code>.
         * </p>
         * 
         * @param regexPatternStrings
         *        Specifies the regular expression (regex) patterns that you want AWS WAF to search for, such as
         *        <code>B[a@]dB[o0]t</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regexPatternStrings(String... regexPatternStrings);
    }

    static final class BuilderImpl implements Builder {
        private String regexPatternSetId;

        private String name;

        private List<String> regexPatternStrings = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RegexPatternSet model) {
            regexPatternSetId(model.regexPatternSetId);
            name(model.name);
            regexPatternStrings(model.regexPatternStrings);
        }

        public final String getRegexPatternSetId() {
            return regexPatternSetId;
        }

        @Override
        public final Builder regexPatternSetId(String regexPatternSetId) {
            this.regexPatternSetId = regexPatternSetId;
            return this;
        }

        public final void setRegexPatternSetId(String regexPatternSetId) {
            this.regexPatternSetId = regexPatternSetId;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Collection<String> getRegexPatternStrings() {
            if (regexPatternStrings instanceof SdkAutoConstructList) {
                return null;
            }
            return regexPatternStrings;
        }

        @Override
        public final Builder regexPatternStrings(Collection<String> regexPatternStrings) {
            this.regexPatternStrings = RegexPatternStringsCopier.copy(regexPatternStrings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder regexPatternStrings(String... regexPatternStrings) {
            regexPatternStrings(Arrays.asList(regexPatternStrings));
            return this;
        }

        public final void setRegexPatternStrings(Collection<String> regexPatternStrings) {
            this.regexPatternStrings = RegexPatternStringsCopier.copy(regexPatternStrings);
        }

        @Override
        public RegexPatternSet build() {
            return new RegexPatternSet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
