/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpoint.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The settings for a custom message activity. This type of activity calls an AWS Lambda function or web hook that sends
 * messages to participants.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CustomMessageActivity implements SdkPojo, Serializable,
        ToCopyableBuilder<CustomMessageActivity.Builder, CustomMessageActivity> {
    private static final SdkField<String> DELIVERY_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeliveryUri").getter(getter(CustomMessageActivity::deliveryUri)).setter(setter(Builder::deliveryUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeliveryUri").build()).build();

    private static final SdkField<List<String>> ENDPOINT_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EndpointTypes")
            .getter(getter(CustomMessageActivity::endpointTypesAsStrings))
            .setter(setter(Builder::endpointTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<JourneyCustomMessage> MESSAGE_CONFIG_FIELD = SdkField
            .<JourneyCustomMessage> builder(MarshallingType.SDK_POJO).memberName("MessageConfig")
            .getter(getter(CustomMessageActivity::messageConfig)).setter(setter(Builder::messageConfig))
            .constructor(JourneyCustomMessage::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageConfig").build()).build();

    private static final SdkField<String> NEXT_ACTIVITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NextActivity").getter(getter(CustomMessageActivity::nextActivity)).setter(setter(Builder::nextActivity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextActivity").build()).build();

    private static final SdkField<String> TEMPLATE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TemplateName").getter(getter(CustomMessageActivity::templateName)).setter(setter(Builder::templateName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemplateName").build()).build();

    private static final SdkField<String> TEMPLATE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TemplateVersion").getter(getter(CustomMessageActivity::templateVersion))
            .setter(setter(Builder::templateVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemplateVersion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DELIVERY_URI_FIELD,
            ENDPOINT_TYPES_FIELD, MESSAGE_CONFIG_FIELD, NEXT_ACTIVITY_FIELD, TEMPLATE_NAME_FIELD, TEMPLATE_VERSION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String deliveryUri;

    private final List<String> endpointTypes;

    private final JourneyCustomMessage messageConfig;

    private final String nextActivity;

    private final String templateName;

    private final String templateVersion;

    private CustomMessageActivity(BuilderImpl builder) {
        this.deliveryUri = builder.deliveryUri;
        this.endpointTypes = builder.endpointTypes;
        this.messageConfig = builder.messageConfig;
        this.nextActivity = builder.nextActivity;
        this.templateName = builder.templateName;
        this.templateVersion = builder.templateVersion;
    }

    /**
     * <p>
     * The destination to send the campaign or treatment to. This value can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The name or Amazon Resource Name (ARN) of an AWS Lambda function to invoke to handle delivery of the campaign or
     * treatment.
     * </p>
     * </li>
     * <li>
     * <p>
     * The URL for a web application or service that supports HTTPS and can receive the message. The URL has to be a
     * full URL, including the HTTPS protocol.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The destination to send the campaign or treatment to. This value can be one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The name or Amazon Resource Name (ARN) of an AWS Lambda function to invoke to handle delivery of the
     *         campaign or treatment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The URL for a web application or service that supports HTTPS and can receive the message. The URL has to
     *         be a full URL, including the HTTPS protocol.
     *         </p>
     *         </li>
     */
    public final String deliveryUri() {
        return deliveryUri;
    }

    /**
     * <p>
     * The types of endpoints to send the custom message to. Each valid value maps to a type of channel that you can
     * associate with an endpoint by using the ChannelType property of an endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEndpointTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The types of endpoints to send the custom message to. Each valid value maps to a type of channel that you
     *         can associate with an endpoint by using the ChannelType property of an endpoint.
     */
    public final List<EndpointTypesElement> endpointTypes() {
        return ListOf__EndpointTypesElementCopier.copyStringToEnum(endpointTypes);
    }

    /**
     * Returns true if the EndpointTypes property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasEndpointTypes() {
        return endpointTypes != null && !(endpointTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The types of endpoints to send the custom message to. Each valid value maps to a type of channel that you can
     * associate with an endpoint by using the ChannelType property of an endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEndpointTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The types of endpoints to send the custom message to. Each valid value maps to a type of channel that you
     *         can associate with an endpoint by using the ChannelType property of an endpoint.
     */
    public final List<String> endpointTypesAsStrings() {
        return endpointTypes;
    }

    /**
     * <p>
     * Specifies the message data included in a custom channel message that's sent to participants in a journey.
     * </p>
     * 
     * @return Specifies the message data included in a custom channel message that's sent to participants in a journey.
     */
    public final JourneyCustomMessage messageConfig() {
        return messageConfig;
    }

    /**
     * <p>
     * The unique identifier for the next activity to perform, after Amazon Pinpoint calls the AWS Lambda function or
     * web hook.
     * </p>
     * 
     * @return The unique identifier for the next activity to perform, after Amazon Pinpoint calls the AWS Lambda
     *         function or web hook.
     */
    public final String nextActivity() {
        return nextActivity;
    }

    /**
     * <p>
     * The name of the custom message template to use for the message. If specified, this value must match the name of
     * an existing message template.
     * </p>
     * 
     * @return The name of the custom message template to use for the message. If specified, this value must match the
     *         name of an existing message template.
     */
    public final String templateName() {
        return templateName;
    }

    /**
     * <p>
     * The unique identifier for the version of the message template to use for the message. If specified, this value
     * must match the identifier for an existing template version. To retrieve a list of versions and version
     * identifiers for a template, use the <link linkend="templates-template-name-template-type-versions">Template
     * Versions</link> resource.
     * </p>
     * <p>
     * If you don't specify a value for this property, Amazon Pinpoint uses the <i>active version</i> of the template.
     * The <i>active version</i> is typically the version of a template that's been most recently reviewed and approved
     * for use, depending on your workflow. It isn't necessarily the latest version of a template.
     * </p>
     * 
     * @return The unique identifier for the version of the message template to use for the message. If specified, this
     *         value must match the identifier for an existing template version. To retrieve a list of versions and
     *         version identifiers for a template, use the <link
     *         linkend="templates-template-name-template-type-versions">Template Versions</link> resource.</p>
     *         <p>
     *         If you don't specify a value for this property, Amazon Pinpoint uses the <i>active version</i> of the
     *         template. The <i>active version</i> is typically the version of a template that's been most recently
     *         reviewed and approved for use, depending on your workflow. It isn't necessarily the latest version of a
     *         template.
     */
    public final String templateVersion() {
        return templateVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deliveryUri());
        hashCode = 31 * hashCode + Objects.hashCode(hasEndpointTypes() ? endpointTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(messageConfig());
        hashCode = 31 * hashCode + Objects.hashCode(nextActivity());
        hashCode = 31 * hashCode + Objects.hashCode(templateName());
        hashCode = 31 * hashCode + Objects.hashCode(templateVersion());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomMessageActivity)) {
            return false;
        }
        CustomMessageActivity other = (CustomMessageActivity) obj;
        return Objects.equals(deliveryUri(), other.deliveryUri()) && hasEndpointTypes() == other.hasEndpointTypes()
                && Objects.equals(endpointTypesAsStrings(), other.endpointTypesAsStrings())
                && Objects.equals(messageConfig(), other.messageConfig()) && Objects.equals(nextActivity(), other.nextActivity())
                && Objects.equals(templateName(), other.templateName())
                && Objects.equals(templateVersion(), other.templateVersion());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CustomMessageActivity").add("DeliveryUri", deliveryUri())
                .add("EndpointTypes", hasEndpointTypes() ? endpointTypesAsStrings() : null).add("MessageConfig", messageConfig())
                .add("NextActivity", nextActivity()).add("TemplateName", templateName())
                .add("TemplateVersion", templateVersion()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DeliveryUri":
            return Optional.ofNullable(clazz.cast(deliveryUri()));
        case "EndpointTypes":
            return Optional.ofNullable(clazz.cast(endpointTypesAsStrings()));
        case "MessageConfig":
            return Optional.ofNullable(clazz.cast(messageConfig()));
        case "NextActivity":
            return Optional.ofNullable(clazz.cast(nextActivity()));
        case "TemplateName":
            return Optional.ofNullable(clazz.cast(templateName()));
        case "TemplateVersion":
            return Optional.ofNullable(clazz.cast(templateVersion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CustomMessageActivity, T> g) {
        return obj -> g.apply((CustomMessageActivity) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CustomMessageActivity> {
        /**
         * <p>
         * The destination to send the campaign or treatment to. This value can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The name or Amazon Resource Name (ARN) of an AWS Lambda function to invoke to handle delivery of the campaign
         * or treatment.
         * </p>
         * </li>
         * <li>
         * <p>
         * The URL for a web application or service that supports HTTPS and can receive the message. The URL has to be a
         * full URL, including the HTTPS protocol.
         * </p>
         * </li>
         * </ul>
         * 
         * @param deliveryUri
         *        The destination to send the campaign or treatment to. This value can be one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The name or Amazon Resource Name (ARN) of an AWS Lambda function to invoke to handle delivery of the
         *        campaign or treatment.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The URL for a web application or service that supports HTTPS and can receive the message. The URL has
         *        to be a full URL, including the HTTPS protocol.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliveryUri(String deliveryUri);

        /**
         * <p>
         * The types of endpoints to send the custom message to. Each valid value maps to a type of channel that you can
         * associate with an endpoint by using the ChannelType property of an endpoint.
         * </p>
         * 
         * @param endpointTypes
         *        The types of endpoints to send the custom message to. Each valid value maps to a type of channel that
         *        you can associate with an endpoint by using the ChannelType property of an endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointTypesWithStrings(Collection<String> endpointTypes);

        /**
         * <p>
         * The types of endpoints to send the custom message to. Each valid value maps to a type of channel that you can
         * associate with an endpoint by using the ChannelType property of an endpoint.
         * </p>
         * 
         * @param endpointTypes
         *        The types of endpoints to send the custom message to. Each valid value maps to a type of channel that
         *        you can associate with an endpoint by using the ChannelType property of an endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointTypesWithStrings(String... endpointTypes);

        /**
         * <p>
         * The types of endpoints to send the custom message to. Each valid value maps to a type of channel that you can
         * associate with an endpoint by using the ChannelType property of an endpoint.
         * </p>
         * 
         * @param endpointTypes
         *        The types of endpoints to send the custom message to. Each valid value maps to a type of channel that
         *        you can associate with an endpoint by using the ChannelType property of an endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointTypes(Collection<EndpointTypesElement> endpointTypes);

        /**
         * <p>
         * The types of endpoints to send the custom message to. Each valid value maps to a type of channel that you can
         * associate with an endpoint by using the ChannelType property of an endpoint.
         * </p>
         * 
         * @param endpointTypes
         *        The types of endpoints to send the custom message to. Each valid value maps to a type of channel that
         *        you can associate with an endpoint by using the ChannelType property of an endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointTypes(EndpointTypesElement... endpointTypes);

        /**
         * <p>
         * Specifies the message data included in a custom channel message that's sent to participants in a journey.
         * </p>
         * 
         * @param messageConfig
         *        Specifies the message data included in a custom channel message that's sent to participants in a
         *        journey.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageConfig(JourneyCustomMessage messageConfig);

        /**
         * <p>
         * Specifies the message data included in a custom channel message that's sent to participants in a journey.
         * </p>
         * This is a convenience that creates an instance of the {@link JourneyCustomMessage.Builder} avoiding the need
         * to create one manually via {@link JourneyCustomMessage#builder()}.
         *
         * When the {@link Consumer} completes, {@link JourneyCustomMessage.Builder#build()} is called immediately and
         * its result is passed to {@link #messageConfig(JourneyCustomMessage)}.
         * 
         * @param messageConfig
         *        a consumer that will call methods on {@link JourneyCustomMessage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #messageConfig(JourneyCustomMessage)
         */
        default Builder messageConfig(Consumer<JourneyCustomMessage.Builder> messageConfig) {
            return messageConfig(JourneyCustomMessage.builder().applyMutation(messageConfig).build());
        }

        /**
         * <p>
         * The unique identifier for the next activity to perform, after Amazon Pinpoint calls the AWS Lambda function
         * or web hook.
         * </p>
         * 
         * @param nextActivity
         *        The unique identifier for the next activity to perform, after Amazon Pinpoint calls the AWS Lambda
         *        function or web hook.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextActivity(String nextActivity);

        /**
         * <p>
         * The name of the custom message template to use for the message. If specified, this value must match the name
         * of an existing message template.
         * </p>
         * 
         * @param templateName
         *        The name of the custom message template to use for the message. If specified, this value must match
         *        the name of an existing message template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateName(String templateName);

        /**
         * <p>
         * The unique identifier for the version of the message template to use for the message. If specified, this
         * value must match the identifier for an existing template version. To retrieve a list of versions and version
         * identifiers for a template, use the <link linkend="templates-template-name-template-type-versions">Template
         * Versions</link> resource.
         * </p>
         * <p>
         * If you don't specify a value for this property, Amazon Pinpoint uses the <i>active version</i> of the
         * template. The <i>active version</i> is typically the version of a template that's been most recently reviewed
         * and approved for use, depending on your workflow. It isn't necessarily the latest version of a template.
         * </p>
         * 
         * @param templateVersion
         *        The unique identifier for the version of the message template to use for the message. If specified,
         *        this value must match the identifier for an existing template version. To retrieve a list of versions
         *        and version identifiers for a template, use the <link
         *        linkend="templates-template-name-template-type-versions">Template Versions</link> resource.</p>
         *        <p>
         *        If you don't specify a value for this property, Amazon Pinpoint uses the <i>active version</i> of the
         *        template. The <i>active version</i> is typically the version of a template that's been most recently
         *        reviewed and approved for use, depending on your workflow. It isn't necessarily the latest version of
         *        a template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateVersion(String templateVersion);
    }

    static final class BuilderImpl implements Builder {
        private String deliveryUri;

        private List<String> endpointTypes = DefaultSdkAutoConstructList.getInstance();

        private JourneyCustomMessage messageConfig;

        private String nextActivity;

        private String templateName;

        private String templateVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(CustomMessageActivity model) {
            deliveryUri(model.deliveryUri);
            endpointTypesWithStrings(model.endpointTypes);
            messageConfig(model.messageConfig);
            nextActivity(model.nextActivity);
            templateName(model.templateName);
            templateVersion(model.templateVersion);
        }

        public final String getDeliveryUri() {
            return deliveryUri;
        }

        @Override
        public final Builder deliveryUri(String deliveryUri) {
            this.deliveryUri = deliveryUri;
            return this;
        }

        public final void setDeliveryUri(String deliveryUri) {
            this.deliveryUri = deliveryUri;
        }

        public final Collection<String> getEndpointTypes() {
            if (endpointTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return endpointTypes;
        }

        @Override
        public final Builder endpointTypesWithStrings(Collection<String> endpointTypes) {
            this.endpointTypes = ListOf__EndpointTypesElementCopier.copy(endpointTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder endpointTypesWithStrings(String... endpointTypes) {
            endpointTypesWithStrings(Arrays.asList(endpointTypes));
            return this;
        }

        @Override
        public final Builder endpointTypes(Collection<EndpointTypesElement> endpointTypes) {
            this.endpointTypes = ListOf__EndpointTypesElementCopier.copyEnumToString(endpointTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder endpointTypes(EndpointTypesElement... endpointTypes) {
            endpointTypes(Arrays.asList(endpointTypes));
            return this;
        }

        public final void setEndpointTypes(Collection<String> endpointTypes) {
            this.endpointTypes = ListOf__EndpointTypesElementCopier.copy(endpointTypes);
        }

        public final JourneyCustomMessage.Builder getMessageConfig() {
            return messageConfig != null ? messageConfig.toBuilder() : null;
        }

        @Override
        public final Builder messageConfig(JourneyCustomMessage messageConfig) {
            this.messageConfig = messageConfig;
            return this;
        }

        public final void setMessageConfig(JourneyCustomMessage.BuilderImpl messageConfig) {
            this.messageConfig = messageConfig != null ? messageConfig.build() : null;
        }

        public final String getNextActivity() {
            return nextActivity;
        }

        @Override
        public final Builder nextActivity(String nextActivity) {
            this.nextActivity = nextActivity;
            return this;
        }

        public final void setNextActivity(String nextActivity) {
            this.nextActivity = nextActivity;
        }

        public final String getTemplateName() {
            return templateName;
        }

        @Override
        public final Builder templateName(String templateName) {
            this.templateName = templateName;
            return this;
        }

        public final void setTemplateName(String templateName) {
            this.templateName = templateName;
        }

        public final String getTemplateVersion() {
            return templateVersion;
        }

        @Override
        public final Builder templateVersion(String templateVersion) {
            this.templateVersion = templateVersion;
            return this;
        }

        public final void setTemplateVersion(String templateVersion) {
            this.templateVersion = templateVersion;
        }

        @Override
        public CustomMessageActivity build() {
            return new CustomMessageActivity(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
