/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutWorkflowRunPropertiesRequest extends GlueRequest implements
        ToCopyableBuilder<PutWorkflowRunPropertiesRequest.Builder, PutWorkflowRunPropertiesRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(PutWorkflowRunPropertiesRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> RUN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("RunId")
            .getter(getter(PutWorkflowRunPropertiesRequest::runId)).setter(setter(Builder::runId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RunId").build()).build();

    private static final SdkField<Map<String, String>> RUN_PROPERTIES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("RunProperties")
            .getter(getter(PutWorkflowRunPropertiesRequest::runProperties))
            .setter(setter(Builder::runProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RunProperties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, RUN_ID_FIELD,
            RUN_PROPERTIES_FIELD));

    private final String name;

    private final String runId;

    private final Map<String, String> runProperties;

    private PutWorkflowRunPropertiesRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.runId = builder.runId;
        this.runProperties = builder.runProperties;
    }

    /**
     * <p>
     * Name of the workflow which was run.
     * </p>
     * 
     * @return Name of the workflow which was run.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The ID of the workflow run for which the run properties should be updated.
     * </p>
     * 
     * @return The ID of the workflow run for which the run properties should be updated.
     */
    public final String runId() {
        return runId;
    }

    /**
     * Returns true if the RunProperties property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasRunProperties() {
        return runProperties != null && !(runProperties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The properties to put for the specified run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRunProperties()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The properties to put for the specified run.
     */
    public final Map<String, String> runProperties() {
        return runProperties;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(runId());
        hashCode = 31 * hashCode + Objects.hashCode(hasRunProperties() ? runProperties() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutWorkflowRunPropertiesRequest)) {
            return false;
        }
        PutWorkflowRunPropertiesRequest other = (PutWorkflowRunPropertiesRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(runId(), other.runId())
                && hasRunProperties() == other.hasRunProperties() && Objects.equals(runProperties(), other.runProperties());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PutWorkflowRunPropertiesRequest").add("Name", name()).add("RunId", runId())
                .add("RunProperties", hasRunProperties() ? runProperties() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "RunId":
            return Optional.ofNullable(clazz.cast(runId()));
        case "RunProperties":
            return Optional.ofNullable(clazz.cast(runProperties()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutWorkflowRunPropertiesRequest, T> g) {
        return obj -> g.apply((PutWorkflowRunPropertiesRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends GlueRequest.Builder, SdkPojo, CopyableBuilder<Builder, PutWorkflowRunPropertiesRequest> {
        /**
         * <p>
         * Name of the workflow which was run.
         * </p>
         * 
         * @param name
         *        Name of the workflow which was run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ID of the workflow run for which the run properties should be updated.
         * </p>
         * 
         * @param runId
         *        The ID of the workflow run for which the run properties should be updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runId(String runId);

        /**
         * <p>
         * The properties to put for the specified run.
         * </p>
         * 
         * @param runProperties
         *        The properties to put for the specified run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runProperties(Map<String, String> runProperties);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends GlueRequest.BuilderImpl implements Builder {
        private String name;

        private String runId;

        private Map<String, String> runProperties = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PutWorkflowRunPropertiesRequest model) {
            super(model);
            name(model.name);
            runId(model.runId);
            runProperties(model.runProperties);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getRunId() {
            return runId;
        }

        @Override
        public final Builder runId(String runId) {
            this.runId = runId;
            return this;
        }

        public final void setRunId(String runId) {
            this.runId = runId;
        }

        public final Map<String, String> getRunProperties() {
            if (runProperties instanceof SdkAutoConstructMap) {
                return null;
            }
            return runProperties;
        }

        @Override
        public final Builder runProperties(Map<String, String> runProperties) {
            this.runProperties = WorkflowRunPropertiesCopier.copy(runProperties);
            return this;
        }

        public final void setRunProperties(Map<String, String> runProperties) {
            this.runProperties = WorkflowRunPropertiesCopier.copy(runProperties);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutWorkflowRunPropertiesRequest build() {
            return new PutWorkflowRunPropertiesRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
