/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databasemigration.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * In response to the <code>DescribeOrderableReplicationInstances</code> operation, this object describes an available
 * replication instance. This description includes the replication instance's type, engine version, and allocated
 * storage.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OrderableReplicationInstance implements SdkPojo, Serializable,
        ToCopyableBuilder<OrderableReplicationInstance.Builder, OrderableReplicationInstance> {
    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EngineVersion").getter(getter(OrderableReplicationInstance::engineVersion))
            .setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<String> REPLICATION_INSTANCE_CLASS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReplicationInstanceClass").getter(getter(OrderableReplicationInstance::replicationInstanceClass))
            .setter(setter(Builder::replicationInstanceClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationInstanceClass").build())
            .build();

    private static final SdkField<String> STORAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StorageType").getter(getter(OrderableReplicationInstance::storageType))
            .setter(setter(Builder::storageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageType").build()).build();

    private static final SdkField<Integer> MIN_ALLOCATED_STORAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinAllocatedStorage").getter(getter(OrderableReplicationInstance::minAllocatedStorage))
            .setter(setter(Builder::minAllocatedStorage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinAllocatedStorage").build())
            .build();

    private static final SdkField<Integer> MAX_ALLOCATED_STORAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxAllocatedStorage").getter(getter(OrderableReplicationInstance::maxAllocatedStorage))
            .setter(setter(Builder::maxAllocatedStorage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxAllocatedStorage").build())
            .build();

    private static final SdkField<Integer> DEFAULT_ALLOCATED_STORAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("DefaultAllocatedStorage").getter(getter(OrderableReplicationInstance::defaultAllocatedStorage))
            .setter(setter(Builder::defaultAllocatedStorage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultAllocatedStorage").build())
            .build();

    private static final SdkField<Integer> INCLUDED_ALLOCATED_STORAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("IncludedAllocatedStorage").getter(getter(OrderableReplicationInstance::includedAllocatedStorage))
            .setter(setter(Builder::includedAllocatedStorage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludedAllocatedStorage").build())
            .build();

    private static final SdkField<List<String>> AVAILABILITY_ZONES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AvailabilityZones")
            .getter(getter(OrderableReplicationInstance::availabilityZones))
            .setter(setter(Builder::availabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RELEASE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReleaseStatus").getter(getter(OrderableReplicationInstance::releaseStatusAsString))
            .setter(setter(Builder::releaseStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReleaseStatus").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENGINE_VERSION_FIELD,
            REPLICATION_INSTANCE_CLASS_FIELD, STORAGE_TYPE_FIELD, MIN_ALLOCATED_STORAGE_FIELD, MAX_ALLOCATED_STORAGE_FIELD,
            DEFAULT_ALLOCATED_STORAGE_FIELD, INCLUDED_ALLOCATED_STORAGE_FIELD, AVAILABILITY_ZONES_FIELD, RELEASE_STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String engineVersion;

    private final String replicationInstanceClass;

    private final String storageType;

    private final Integer minAllocatedStorage;

    private final Integer maxAllocatedStorage;

    private final Integer defaultAllocatedStorage;

    private final Integer includedAllocatedStorage;

    private final List<String> availabilityZones;

    private final String releaseStatus;

    private OrderableReplicationInstance(BuilderImpl builder) {
        this.engineVersion = builder.engineVersion;
        this.replicationInstanceClass = builder.replicationInstanceClass;
        this.storageType = builder.storageType;
        this.minAllocatedStorage = builder.minAllocatedStorage;
        this.maxAllocatedStorage = builder.maxAllocatedStorage;
        this.defaultAllocatedStorage = builder.defaultAllocatedStorage;
        this.includedAllocatedStorage = builder.includedAllocatedStorage;
        this.availabilityZones = builder.availabilityZones;
        this.releaseStatus = builder.releaseStatus;
    }

    /**
     * <p>
     * The version of the replication engine.
     * </p>
     * 
     * @return The version of the replication engine.
     */
    public final String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * The compute and memory capacity of the replication instance as defined for the specified replication instance
     * class. For example to specify the instance class dms.c4.large, set this parameter to <code>"dms.c4.large"</code>.
     * </p>
     * <p>
     * For more information on the settings and capacities for the available replication instance classes, see <a href=
     * "https://docs.aws.amazon.com/dms/latest/userguide/CHAP_ReplicationInstance.html#CHAP_ReplicationInstance.InDepth"
     * > Selecting the right AWS DMS replication instance for your migration</a>.
     * </p>
     * 
     * @return The compute and memory capacity of the replication instance as defined for the specified replication
     *         instance class. For example to specify the instance class dms.c4.large, set this parameter to
     *         <code>"dms.c4.large"</code>.</p>
     *         <p>
     *         For more information on the settings and capacities for the available replication instance classes, see
     *         <a href=
     *         "https://docs.aws.amazon.com/dms/latest/userguide/CHAP_ReplicationInstance.html#CHAP_ReplicationInstance.InDepth"
     *         > Selecting the right AWS DMS replication instance for your migration</a>.
     */
    public final String replicationInstanceClass() {
        return replicationInstanceClass;
    }

    /**
     * <p>
     * The type of storage used by the replication instance.
     * </p>
     * 
     * @return The type of storage used by the replication instance.
     */
    public final String storageType() {
        return storageType;
    }

    /**
     * <p>
     * The minimum amount of storage (in gigabytes) that can be allocated for the replication instance.
     * </p>
     * 
     * @return The minimum amount of storage (in gigabytes) that can be allocated for the replication instance.
     */
    public final Integer minAllocatedStorage() {
        return minAllocatedStorage;
    }

    /**
     * <p>
     * The minimum amount of storage (in gigabytes) that can be allocated for the replication instance.
     * </p>
     * 
     * @return The minimum amount of storage (in gigabytes) that can be allocated for the replication instance.
     */
    public final Integer maxAllocatedStorage() {
        return maxAllocatedStorage;
    }

    /**
     * <p>
     * The default amount of storage (in gigabytes) that is allocated for the replication instance.
     * </p>
     * 
     * @return The default amount of storage (in gigabytes) that is allocated for the replication instance.
     */
    public final Integer defaultAllocatedStorage() {
        return defaultAllocatedStorage;
    }

    /**
     * <p>
     * The amount of storage (in gigabytes) that is allocated for the replication instance.
     * </p>
     * 
     * @return The amount of storage (in gigabytes) that is allocated for the replication instance.
     */
    public final Integer includedAllocatedStorage() {
        return includedAllocatedStorage;
    }

    /**
     * Returns true if the AvailabilityZones property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasAvailabilityZones() {
        return availabilityZones != null && !(availabilityZones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of Availability Zones for this replication instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAvailabilityZones()} to see if a value was sent in this field.
     * </p>
     * 
     * @return List of Availability Zones for this replication instance.
     */
    public final List<String> availabilityZones() {
        return availabilityZones;
    }

    /**
     * <p>
     * The value returned when the specified <code>EngineVersion</code> of the replication instance is in Beta or test
     * mode. This indicates some features might not work as expected.
     * </p>
     * <note>
     * <p>
     * AWS DMS supports the <code>ReleaseStatus</code> parameter in versions 3.1.4 and later.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #releaseStatus}
     * will return {@link ReleaseStatusValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #releaseStatusAsString}.
     * </p>
     * 
     * @return The value returned when the specified <code>EngineVersion</code> of the replication instance is in Beta
     *         or test mode. This indicates some features might not work as expected.</p> <note>
     *         <p>
     *         AWS DMS supports the <code>ReleaseStatus</code> parameter in versions 3.1.4 and later.
     *         </p>
     * @see ReleaseStatusValues
     */
    public final ReleaseStatusValues releaseStatus() {
        return ReleaseStatusValues.fromValue(releaseStatus);
    }

    /**
     * <p>
     * The value returned when the specified <code>EngineVersion</code> of the replication instance is in Beta or test
     * mode. This indicates some features might not work as expected.
     * </p>
     * <note>
     * <p>
     * AWS DMS supports the <code>ReleaseStatus</code> parameter in versions 3.1.4 and later.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #releaseStatus}
     * will return {@link ReleaseStatusValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #releaseStatusAsString}.
     * </p>
     * 
     * @return The value returned when the specified <code>EngineVersion</code> of the replication instance is in Beta
     *         or test mode. This indicates some features might not work as expected.</p> <note>
     *         <p>
     *         AWS DMS supports the <code>ReleaseStatus</code> parameter in versions 3.1.4 and later.
     *         </p>
     * @see ReleaseStatusValues
     */
    public final String releaseStatusAsString() {
        return releaseStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(replicationInstanceClass());
        hashCode = 31 * hashCode + Objects.hashCode(storageType());
        hashCode = 31 * hashCode + Objects.hashCode(minAllocatedStorage());
        hashCode = 31 * hashCode + Objects.hashCode(maxAllocatedStorage());
        hashCode = 31 * hashCode + Objects.hashCode(defaultAllocatedStorage());
        hashCode = 31 * hashCode + Objects.hashCode(includedAllocatedStorage());
        hashCode = 31 * hashCode + Objects.hashCode(hasAvailabilityZones() ? availabilityZones() : null);
        hashCode = 31 * hashCode + Objects.hashCode(releaseStatusAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OrderableReplicationInstance)) {
            return false;
        }
        OrderableReplicationInstance other = (OrderableReplicationInstance) obj;
        return Objects.equals(engineVersion(), other.engineVersion())
                && Objects.equals(replicationInstanceClass(), other.replicationInstanceClass())
                && Objects.equals(storageType(), other.storageType())
                && Objects.equals(minAllocatedStorage(), other.minAllocatedStorage())
                && Objects.equals(maxAllocatedStorage(), other.maxAllocatedStorage())
                && Objects.equals(defaultAllocatedStorage(), other.defaultAllocatedStorage())
                && Objects.equals(includedAllocatedStorage(), other.includedAllocatedStorage())
                && hasAvailabilityZones() == other.hasAvailabilityZones()
                && Objects.equals(availabilityZones(), other.availabilityZones())
                && Objects.equals(releaseStatusAsString(), other.releaseStatusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OrderableReplicationInstance").add("EngineVersion", engineVersion())
                .add("ReplicationInstanceClass", replicationInstanceClass()).add("StorageType", storageType())
                .add("MinAllocatedStorage", minAllocatedStorage()).add("MaxAllocatedStorage", maxAllocatedStorage())
                .add("DefaultAllocatedStorage", defaultAllocatedStorage())
                .add("IncludedAllocatedStorage", includedAllocatedStorage())
                .add("AvailabilityZones", hasAvailabilityZones() ? availabilityZones() : null)
                .add("ReleaseStatus", releaseStatusAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "ReplicationInstanceClass":
            return Optional.ofNullable(clazz.cast(replicationInstanceClass()));
        case "StorageType":
            return Optional.ofNullable(clazz.cast(storageType()));
        case "MinAllocatedStorage":
            return Optional.ofNullable(clazz.cast(minAllocatedStorage()));
        case "MaxAllocatedStorage":
            return Optional.ofNullable(clazz.cast(maxAllocatedStorage()));
        case "DefaultAllocatedStorage":
            return Optional.ofNullable(clazz.cast(defaultAllocatedStorage()));
        case "IncludedAllocatedStorage":
            return Optional.ofNullable(clazz.cast(includedAllocatedStorage()));
        case "AvailabilityZones":
            return Optional.ofNullable(clazz.cast(availabilityZones()));
        case "ReleaseStatus":
            return Optional.ofNullable(clazz.cast(releaseStatusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OrderableReplicationInstance, T> g) {
        return obj -> g.apply((OrderableReplicationInstance) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OrderableReplicationInstance> {
        /**
         * <p>
         * The version of the replication engine.
         * </p>
         * 
         * @param engineVersion
         *        The version of the replication engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * The compute and memory capacity of the replication instance as defined for the specified replication instance
         * class. For example to specify the instance class dms.c4.large, set this parameter to
         * <code>"dms.c4.large"</code>.
         * </p>
         * <p>
         * For more information on the settings and capacities for the available replication instance classes, see <a
         * href=
         * "https://docs.aws.amazon.com/dms/latest/userguide/CHAP_ReplicationInstance.html#CHAP_ReplicationInstance.InDepth"
         * > Selecting the right AWS DMS replication instance for your migration</a>.
         * </p>
         * 
         * @param replicationInstanceClass
         *        The compute and memory capacity of the replication instance as defined for the specified replication
         *        instance class. For example to specify the instance class dms.c4.large, set this parameter to
         *        <code>"dms.c4.large"</code>.</p>
         *        <p>
         *        For more information on the settings and capacities for the available replication instance classes,
         *        see <a href=
         *        "https://docs.aws.amazon.com/dms/latest/userguide/CHAP_ReplicationInstance.html#CHAP_ReplicationInstance.InDepth"
         *        > Selecting the right AWS DMS replication instance for your migration</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationInstanceClass(String replicationInstanceClass);

        /**
         * <p>
         * The type of storage used by the replication instance.
         * </p>
         * 
         * @param storageType
         *        The type of storage used by the replication instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageType(String storageType);

        /**
         * <p>
         * The minimum amount of storage (in gigabytes) that can be allocated for the replication instance.
         * </p>
         * 
         * @param minAllocatedStorage
         *        The minimum amount of storage (in gigabytes) that can be allocated for the replication instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minAllocatedStorage(Integer minAllocatedStorage);

        /**
         * <p>
         * The minimum amount of storage (in gigabytes) that can be allocated for the replication instance.
         * </p>
         * 
         * @param maxAllocatedStorage
         *        The minimum amount of storage (in gigabytes) that can be allocated for the replication instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxAllocatedStorage(Integer maxAllocatedStorage);

        /**
         * <p>
         * The default amount of storage (in gigabytes) that is allocated for the replication instance.
         * </p>
         * 
         * @param defaultAllocatedStorage
         *        The default amount of storage (in gigabytes) that is allocated for the replication instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultAllocatedStorage(Integer defaultAllocatedStorage);

        /**
         * <p>
         * The amount of storage (in gigabytes) that is allocated for the replication instance.
         * </p>
         * 
         * @param includedAllocatedStorage
         *        The amount of storage (in gigabytes) that is allocated for the replication instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includedAllocatedStorage(Integer includedAllocatedStorage);

        /**
         * <p>
         * List of Availability Zones for this replication instance.
         * </p>
         * 
         * @param availabilityZones
         *        List of Availability Zones for this replication instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(Collection<String> availabilityZones);

        /**
         * <p>
         * List of Availability Zones for this replication instance.
         * </p>
         * 
         * @param availabilityZones
         *        List of Availability Zones for this replication instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(String... availabilityZones);

        /**
         * <p>
         * The value returned when the specified <code>EngineVersion</code> of the replication instance is in Beta or
         * test mode. This indicates some features might not work as expected.
         * </p>
         * <note>
         * <p>
         * AWS DMS supports the <code>ReleaseStatus</code> parameter in versions 3.1.4 and later.
         * </p>
         * </note>
         * 
         * @param releaseStatus
         *        The value returned when the specified <code>EngineVersion</code> of the replication instance is in
         *        Beta or test mode. This indicates some features might not work as expected.</p> <note>
         *        <p>
         *        AWS DMS supports the <code>ReleaseStatus</code> parameter in versions 3.1.4 and later.
         *        </p>
         * @see ReleaseStatusValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReleaseStatusValues
         */
        Builder releaseStatus(String releaseStatus);

        /**
         * <p>
         * The value returned when the specified <code>EngineVersion</code> of the replication instance is in Beta or
         * test mode. This indicates some features might not work as expected.
         * </p>
         * <note>
         * <p>
         * AWS DMS supports the <code>ReleaseStatus</code> parameter in versions 3.1.4 and later.
         * </p>
         * </note>
         * 
         * @param releaseStatus
         *        The value returned when the specified <code>EngineVersion</code> of the replication instance is in
         *        Beta or test mode. This indicates some features might not work as expected.</p> <note>
         *        <p>
         *        AWS DMS supports the <code>ReleaseStatus</code> parameter in versions 3.1.4 and later.
         *        </p>
         * @see ReleaseStatusValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReleaseStatusValues
         */
        Builder releaseStatus(ReleaseStatusValues releaseStatus);
    }

    static final class BuilderImpl implements Builder {
        private String engineVersion;

        private String replicationInstanceClass;

        private String storageType;

        private Integer minAllocatedStorage;

        private Integer maxAllocatedStorage;

        private Integer defaultAllocatedStorage;

        private Integer includedAllocatedStorage;

        private List<String> availabilityZones = DefaultSdkAutoConstructList.getInstance();

        private String releaseStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(OrderableReplicationInstance model) {
            engineVersion(model.engineVersion);
            replicationInstanceClass(model.replicationInstanceClass);
            storageType(model.storageType);
            minAllocatedStorage(model.minAllocatedStorage);
            maxAllocatedStorage(model.maxAllocatedStorage);
            defaultAllocatedStorage(model.defaultAllocatedStorage);
            includedAllocatedStorage(model.includedAllocatedStorage);
            availabilityZones(model.availabilityZones);
            releaseStatus(model.releaseStatus);
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        public final String getReplicationInstanceClass() {
            return replicationInstanceClass;
        }

        @Override
        public final Builder replicationInstanceClass(String replicationInstanceClass) {
            this.replicationInstanceClass = replicationInstanceClass;
            return this;
        }

        public final void setReplicationInstanceClass(String replicationInstanceClass) {
            this.replicationInstanceClass = replicationInstanceClass;
        }

        public final String getStorageType() {
            return storageType;
        }

        @Override
        public final Builder storageType(String storageType) {
            this.storageType = storageType;
            return this;
        }

        public final void setStorageType(String storageType) {
            this.storageType = storageType;
        }

        public final Integer getMinAllocatedStorage() {
            return minAllocatedStorage;
        }

        @Override
        public final Builder minAllocatedStorage(Integer minAllocatedStorage) {
            this.minAllocatedStorage = minAllocatedStorage;
            return this;
        }

        public final void setMinAllocatedStorage(Integer minAllocatedStorage) {
            this.minAllocatedStorage = minAllocatedStorage;
        }

        public final Integer getMaxAllocatedStorage() {
            return maxAllocatedStorage;
        }

        @Override
        public final Builder maxAllocatedStorage(Integer maxAllocatedStorage) {
            this.maxAllocatedStorage = maxAllocatedStorage;
            return this;
        }

        public final void setMaxAllocatedStorage(Integer maxAllocatedStorage) {
            this.maxAllocatedStorage = maxAllocatedStorage;
        }

        public final Integer getDefaultAllocatedStorage() {
            return defaultAllocatedStorage;
        }

        @Override
        public final Builder defaultAllocatedStorage(Integer defaultAllocatedStorage) {
            this.defaultAllocatedStorage = defaultAllocatedStorage;
            return this;
        }

        public final void setDefaultAllocatedStorage(Integer defaultAllocatedStorage) {
            this.defaultAllocatedStorage = defaultAllocatedStorage;
        }

        public final Integer getIncludedAllocatedStorage() {
            return includedAllocatedStorage;
        }

        @Override
        public final Builder includedAllocatedStorage(Integer includedAllocatedStorage) {
            this.includedAllocatedStorage = includedAllocatedStorage;
            return this;
        }

        public final void setIncludedAllocatedStorage(Integer includedAllocatedStorage) {
            this.includedAllocatedStorage = includedAllocatedStorage;
        }

        public final Collection<String> getAvailabilityZones() {
            if (availabilityZones instanceof SdkAutoConstructList) {
                return null;
            }
            return availabilityZones;
        }

        @Override
        public final Builder availabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZonesListCopier.copy(availabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZones(String... availabilityZones) {
            availabilityZones(Arrays.asList(availabilityZones));
            return this;
        }

        public final void setAvailabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZonesListCopier.copy(availabilityZones);
        }

        public final String getReleaseStatus() {
            return releaseStatus;
        }

        @Override
        public final Builder releaseStatus(String releaseStatus) {
            this.releaseStatus = releaseStatus;
            return this;
        }

        @Override
        public final Builder releaseStatus(ReleaseStatusValues releaseStatus) {
            this.releaseStatus(releaseStatus == null ? null : releaseStatus.toString());
            return this;
        }

        public final void setReleaseStatus(String releaseStatus) {
            this.releaseStatus = releaseStatus;
        }

        @Override
        public OrderableReplicationInstance build() {
            return new OrderableReplicationInstance(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
