/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codepipeline.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Summary information about a pipeline execution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PipelineExecutionSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<PipelineExecutionSummary.Builder, PipelineExecutionSummary> {
    private static final SdkField<String> PIPELINE_EXECUTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("pipelineExecutionId").getter(getter(PipelineExecutionSummary::pipelineExecutionId))
            .setter(setter(Builder::pipelineExecutionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pipelineExecutionId").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(PipelineExecutionSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(PipelineExecutionSummary::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdateTime").getter(getter(PipelineExecutionSummary::lastUpdateTime))
            .setter(setter(Builder::lastUpdateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdateTime").build()).build();

    private static final SdkField<List<SourceRevision>> SOURCE_REVISIONS_FIELD = SdkField
            .<List<SourceRevision>> builder(MarshallingType.LIST)
            .memberName("sourceRevisions")
            .getter(getter(PipelineExecutionSummary::sourceRevisions))
            .setter(setter(Builder::sourceRevisions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceRevisions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SourceRevision> builder(MarshallingType.SDK_POJO)
                                            .constructor(SourceRevision::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ExecutionTrigger> TRIGGER_FIELD = SdkField.<ExecutionTrigger> builder(MarshallingType.SDK_POJO)
            .memberName("trigger").getter(getter(PipelineExecutionSummary::trigger)).setter(setter(Builder::trigger))
            .constructor(ExecutionTrigger::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("trigger").build()).build();

    private static final SdkField<StopExecutionTrigger> STOP_TRIGGER_FIELD = SdkField
            .<StopExecutionTrigger> builder(MarshallingType.SDK_POJO).memberName("stopTrigger")
            .getter(getter(PipelineExecutionSummary::stopTrigger)).setter(setter(Builder::stopTrigger))
            .constructor(StopExecutionTrigger::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stopTrigger").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PIPELINE_EXECUTION_ID_FIELD,
            STATUS_FIELD, START_TIME_FIELD, LAST_UPDATE_TIME_FIELD, SOURCE_REVISIONS_FIELD, TRIGGER_FIELD, STOP_TRIGGER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String pipelineExecutionId;

    private final String status;

    private final Instant startTime;

    private final Instant lastUpdateTime;

    private final List<SourceRevision> sourceRevisions;

    private final ExecutionTrigger trigger;

    private final StopExecutionTrigger stopTrigger;

    private PipelineExecutionSummary(BuilderImpl builder) {
        this.pipelineExecutionId = builder.pipelineExecutionId;
        this.status = builder.status;
        this.startTime = builder.startTime;
        this.lastUpdateTime = builder.lastUpdateTime;
        this.sourceRevisions = builder.sourceRevisions;
        this.trigger = builder.trigger;
        this.stopTrigger = builder.stopTrigger;
    }

    /**
     * <p>
     * The ID of the pipeline execution.
     * </p>
     * 
     * @return The ID of the pipeline execution.
     */
    public final String pipelineExecutionId() {
        return pipelineExecutionId;
    }

    /**
     * <p>
     * The status of the pipeline execution.
     * </p>
     * <ul>
     * <li>
     * <p>
     * InProgress: The pipeline execution is currently running.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stopped: The pipeline execution was manually stopped. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     * >Stopped Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected stop mode,
     * the execution is either completing or abandoning in-progress actions. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     * >Stopped Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Succeeded: The pipeline execution was completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer pipeline
     * execution advanced and continued through the pipeline instead. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
     * Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Failed: The pipeline execution was not completed successfully.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link PipelineExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the pipeline execution.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         InProgress: The pipeline execution is currently running.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stopped: The pipeline execution was manually stopped. For more information, see <a href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     *         >Stopped Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected
     *         stop mode, the execution is either completing or abandoning in-progress actions. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     *         >Stopped Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Succeeded: The pipeline execution was completed successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer
     *         pipeline execution advanced and continued through the pipeline instead. For more information, see <a
     *         href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
     *         Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Failed: The pipeline execution was not completed successfully.
     *         </p>
     *         </li>
     * @see PipelineExecutionStatus
     */
    public final PipelineExecutionStatus status() {
        return PipelineExecutionStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the pipeline execution.
     * </p>
     * <ul>
     * <li>
     * <p>
     * InProgress: The pipeline execution is currently running.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stopped: The pipeline execution was manually stopped. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     * >Stopped Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected stop mode,
     * the execution is either completing or abandoning in-progress actions. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     * >Stopped Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Succeeded: The pipeline execution was completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer pipeline
     * execution advanced and continued through the pipeline instead. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
     * Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Failed: The pipeline execution was not completed successfully.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link PipelineExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the pipeline execution.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         InProgress: The pipeline execution is currently running.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stopped: The pipeline execution was manually stopped. For more information, see <a href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     *         >Stopped Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected
     *         stop mode, the execution is either completing or abandoning in-progress actions. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     *         >Stopped Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Succeeded: The pipeline execution was completed successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer
     *         pipeline execution advanced and continued through the pipeline instead. For more information, see <a
     *         href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
     *         Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Failed: The pipeline execution was not completed successfully.
     *         </p>
     *         </li>
     * @see PipelineExecutionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The date and time when the pipeline execution began, in timestamp format.
     * </p>
     * 
     * @return The date and time when the pipeline execution began, in timestamp format.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The date and time of the last change to the pipeline execution, in timestamp format.
     * </p>
     * 
     * @return The date and time of the last change to the pipeline execution, in timestamp format.
     */
    public final Instant lastUpdateTime() {
        return lastUpdateTime;
    }

    /**
     * Returns true if the SourceRevisions property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasSourceRevisions() {
        return sourceRevisions != null && !(sourceRevisions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the source artifact revisions that initiated a pipeline execution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSourceRevisions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the source artifact revisions that initiated a pipeline execution.
     */
    public final List<SourceRevision> sourceRevisions() {
        return sourceRevisions;
    }

    /**
     * <p>
     * The interaction or event that started a pipeline execution, such as automated change detection or a
     * <code>StartPipelineExecution</code> API call.
     * </p>
     * 
     * @return The interaction or event that started a pipeline execution, such as automated change detection or a
     *         <code>StartPipelineExecution</code> API call.
     */
    public final ExecutionTrigger trigger() {
        return trigger;
    }

    /**
     * <p>
     * The interaction that stopped a pipeline execution.
     * </p>
     * 
     * @return The interaction that stopped a pipeline execution.
     */
    public final StopExecutionTrigger stopTrigger() {
        return stopTrigger;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(pipelineExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceRevisions() ? sourceRevisions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(trigger());
        hashCode = 31 * hashCode + Objects.hashCode(stopTrigger());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PipelineExecutionSummary)) {
            return false;
        }
        PipelineExecutionSummary other = (PipelineExecutionSummary) obj;
        return Objects.equals(pipelineExecutionId(), other.pipelineExecutionId())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(lastUpdateTime(), other.lastUpdateTime()) && hasSourceRevisions() == other.hasSourceRevisions()
                && Objects.equals(sourceRevisions(), other.sourceRevisions()) && Objects.equals(trigger(), other.trigger())
                && Objects.equals(stopTrigger(), other.stopTrigger());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PipelineExecutionSummary").add("PipelineExecutionId", pipelineExecutionId())
                .add("Status", statusAsString()).add("StartTime", startTime()).add("LastUpdateTime", lastUpdateTime())
                .add("SourceRevisions", hasSourceRevisions() ? sourceRevisions() : null).add("Trigger", trigger())
                .add("StopTrigger", stopTrigger()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "pipelineExecutionId":
            return Optional.ofNullable(clazz.cast(pipelineExecutionId()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "lastUpdateTime":
            return Optional.ofNullable(clazz.cast(lastUpdateTime()));
        case "sourceRevisions":
            return Optional.ofNullable(clazz.cast(sourceRevisions()));
        case "trigger":
            return Optional.ofNullable(clazz.cast(trigger()));
        case "stopTrigger":
            return Optional.ofNullable(clazz.cast(stopTrigger()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PipelineExecutionSummary, T> g) {
        return obj -> g.apply((PipelineExecutionSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PipelineExecutionSummary> {
        /**
         * <p>
         * The ID of the pipeline execution.
         * </p>
         * 
         * @param pipelineExecutionId
         *        The ID of the pipeline execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pipelineExecutionId(String pipelineExecutionId);

        /**
         * <p>
         * The status of the pipeline execution.
         * </p>
         * <ul>
         * <li>
         * <p>
         * InProgress: The pipeline execution is currently running.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stopped: The pipeline execution was manually stopped. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         * >Stopped Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected stop
         * mode, the execution is either completing or abandoning in-progress actions. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped">
         * Stopped Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Succeeded: The pipeline execution was completed successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer pipeline
         * execution advanced and continued through the pipeline instead. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
         * Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Failed: The pipeline execution was not completed successfully.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the pipeline execution.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        InProgress: The pipeline execution is currently running.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stopped: The pipeline execution was manually stopped. For more information, see <a href=
         *        "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         *        >Stopped Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected
         *        stop mode, the execution is either completing or abandoning in-progress actions. For more information,
         *        see <a href=
         *        "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         *        >Stopped Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Succeeded: The pipeline execution was completed successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer
         *        pipeline execution advanced and continued through the pipeline instead. For more information, see <a
         *        href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">
         *        Superseded Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Failed: The pipeline execution was not completed successfully.
         *        </p>
         *        </li>
         * @see PipelineExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PipelineExecutionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the pipeline execution.
         * </p>
         * <ul>
         * <li>
         * <p>
         * InProgress: The pipeline execution is currently running.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stopped: The pipeline execution was manually stopped. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         * >Stopped Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected stop
         * mode, the execution is either completing or abandoning in-progress actions. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped">
         * Stopped Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Succeeded: The pipeline execution was completed successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer pipeline
         * execution advanced and continued through the pipeline instead. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
         * Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Failed: The pipeline execution was not completed successfully.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the pipeline execution.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        InProgress: The pipeline execution is currently running.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stopped: The pipeline execution was manually stopped. For more information, see <a href=
         *        "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         *        >Stopped Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected
         *        stop mode, the execution is either completing or abandoning in-progress actions. For more information,
         *        see <a href=
         *        "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         *        >Stopped Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Succeeded: The pipeline execution was completed successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer
         *        pipeline execution advanced and continued through the pipeline instead. For more information, see <a
         *        href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">
         *        Superseded Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Failed: The pipeline execution was not completed successfully.
         *        </p>
         *        </li>
         * @see PipelineExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PipelineExecutionStatus
         */
        Builder status(PipelineExecutionStatus status);

        /**
         * <p>
         * The date and time when the pipeline execution began, in timestamp format.
         * </p>
         * 
         * @param startTime
         *        The date and time when the pipeline execution began, in timestamp format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The date and time of the last change to the pipeline execution, in timestamp format.
         * </p>
         * 
         * @param lastUpdateTime
         *        The date and time of the last change to the pipeline execution, in timestamp format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateTime(Instant lastUpdateTime);

        /**
         * <p>
         * A list of the source artifact revisions that initiated a pipeline execution.
         * </p>
         * 
         * @param sourceRevisions
         *        A list of the source artifact revisions that initiated a pipeline execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRevisions(Collection<SourceRevision> sourceRevisions);

        /**
         * <p>
         * A list of the source artifact revisions that initiated a pipeline execution.
         * </p>
         * 
         * @param sourceRevisions
         *        A list of the source artifact revisions that initiated a pipeline execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRevisions(SourceRevision... sourceRevisions);

        /**
         * <p>
         * A list of the source artifact revisions that initiated a pipeline execution.
         * </p>
         * This is a convenience that creates an instance of the {@link List<SourceRevision>.Builder} avoiding the need
         * to create one manually via {@link List<SourceRevision>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<SourceRevision>.Builder#build()} is called immediately and
         * its result is passed to {@link #sourceRevisions(List<SourceRevision>)}.
         * 
         * @param sourceRevisions
         *        a consumer that will call methods on {@link List<SourceRevision>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceRevisions(List<SourceRevision>)
         */
        Builder sourceRevisions(Consumer<SourceRevision.Builder>... sourceRevisions);

        /**
         * <p>
         * The interaction or event that started a pipeline execution, such as automated change detection or a
         * <code>StartPipelineExecution</code> API call.
         * </p>
         * 
         * @param trigger
         *        The interaction or event that started a pipeline execution, such as automated change detection or a
         *        <code>StartPipelineExecution</code> API call.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trigger(ExecutionTrigger trigger);

        /**
         * <p>
         * The interaction or event that started a pipeline execution, such as automated change detection or a
         * <code>StartPipelineExecution</code> API call.
         * </p>
         * This is a convenience that creates an instance of the {@link ExecutionTrigger.Builder} avoiding the need to
         * create one manually via {@link ExecutionTrigger#builder()}.
         *
         * When the {@link Consumer} completes, {@link ExecutionTrigger.Builder#build()} is called immediately and its
         * result is passed to {@link #trigger(ExecutionTrigger)}.
         * 
         * @param trigger
         *        a consumer that will call methods on {@link ExecutionTrigger.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #trigger(ExecutionTrigger)
         */
        default Builder trigger(Consumer<ExecutionTrigger.Builder> trigger) {
            return trigger(ExecutionTrigger.builder().applyMutation(trigger).build());
        }

        /**
         * <p>
         * The interaction that stopped a pipeline execution.
         * </p>
         * 
         * @param stopTrigger
         *        The interaction that stopped a pipeline execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stopTrigger(StopExecutionTrigger stopTrigger);

        /**
         * <p>
         * The interaction that stopped a pipeline execution.
         * </p>
         * This is a convenience that creates an instance of the {@link StopExecutionTrigger.Builder} avoiding the need
         * to create one manually via {@link StopExecutionTrigger#builder()}.
         *
         * When the {@link Consumer} completes, {@link StopExecutionTrigger.Builder#build()} is called immediately and
         * its result is passed to {@link #stopTrigger(StopExecutionTrigger)}.
         * 
         * @param stopTrigger
         *        a consumer that will call methods on {@link StopExecutionTrigger.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stopTrigger(StopExecutionTrigger)
         */
        default Builder stopTrigger(Consumer<StopExecutionTrigger.Builder> stopTrigger) {
            return stopTrigger(StopExecutionTrigger.builder().applyMutation(stopTrigger).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String pipelineExecutionId;

        private String status;

        private Instant startTime;

        private Instant lastUpdateTime;

        private List<SourceRevision> sourceRevisions = DefaultSdkAutoConstructList.getInstance();

        private ExecutionTrigger trigger;

        private StopExecutionTrigger stopTrigger;

        private BuilderImpl() {
        }

        private BuilderImpl(PipelineExecutionSummary model) {
            pipelineExecutionId(model.pipelineExecutionId);
            status(model.status);
            startTime(model.startTime);
            lastUpdateTime(model.lastUpdateTime);
            sourceRevisions(model.sourceRevisions);
            trigger(model.trigger);
            stopTrigger(model.stopTrigger);
        }

        public final String getPipelineExecutionId() {
            return pipelineExecutionId;
        }

        @Override
        public final Builder pipelineExecutionId(String pipelineExecutionId) {
            this.pipelineExecutionId = pipelineExecutionId;
            return this;
        }

        public final void setPipelineExecutionId(String pipelineExecutionId) {
            this.pipelineExecutionId = pipelineExecutionId;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(PipelineExecutionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getLastUpdateTime() {
            return lastUpdateTime;
        }

        @Override
        public final Builder lastUpdateTime(Instant lastUpdateTime) {
            this.lastUpdateTime = lastUpdateTime;
            return this;
        }

        public final void setLastUpdateTime(Instant lastUpdateTime) {
            this.lastUpdateTime = lastUpdateTime;
        }

        public final List<SourceRevision.Builder> getSourceRevisions() {
            List<SourceRevision.Builder> result = SourceRevisionListCopier.copyToBuilder(this.sourceRevisions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder sourceRevisions(Collection<SourceRevision> sourceRevisions) {
            this.sourceRevisions = SourceRevisionListCopier.copy(sourceRevisions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceRevisions(SourceRevision... sourceRevisions) {
            sourceRevisions(Arrays.asList(sourceRevisions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceRevisions(Consumer<SourceRevision.Builder>... sourceRevisions) {
            sourceRevisions(Stream.of(sourceRevisions).map(c -> SourceRevision.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSourceRevisions(Collection<SourceRevision.BuilderImpl> sourceRevisions) {
            this.sourceRevisions = SourceRevisionListCopier.copyFromBuilder(sourceRevisions);
        }

        public final ExecutionTrigger.Builder getTrigger() {
            return trigger != null ? trigger.toBuilder() : null;
        }

        @Override
        public final Builder trigger(ExecutionTrigger trigger) {
            this.trigger = trigger;
            return this;
        }

        public final void setTrigger(ExecutionTrigger.BuilderImpl trigger) {
            this.trigger = trigger != null ? trigger.build() : null;
        }

        public final StopExecutionTrigger.Builder getStopTrigger() {
            return stopTrigger != null ? stopTrigger.toBuilder() : null;
        }

        @Override
        public final Builder stopTrigger(StopExecutionTrigger stopTrigger) {
            this.stopTrigger = stopTrigger;
            return this;
        }

        public final void setStopTrigger(StopExecutionTrigger.BuilderImpl stopTrigger) {
            this.stopTrigger = stopTrigger != null ? stopTrigger.build() : null;
        }

        @Override
        public PipelineExecutionSummary build() {
            return new PipelineExecutionSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
