/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudtrail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Amazon S3 buckets or AWS Lambda functions that you specify in your event selectors for your trail to log data
 * events. Data events provide information about the resource operations performed on or within a resource itself. These
 * are also known as data plane operations. You can specify up to 250 data resources for a trail.
 * </p>
 * <note>
 * <p>
 * The total number of allowed data resources is 250. This number can be distributed between 1 and 5 event selectors,
 * but the total cannot exceed 250 across all selectors.
 * </p>
 * <p>
 * If you are using advanced event selectors, the maximum total number of values for all conditions, across all advanced
 * event selectors for the trail, is 500.
 * </p>
 * </note>
 * <p>
 * The following example demonstrates how logging works when you configure logging of all data events for an S3 bucket
 * named <code>bucket-1</code>. In this example, the CloudTrail user specified an empty prefix, and the option to log
 * both <code>Read</code> and <code>Write</code> data events.
 * </p>
 * <ol>
 * <li>
 * <p>
 * A user uploads an image file to <code>bucket-1</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * The <code>PutObject</code> API operation is an Amazon S3 object-level API. It is recorded as a data event in
 * CloudTrail. Because the CloudTrail user specified an S3 bucket with an empty prefix, events that occur on any object
 * in that bucket are logged. The trail processes and logs the event.
 * </p>
 * </li>
 * <li>
 * <p>
 * A user uploads an object to an Amazon S3 bucket named <code>arn:aws:s3:::bucket-2</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * The <code>PutObject</code> API operation occurred for an object in an S3 bucket that the CloudTrail user didn't
 * specify for the trail. The trail doesn’t log the event.
 * </p>
 * </li>
 * </ol>
 * <p>
 * The following example demonstrates how logging works when you configure logging of AWS Lambda data events for a
 * Lambda function named <i>MyLambdaFunction</i>, but not for all AWS Lambda functions.
 * </p>
 * <ol>
 * <li>
 * <p>
 * A user runs a script that includes a call to the <i>MyLambdaFunction</i> function and the
 * <i>MyOtherLambdaFunction</i> function.
 * </p>
 * </li>
 * <li>
 * <p>
 * The <code>Invoke</code> API operation on <i>MyLambdaFunction</i> is an AWS Lambda API. It is recorded as a data event
 * in CloudTrail. Because the CloudTrail user specified logging data events for <i>MyLambdaFunction</i>, any invocations
 * of that function are logged. The trail processes and logs the event.
 * </p>
 * </li>
 * <li>
 * <p>
 * The <code>Invoke</code> API operation on <i>MyOtherLambdaFunction</i> is an AWS Lambda API. Because the CloudTrail
 * user did not specify logging data events for all Lambda functions, the <code>Invoke</code> operation for
 * <i>MyOtherLambdaFunction</i> does not match the function specified for the trail. The trail doesn’t log the event.
 * </p>
 * </li>
 * </ol>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataResource implements SdkPojo, Serializable, ToCopyableBuilder<DataResource.Builder, DataResource> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(DataResource::type)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<List<String>> VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Values")
            .getter(getter(DataResource::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, VALUES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final List<String> values;

    private DataResource(BuilderImpl builder) {
        this.type = builder.type;
        this.values = builder.values;
    }

    /**
     * <p>
     * The resource type in which you want to log data events. You can specify <code>AWS::S3::Object</code> or
     * <code>AWS::Lambda::Function</code> resources.
     * </p>
     * <p>
     * The <code>AWS::S3Outposts::Object</code> resource type is not valid in basic event selectors. To log data events
     * on this resource type, use advanced event selectors.
     * </p>
     * 
     * @return The resource type in which you want to log data events. You can specify <code>AWS::S3::Object</code> or
     *         <code>AWS::Lambda::Function</code> resources.</p>
     *         <p>
     *         The <code>AWS::S3Outposts::Object</code> resource type is not valid in basic event selectors. To log data
     *         events on this resource type, use advanced event selectors.
     */
    public final String type() {
        return type;
    }

    /**
     * Returns true if the Values property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of Amazon Resource Name (ARN) strings or partial ARN strings for the specified objects.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To log data events for all objects in all S3 buckets in your AWS account, specify the prefix as
     * <code>arn:aws:s3:::</code>.
     * </p>
     * <note>
     * <p>
     * This will also enable logging of data event activity performed by any user or role in your AWS account, even if
     * that activity is performed on a bucket that belongs to another AWS account.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * To log data events for all objects in an S3 bucket, specify the bucket and an empty object prefix such as
     * <code>arn:aws:s3:::bucket-1/</code>. The trail logs data events for all objects in this S3 bucket.
     * </p>
     * </li>
     * <li>
     * <p>
     * To log data events for specific objects, specify the S3 bucket and object prefix such as
     * <code>arn:aws:s3:::bucket-1/example-images</code>. The trail logs data events for objects in this S3 bucket that
     * match the prefix.
     * </p>
     * </li>
     * <li>
     * <p>
     * To log data events for all functions in your AWS account, specify the prefix as <code>arn:aws:lambda</code>.
     * </p>
     * <note>
     * <p>
     * This will also enable logging of <code>Invoke</code> activity performed by any user or role in your AWS account,
     * even if that activity is performed on a function that belongs to another AWS account.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * To log data events for a specific Lambda function, specify the function ARN.
     * </p>
     * <note>
     * <p>
     * Lambda function ARNs are exact. For example, if you specify a function ARN
     * <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld</i>, data events will only be logged for
     * <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld</i>. They will not be logged for
     * <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld2</i>.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of Amazon Resource Name (ARN) strings or partial ARN strings for the specified objects.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To log data events for all objects in all S3 buckets in your AWS account, specify the prefix as
     *         <code>arn:aws:s3:::</code>.
     *         </p>
     *         <note>
     *         <p>
     *         This will also enable logging of data event activity performed by any user or role in your AWS account,
     *         even if that activity is performed on a bucket that belongs to another AWS account.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         To log data events for all objects in an S3 bucket, specify the bucket and an empty object prefix such as
     *         <code>arn:aws:s3:::bucket-1/</code>. The trail logs data events for all objects in this S3 bucket.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To log data events for specific objects, specify the S3 bucket and object prefix such as
     *         <code>arn:aws:s3:::bucket-1/example-images</code>. The trail logs data events for objects in this S3
     *         bucket that match the prefix.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To log data events for all functions in your AWS account, specify the prefix as
     *         <code>arn:aws:lambda</code>.
     *         </p>
     *         <note>
     *         <p>
     *         This will also enable logging of <code>Invoke</code> activity performed by any user or role in your AWS
     *         account, even if that activity is performed on a function that belongs to another AWS account.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         To log data events for a specific Lambda function, specify the function ARN.
     *         </p>
     *         <note>
     *         <p>
     *         Lambda function ARNs are exact. For example, if you specify a function ARN
     *         <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld</i>, data events will only be logged for
     *         <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld</i>. They will not be logged for
     *         <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld2</i>.
     *         </p>
     *         </note></li>
     */
    public final List<String> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(type());
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataResource)) {
            return false;
        }
        DataResource other = (DataResource) obj;
        return Objects.equals(type(), other.type()) && hasValues() == other.hasValues()
                && Objects.equals(values(), other.values());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataResource").add("Type", type()).add("Values", hasValues() ? values() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(type()));
        case "Values":
            return Optional.ofNullable(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DataResource, T> g) {
        return obj -> g.apply((DataResource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataResource> {
        /**
         * <p>
         * The resource type in which you want to log data events. You can specify <code>AWS::S3::Object</code> or
         * <code>AWS::Lambda::Function</code> resources.
         * </p>
         * <p>
         * The <code>AWS::S3Outposts::Object</code> resource type is not valid in basic event selectors. To log data
         * events on this resource type, use advanced event selectors.
         * </p>
         * 
         * @param type
         *        The resource type in which you want to log data events. You can specify <code>AWS::S3::Object</code>
         *        or <code>AWS::Lambda::Function</code> resources.</p>
         *        <p>
         *        The <code>AWS::S3Outposts::Object</code> resource type is not valid in basic event selectors. To log
         *        data events on this resource type, use advanced event selectors.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder type(String type);

        /**
         * <p>
         * An array of Amazon Resource Name (ARN) strings or partial ARN strings for the specified objects.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To log data events for all objects in all S3 buckets in your AWS account, specify the prefix as
         * <code>arn:aws:s3:::</code>.
         * </p>
         * <note>
         * <p>
         * This will also enable logging of data event activity performed by any user or role in your AWS account, even
         * if that activity is performed on a bucket that belongs to another AWS account.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * To log data events for all objects in an S3 bucket, specify the bucket and an empty object prefix such as
         * <code>arn:aws:s3:::bucket-1/</code>. The trail logs data events for all objects in this S3 bucket.
         * </p>
         * </li>
         * <li>
         * <p>
         * To log data events for specific objects, specify the S3 bucket and object prefix such as
         * <code>arn:aws:s3:::bucket-1/example-images</code>. The trail logs data events for objects in this S3 bucket
         * that match the prefix.
         * </p>
         * </li>
         * <li>
         * <p>
         * To log data events for all functions in your AWS account, specify the prefix as <code>arn:aws:lambda</code>.
         * </p>
         * <note>
         * <p>
         * This will also enable logging of <code>Invoke</code> activity performed by any user or role in your AWS
         * account, even if that activity is performed on a function that belongs to another AWS account.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * To log data events for a specific Lambda function, specify the function ARN.
         * </p>
         * <note>
         * <p>
         * Lambda function ARNs are exact. For example, if you specify a function ARN
         * <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld</i>, data events will only be logged for
         * <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld</i>. They will not be logged for
         * <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld2</i>.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param values
         *        An array of Amazon Resource Name (ARN) strings or partial ARN strings for the specified objects.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To log data events for all objects in all S3 buckets in your AWS account, specify the prefix as
         *        <code>arn:aws:s3:::</code>.
         *        </p>
         *        <note>
         *        <p>
         *        This will also enable logging of data event activity performed by any user or role in your AWS
         *        account, even if that activity is performed on a bucket that belongs to another AWS account.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        To log data events for all objects in an S3 bucket, specify the bucket and an empty object prefix such
         *        as <code>arn:aws:s3:::bucket-1/</code>. The trail logs data events for all objects in this S3 bucket.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To log data events for specific objects, specify the S3 bucket and object prefix such as
         *        <code>arn:aws:s3:::bucket-1/example-images</code>. The trail logs data events for objects in this S3
         *        bucket that match the prefix.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To log data events for all functions in your AWS account, specify the prefix as
         *        <code>arn:aws:lambda</code>.
         *        </p>
         *        <note>
         *        <p>
         *        This will also enable logging of <code>Invoke</code> activity performed by any user or role in your
         *        AWS account, even if that activity is performed on a function that belongs to another AWS account.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        To log data events for a specific Lambda function, specify the function ARN.
         *        </p>
         *        <note>
         *        <p>
         *        Lambda function ARNs are exact. For example, if you specify a function ARN
         *        <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld</i>, data events will only be logged for
         *        <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld</i>. They will not be logged for
         *        <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld2</i>.
         *        </p>
         *        </note></li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * An array of Amazon Resource Name (ARN) strings or partial ARN strings for the specified objects.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To log data events for all objects in all S3 buckets in your AWS account, specify the prefix as
         * <code>arn:aws:s3:::</code>.
         * </p>
         * <note>
         * <p>
         * This will also enable logging of data event activity performed by any user or role in your AWS account, even
         * if that activity is performed on a bucket that belongs to another AWS account.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * To log data events for all objects in an S3 bucket, specify the bucket and an empty object prefix such as
         * <code>arn:aws:s3:::bucket-1/</code>. The trail logs data events for all objects in this S3 bucket.
         * </p>
         * </li>
         * <li>
         * <p>
         * To log data events for specific objects, specify the S3 bucket and object prefix such as
         * <code>arn:aws:s3:::bucket-1/example-images</code>. The trail logs data events for objects in this S3 bucket
         * that match the prefix.
         * </p>
         * </li>
         * <li>
         * <p>
         * To log data events for all functions in your AWS account, specify the prefix as <code>arn:aws:lambda</code>.
         * </p>
         * <note>
         * <p>
         * This will also enable logging of <code>Invoke</code> activity performed by any user or role in your AWS
         * account, even if that activity is performed on a function that belongs to another AWS account.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * To log data events for a specific Lambda function, specify the function ARN.
         * </p>
         * <note>
         * <p>
         * Lambda function ARNs are exact. For example, if you specify a function ARN
         * <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld</i>, data events will only be logged for
         * <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld</i>. They will not be logged for
         * <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld2</i>.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param values
         *        An array of Amazon Resource Name (ARN) strings or partial ARN strings for the specified objects.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To log data events for all objects in all S3 buckets in your AWS account, specify the prefix as
         *        <code>arn:aws:s3:::</code>.
         *        </p>
         *        <note>
         *        <p>
         *        This will also enable logging of data event activity performed by any user or role in your AWS
         *        account, even if that activity is performed on a bucket that belongs to another AWS account.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        To log data events for all objects in an S3 bucket, specify the bucket and an empty object prefix such
         *        as <code>arn:aws:s3:::bucket-1/</code>. The trail logs data events for all objects in this S3 bucket.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To log data events for specific objects, specify the S3 bucket and object prefix such as
         *        <code>arn:aws:s3:::bucket-1/example-images</code>. The trail logs data events for objects in this S3
         *        bucket that match the prefix.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To log data events for all functions in your AWS account, specify the prefix as
         *        <code>arn:aws:lambda</code>.
         *        </p>
         *        <note>
         *        <p>
         *        This will also enable logging of <code>Invoke</code> activity performed by any user or role in your
         *        AWS account, even if that activity is performed on a function that belongs to another AWS account.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        To log data events for a specific Lambda function, specify the function ARN.
         *        </p>
         *        <note>
         *        <p>
         *        Lambda function ARNs are exact. For example, if you specify a function ARN
         *        <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld</i>, data events will only be logged for
         *        <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld</i>. They will not be logged for
         *        <i>arn:aws:lambda:us-west-2:111111111111:function:helloworld2</i>.
         *        </p>
         *        </note></li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private List<String> values = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DataResource model) {
            type(model.type);
            values(model.values);
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final Collection<String> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = DataResourceValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        public final void setValues(Collection<String> values) {
            this.values = DataResourceValuesCopier.copy(values);
        }

        @Override
        public DataResource build() {
            return new DataResource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
