/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.configuration;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.configuration.SettingMigrator;
import org.neo4j.configuration.SettingValueParsers;
import org.neo4j.configuration.connectors.BoltConnector;
import org.neo4j.configuration.connectors.HttpConnector;
import org.neo4j.configuration.connectors.HttpsConnector;
import org.neo4j.configuration.helpers.SocketAddress;
import org.neo4j.configuration.ssl.SslPolicyConfig;
import org.neo4j.configuration.ssl.SslPolicyScope;
import org.neo4j.graphdb.config.Setting;
import org.neo4j.logging.Log;
import org.neo4j.values.storable.CoordinateReferenceSystem;

public final class SettingMigrators {
    private SettingMigrators() {
    }

    public static void migrateAdvertisedAddressInheritanceChange(Map<String, String> values, Map<String, String> defaultValues, Log log, String listenAddress, String advertisedAddress) {
        String listenValue = values.get(listenAddress);
        if (StringUtils.isNotBlank((CharSequence)listenValue)) {
            String advertisedValue = values.get(advertisedAddress);
            boolean advertisedAlreadyHasPort = false;
            try {
                if (StringUtils.isNotBlank((CharSequence)advertisedValue)) {
                    advertisedAlreadyHasPort = SettingValueParsers.SOCKET_ADDRESS.parse(advertisedValue).getPort() >= 0;
                }
            }
            catch (RuntimeException runtimeException) {
                // empty catch block
            }
            if (!advertisedAlreadyHasPort) {
                try {
                    int port = SettingValueParsers.SOCKET_ADDRESS.parse(listenValue).getPort();
                    if (port >= 0) {
                        SocketAddress newAdvertised = new SocketAddress(advertisedValue, port);
                        String msg = "Note that since you did not explicitly set the port in %s Neo4j automatically set it to %s to match %s. This behavior may change in the future and we recommend you to explicitly set it.";
                        if (StringUtils.isNotBlank((CharSequence)advertisedValue)) {
                            log.warn(msg, new Object[]{advertisedAddress, port, listenAddress});
                        } else {
                            log.info(msg, new Object[]{advertisedAddress, port, listenAddress});
                        }
                        defaultValues.put(advertisedAddress, newAdvertised.toString());
                    }
                }
                catch (RuntimeException runtimeException) {
                    // empty catch block
                }
            }
        }
    }

    public static void migrateSettingNameChange(Map<String, String> values, Log log, String oldSetting, Setting<?> newSetting) {
        String value = values.remove(oldSetting);
        if (StringUtils.isNotBlank((CharSequence)value)) {
            log.warn("Use of deprecated setting %s. It is replaced by %s", new Object[]{oldSetting, newSetting.name()});
            values.putIfAbsent(newSetting.name(), value);
        }
    }

    public static class DatababaseMigrator
    implements SettingMigrator {
        @Override
        public void migrate(Map<String, String> values, Map<String, String> defaultValues, Log log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.transaction.datababase_max_size", GraphDatabaseSettings.memory_transaction_database_max_size);
        }
    }

    public static class WhitelistSettingsMigrator
    implements SettingMigrator {
        @Override
        public void migrate(Map<String, String> values, Map<String, String> defaultValues, Log log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.memory.pagecache.warmup.preload.whitelist", GraphDatabaseSettings.pagecache_warmup_prefetch_allowlist);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.security.procedures.whitelist", GraphDatabaseSettings.procedure_allowlist);
        }
    }

    public static class DatabaseMemoryMigrator
    implements SettingMigrator {
        @Override
        public void migrate(Map<String, String> values, Map<String, String> defaultValues, Log log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.tx_state.max_off_heap_memory", GraphDatabaseSettings.tx_state_max_off_heap_memory);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.tx_state.off_heap.max_cacheable_block_size", GraphDatabaseSettings.tx_state_off_heap_max_cacheable_block_size);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.tx_state.off_heap.block_cache_size", GraphDatabaseSettings.tx_state_off_heap_block_cache_size);
            String maxAllocations = values.remove("cypher.query_max_allocations");
            if (StringUtils.isNotBlank((CharSequence)maxAllocations)) {
                if (!values.containsKey(GraphDatabaseSettings.memory_transaction_max_size.name())) {
                    log.warn("The setting cypher.query_max_allocations is removed and replaced by %s.", new Object[]{GraphDatabaseSettings.memory_transaction_max_size.name()});
                    values.put(GraphDatabaseSettings.memory_transaction_max_size.name(), maxAllocations);
                } else {
                    log.warn("The setting cypher.query_max_allocations is removed and replaced by %s. Since both are set, %s will take precedence and the value of cypher.query_max_allocations, %s, will be ignored.", new Object[]{GraphDatabaseSettings.memory_transaction_max_size.name(), GraphDatabaseSettings.memory_transaction_max_size.name(), maxAllocations});
                }
            }
        }
    }

    public static class QueryLoggerMigrator
    implements SettingMigrator {
        private static final String deprecationMessage = "Use of deprecated setting value %s=%s. It is replaced by %s=%s";
        private static final String settingName = GraphDatabaseSettings.log_queries.name();

        @Override
        public void migrate(Map<String, String> values, Map<String, String> defaultValues, Log log) {
            String value = values.get(settingName);
            if ("true".equalsIgnoreCase(value)) {
                log.warn(deprecationMessage, new Object[]{settingName, value, settingName, GraphDatabaseSettings.LogQueryLevel.INFO.name()});
                values.put(settingName, GraphDatabaseSettings.LogQueryLevel.INFO.name());
            } else if ("false".equalsIgnoreCase(value)) {
                log.warn(deprecationMessage, new Object[]{settingName, value, settingName, GraphDatabaseSettings.LogQueryLevel.OFF.name()});
                values.put(settingName, GraphDatabaseSettings.LogQueryLevel.OFF.name());
            }
        }
    }

    public static class MultiThreadedSchemaIndexPopulationEnabledMigrator
    implements SettingMigrator {
        private static final String settingName = "unsupported.dbms.multi_threaded_schema_index_population_enabled";

        @Override
        public void migrate(Map<String, String> values, Map<String, String> defaultValues, Log log) {
            if (values.containsKey(settingName)) {
                log.warn("Setting %s is removed. It's no longer possible to disable multi-threaded index population.", new Object[]{settingName});
                values.remove(settingName);
            }
        }
    }

    public static class KillQueryVerboseMigrator
    implements SettingMigrator {
        private static final String settingName = "dbms.procedures.kill_query_verbose";

        @Override
        public void migrate(Map<String, String> values, Map<String, String> defaultValues, Log log) {
            if (values.containsKey(settingName)) {
                log.warn("Setting %s is removed. It's no longer possible to disable verbose kill query logging.", new Object[]{settingName});
                values.remove(settingName);
            }
        }
    }

    public static class AllowKeyGenerationMigrator
    implements SettingMigrator {
        private static final Pattern pattern = Pattern.compile("^dbms\\.ssl\\.policy\\.([^.]+)\\.allow_key_generation$");

        @Override
        public void migrate(Map<String, String> values, Map<String, String> defaultValues, Log log) {
            HashSet<String> toRemove = new HashSet<String>();
            for (String setting : values.keySet()) {
                Matcher matcher = pattern.matcher(setting);
                if (!matcher.find()) continue;
                log.warn("Setting %s is removed. A valid key and certificate are required to be present in the key and certificate path configured in this ssl policy.", new Object[]{setting});
                toRemove.add(setting);
            }
            values.keySet().removeAll(toRemove);
        }
    }

    public static class SslPolicyMigrator
    implements SettingMigrator {
        private static final Pattern pattern = Pattern.compile("^(dbms\\.ssl\\.policy\\.)([^.]+)(\\.[^.]+)$");
        private static final Map<String, SslPolicyScope> settingScopeMap = Map.of("bolt.ssl_policy", SslPolicyScope.BOLT, "https.ssl_policy", SslPolicyScope.HTTPS, "dbms.backup.ssl_policy", SslPolicyScope.BACKUP, "causal_clustering.ssl_policy", SslPolicyScope.CLUSTER);
        private static final List<String> legacySettings = List.of("dbms.directories.certificates", "unsupported.dbms.security.tls_certificate_file", "unsupported.dbms.security.tls_key_file");

        @Override
        public void migrate(Map<String, String> values, Map<String, String> defaultValues, Log log) {
            SslPolicyMigrator.migratePolicies(values, log);
            SslPolicyMigrator.warnUseOfLegacyPolicy(values, log);
        }

        private static void migratePolicies(Map<String, String> values, Log log) {
            HashMap<String, String> valueCopy = new HashMap<String, String>(values);
            HashMap oldNameToScope = new HashMap();
            valueCopy.forEach((setting, value) -> {
                if (settingScopeMap.containsKey(setting)) {
                    log.warn("Use of deprecated setting %s.", new Object[]{setting});
                    SslPolicyScope scope = settingScopeMap.get(setting);
                    oldNameToScope.put(value, scope);
                    values.put(SslPolicyConfig.forScope((SslPolicyScope)scope).enabled.name(), Boolean.TRUE.toString());
                    values.remove(setting);
                }
            });
            valueCopy.forEach((setting, value) -> {
                String newGroupName;
                String groupName;
                Matcher matcher = pattern.matcher((CharSequence)setting);
                if (matcher.find() && oldNameToScope.containsKey(groupName = matcher.group(2)) && !Objects.equals(groupName, newGroupName = ((SslPolicyScope)((Object)((Object)oldNameToScope.get(groupName)))).name().toLowerCase())) {
                    String prefix = matcher.group(1);
                    String suffix = matcher.group(3);
                    String newSetting = prefix + newGroupName + suffix;
                    log.warn("Use of deprecated setting %s. It is replaced by %s", new Object[]{setting, newSetting});
                    values.remove(setting);
                    values.put(newSetting, (String)value);
                }
            });
        }

        private static void warnUseOfLegacyPolicy(Map<String, String> values, Log log) {
            for (String legacySetting : legacySettings) {
                if (values.remove(legacySetting) == null) continue;
                log.warn("Use of deprecated setting %s. Legacy ssl policy is no longer supported.", new Object[]{legacySetting});
            }
        }
    }

    public static class DefaultAddressMigrator
    implements SettingMigrator {
        @Override
        public void migrate(Map<String, String> values, Map<String, String> defaultValues, Log log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.connectors.default_listen_address", GraphDatabaseSettings.default_listen_address);
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.connectors.default_advertised_address", GraphDatabaseSettings.default_advertised_address);
        }
    }

    public static class ConnectorMigrator
    implements SettingMigrator {
        private static final Pattern oldConnector = Pattern.compile("^dbms\\.connector\\.([^.]+)\\.([^.]+)$");
        private static final String ANY_CONNECTOR = "bolt|http|https";

        @Override
        public void migrate(Map<String, String> values, Map<String, String> defaultValues, Log log) {
            ConnectorMigrator.migrateOldConnectors(values, log);
            ConnectorMigrator.migrateConnectorAddresses(values, defaultValues, log);
        }

        private static void migrateOldConnectors(Map<String, String> values, Log log) {
            HashMap<String, Matcher> oldConnectors = new HashMap<String, Matcher>();
            values.forEach((setting, value) -> {
                Matcher matcher = oldConnector.matcher((CharSequence)setting);
                if (matcher.find()) {
                    oldConnectors.put((String)setting, matcher);
                }
            });
            oldConnectors.forEach((setting, matcher) -> {
                String settingName = matcher.group(2);
                String id = matcher.group(1);
                if (id.matches(ANY_CONNECTOR)) {
                    if (Objects.equals("type", settingName)) {
                        values.remove(setting);
                        log.warn("Use of deprecated setting %s. Type is no longer required", new Object[]{setting});
                    }
                } else {
                    values.remove(setting);
                    log.warn("Use of deprecated setting %s. No longer supports multiple connectors. Setting discarded.", new Object[]{setting});
                }
            });
        }

        private static void migrateConnectorAddresses(Map<String, String> values, Map<String, String> defValues, Log log) {
            SettingMigrators.migrateAdvertisedAddressInheritanceChange(values, defValues, log, BoltConnector.listen_address.name(), BoltConnector.advertised_address.name());
            SettingMigrators.migrateAdvertisedAddressInheritanceChange(values, defValues, log, HttpConnector.listen_address.name(), HttpConnector.advertised_address.name());
            SettingMigrators.migrateAdvertisedAddressInheritanceChange(values, defValues, log, HttpsConnector.listen_address.name(), HttpsConnector.advertised_address.name());
        }
    }

    public static class CrsConfigMigrator
    implements SettingMigrator {
        private static final String PREFIX = "unsupported.dbms.db.spatial.crs";
        private static final Pattern oldConnector = Pattern.compile("^unsupported\\.dbms\\.db\\.spatial\\.crs\\.([^.]+)\\.(min|max)\\.([xyz])$");

        @Override
        public void migrate(Map<String, String> values, Map<String, String> defaultValues, Log log) {
            ArrayList oldCrs = new ArrayList();
            HashMap<String, List> crsValues = new HashMap<String, List>();
            values.forEach((setting, value) -> {
                Matcher matcher = oldConnector.matcher((CharSequence)setting);
                if (matcher.find()) {
                    String crsName = matcher.group(1);
                    String crsPlusSetting = String.format("%s.%s", crsName, matcher.group(2));
                    CoordinateReferenceSystem crs = CoordinateReferenceSystem.byName((String)crsName);
                    List valueList = crsValues.computeIfAbsent(crsPlusSetting, s -> new ArrayList<String>(Collections.nCopies(crs.getDimension(), Double.toString(Double.NaN))));
                    valueList.set(matcher.group(3).charAt(0) - 120, value);
                    oldCrs.add(setting);
                }
            });
            oldCrs.forEach(setting -> {
                values.remove(setting);
                log.warn("Use of deprecated setting %s.", new Object[]{setting});
            });
            crsValues.forEach((name, valueList) -> {
                String setting = String.format("%s.%s", PREFIX, name);
                String value = String.join((CharSequence)",", valueList);
                values.putIfAbsent(setting, value);
                log.warn("Settings migrated to %s = %s", new Object[]{setting, value});
            });
        }
    }

    public static class DefaultSchemaProviderMigrator
    implements SettingMigrator {
        private static final String INDEX_PROVIDER_SETTING = "dbms.index.default_schema_provider";

        @Override
        public void migrate(Map<String, String> values, Map<String, String> defaultValues, Log log) {
            String oldSchemaProvider = values.remove(INDEX_PROVIDER_SETTING);
            if (StringUtils.isBlank((CharSequence)oldSchemaProvider)) {
                return;
            }
            String newSchemaProvider = oldSchemaProvider;
            boolean migratedValue = false;
            switch (oldSchemaProvider) {
                case "lucene-1.0": 
                case "lucene+native-1.0": {
                    newSchemaProvider = "lucene+native-3.0";
                    migratedValue = true;
                    break;
                }
                case "lucene+native-2.0": {
                    newSchemaProvider = "native-btree-1.0";
                    migratedValue = true;
                    break;
                }
            }
            Object warning = "Use of deprecated setting dbms.index.default_schema_provider.";
            if (migratedValue) {
                warning = (String)warning + " Value migrated from " + oldSchemaProvider + " to " + newSchemaProvider + ".";
            }
            log.warn((String)warning);
            values.put(INDEX_PROVIDER_SETTING, newSchemaProvider);
        }
    }

    public static class ActiveDatabaseMigrator
    implements SettingMigrator {
        @Override
        public void migrate(Map<String, String> values, Map<String, String> defaultValues, Log log) {
            SettingMigrators.migrateSettingNameChange(values, log, "dbms.active_database", GraphDatabaseSettings.default_database);
        }
    }
}

