/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.sdk.server;

import com.google.common.base.Objects;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.math.DoubleMath;
import com.google.common.primitives.Bytes;
import java.math.RoundingMode;
import java.nio.ByteBuffer;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.eclipse.milo.opcua.sdk.server.OpcUaServer;
import org.eclipse.milo.opcua.sdk.server.SecurityConfiguration;
import org.eclipse.milo.opcua.sdk.server.Session;
import org.eclipse.milo.opcua.sdk.server.identity.IdentityValidator;
import org.eclipse.milo.opcua.sdk.server.services.ServiceAttributes;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.UaRuntimeException;
import org.eclipse.milo.opcua.stack.core.security.CertificateValidator;
import org.eclipse.milo.opcua.stack.core.security.SecurityAlgorithm;
import org.eclipse.milo.opcua.stack.core.security.SecurityPolicy;
import org.eclipse.milo.opcua.stack.core.serialization.UaResponseMessage;
import org.eclipse.milo.opcua.stack.core.types.builtin.ByteString;
import org.eclipse.milo.opcua.stack.core.types.builtin.DiagnosticInfo;
import org.eclipse.milo.opcua.stack.core.types.builtin.ExtensionObject;
import org.eclipse.milo.opcua.stack.core.types.builtin.NodeId;
import org.eclipse.milo.opcua.stack.core.types.builtin.StatusCode;
import org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.Unsigned;
import org.eclipse.milo.opcua.stack.core.types.enumerated.MessageSecurityMode;
import org.eclipse.milo.opcua.stack.core.types.enumerated.UserTokenType;
import org.eclipse.milo.opcua.stack.core.types.structured.ActivateSessionRequest;
import org.eclipse.milo.opcua.stack.core.types.structured.ActivateSessionResponse;
import org.eclipse.milo.opcua.stack.core.types.structured.AnonymousIdentityToken;
import org.eclipse.milo.opcua.stack.core.types.structured.CreateSessionRequest;
import org.eclipse.milo.opcua.stack.core.types.structured.CreateSessionResponse;
import org.eclipse.milo.opcua.stack.core.types.structured.EndpointDescription;
import org.eclipse.milo.opcua.stack.core.types.structured.SignatureData;
import org.eclipse.milo.opcua.stack.core.types.structured.SignedSoftwareCertificate;
import org.eclipse.milo.opcua.stack.core.types.structured.UserIdentityToken;
import org.eclipse.milo.opcua.stack.core.types.structured.UserTokenPolicy;
import org.eclipse.milo.opcua.stack.core.util.CertificateUtil;
import org.eclipse.milo.opcua.stack.core.util.CertificateValidationUtil;
import org.eclipse.milo.opcua.stack.core.util.ConversionUtil;
import org.eclipse.milo.opcua.stack.core.util.DigestUtil;
import org.eclipse.milo.opcua.stack.core.util.EndpointUtil;
import org.eclipse.milo.opcua.stack.core.util.NonceUtil;
import org.eclipse.milo.opcua.stack.core.util.SignatureUtil;
import org.eclipse.milo.opcua.stack.server.services.AttributeHistoryServiceSet;
import org.eclipse.milo.opcua.stack.server.services.AttributeServiceSet;
import org.eclipse.milo.opcua.stack.server.services.MethodServiceSet;
import org.eclipse.milo.opcua.stack.server.services.MonitoredItemServiceSet;
import org.eclipse.milo.opcua.stack.server.services.NodeManagementServiceSet;
import org.eclipse.milo.opcua.stack.server.services.QueryServiceSet;
import org.eclipse.milo.opcua.stack.server.services.ServiceRequest;
import org.eclipse.milo.opcua.stack.server.services.SessionServiceSet;
import org.eclipse.milo.opcua.stack.server.services.SubscriptionServiceSet;
import org.eclipse.milo.opcua.stack.server.services.ViewServiceSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionManager
implements AttributeServiceSet,
AttributeHistoryServiceSet,
MethodServiceSet,
MonitoredItemServiceSet,
NodeManagementServiceSet,
QueryServiceSet,
SessionServiceSet,
SubscriptionServiceSet,
ViewServiceSet {
    private static final int MAX_SESSION_TIMEOUT_MS = 120000;
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final Map<NodeId, Session> createdSessions = Maps.newConcurrentMap();
    private final Map<NodeId, Session> activeSessions = Maps.newConcurrentMap();
    private final Map<NodeId, Session> inactiveSessions = Maps.newConcurrentMap();
    private final List<ByteString> clientNonces = Lists.newCopyOnWriteArrayList();
    private final OpcUaServer server;

    public SessionManager(OpcUaServer server) {
        this.server = server;
    }

    public List<Session> getActiveSessions() {
        return Lists.newArrayList(this.activeSessions.values());
    }

    public List<Session> getInactiveSessions() {
        return Lists.newArrayList(this.inactiveSessions.values());
    }

    public void killSession(NodeId nodeId, boolean deleteSubscriptions) {
        this.activeSessions.values().stream().filter(s -> s.getSessionId().equals((Object)nodeId)).findFirst().ifPresent(s -> s.close(deleteSubscriptions));
    }

    private Session session(ServiceRequest service) throws UaException {
        long secureChannelId = service.getSecureChannelId();
        NodeId authToken = service.getRequest().getRequestHeader().getAuthenticationToken();
        Session session = this.activeSessions.get(authToken);
        if (session == null) {
            session = this.createdSessions.remove(authToken);
            if (session == null) {
                throw new UaException(2149908480L);
            }
            if (session.getSecureChannelId() != secureChannelId) {
                this.createdSessions.put(authToken, session);
                throw new UaException(2148728832L);
            }
            throw new UaException(2150039552L);
        }
        if (session.getSecureChannelId() != secureChannelId) {
            throw new UaException(0x80220000L);
        }
        session.updateLastActivity();
        service.attr(ServiceAttributes.SERVER_KEY).set((Object)this.server);
        service.attr(ServiceAttributes.SESSION_KEY).set((Object)session);
        return session;
    }

    public void onCreateSession(ServiceRequest serviceRequest) throws UaException {
        CreateSessionRequest request = (CreateSessionRequest)serviceRequest.getRequest();
        long maxSessionCount = this.server.getConfig().getLimits().getMaxSessionCount().longValue();
        if ((long)(this.createdSessions.size() + this.activeSessions.size()) >= maxSessionCount) {
            serviceRequest.setServiceFault(2153119744L);
            return;
        }
        ByteString serverNonce = NonceUtil.generateNonce((int)32);
        NodeId authenticationToken = new NodeId(0, NonceUtil.generateNonce((int)32));
        long maxRequestMessageSize = serviceRequest.getServer().getConfig().getMessageLimits().getMaxMessageSize();
        double revisedSessionTimeout = Math.max(5000.0, Math.min(120000.0, request.getRequestedSessionTimeout()));
        long secureChannelId = serviceRequest.getSecureChannelId();
        EndpointDescription endpoint = serviceRequest.getEndpoint();
        SecurityPolicy securityPolicy = SecurityPolicy.fromUri((String)endpoint.getSecurityPolicyUri());
        EndpointDescription[] serverEndpoints = (EndpointDescription[])this.server.getEndpointDescriptions().stream().filter(ed -> SessionManager.endpointMatchesUrl(ed, request.getEndpointUrl())).toArray(EndpointDescription[]::new);
        ByteString clientNonce = request.getClientNonce();
        NonceUtil.validateNonce((ByteString)clientNonce);
        if (securityPolicy != SecurityPolicy.None && this.clientNonces.contains(clientNonce)) {
            throw new UaException(2149842944L);
        }
        if (securityPolicy != SecurityPolicy.None && clientNonce.isNotNull()) {
            this.clientNonces.add(clientNonce);
            while (this.clientNonces.size() > 64) {
                this.clientNonces.remove(0);
            }
        }
        ByteString clientCertificateBytes = request.getClientCertificate();
        if (securityPolicy != SecurityPolicy.None && serviceRequest.getClientCertificateBytes() != null && !Objects.equal((Object)clientCertificateBytes, (Object)serviceRequest.getClientCertificateBytes())) {
            throw new UaException(2148728832L, "certificate used to open secure channel differs from certificate used to create session");
        }
        SecurityConfiguration securityConfiguration = this.createSecurityConfiguration(endpoint, clientCertificateBytes);
        if (securityPolicy != SecurityPolicy.None) {
            X509Certificate clientCertificate = securityConfiguration.getClientCertificate();
            List<X509Certificate> clientCertificateChain = securityConfiguration.getClientCertificateChain();
            if (clientCertificate == null || clientCertificateChain == null) {
                throw new UaException(2148728832L, "client certificate must be non-null");
            }
            String applicationUri = request.getClientDescription().getApplicationUri();
            CertificateValidationUtil.validateApplicationUri((X509Certificate)clientCertificate, (String)applicationUri);
            CertificateValidator certificateValidator = this.server.getConfig().getCertificateValidator();
            certificateValidator.validate(clientCertificate);
            certificateValidator.verifyTrustChain(clientCertificateChain);
        }
        SignatureData serverSignature = this.getServerSignature(securityPolicy, securityConfiguration.getKeyPair(), clientNonce, securityConfiguration.getClientCertificateBytes());
        NodeId sessionId = new NodeId(1, "Session:" + UUID.randomUUID());
        String sessionName = request.getSessionName();
        Duration sessionTimeout = Duration.ofMillis(DoubleMath.roundToLong((double)revisedSessionTimeout, (RoundingMode)RoundingMode.UP));
        Session session = new Session(this.server, sessionId, sessionName, sessionTimeout, secureChannelId, endpoint, securityConfiguration);
        session.setLastNonce(serverNonce);
        session.addLifecycleListener((s, remove) -> {
            this.createdSessions.remove(authenticationToken);
            this.activeSessions.remove(authenticationToken);
        });
        this.createdSessions.put(authenticationToken, session);
        CreateSessionResponse response = new CreateSessionResponse(serviceRequest.createResponseHeader(), sessionId, authenticationToken, Double.valueOf(revisedSessionTimeout), serverNonce, endpoint.getServerCertificate(), serverEndpoints, new SignedSoftwareCertificate[0], serverSignature, Unsigned.uint((long)maxRequestMessageSize));
        serviceRequest.setResponse((UaResponseMessage)response);
    }

    private SecurityConfiguration createSecurityConfiguration(EndpointDescription endpoint, ByteString clientCertificateBytes) throws UaException {
        SecurityPolicy securityPolicy = SecurityPolicy.fromUri((String)endpoint.getSecurityPolicyUri());
        MessageSecurityMode securityMode = endpoint.getSecurityMode();
        X509Certificate clientCertificate = null;
        List clientCertificateChain = null;
        KeyPair keyPair = null;
        X509Certificate serverCertificate = null;
        List serverCertificateChain = null;
        if (securityPolicy != SecurityPolicy.None) {
            clientCertificate = CertificateUtil.decodeCertificate((byte[])clientCertificateBytes.bytes());
            clientCertificateChain = CertificateUtil.decodeCertificates((byte[])clientCertificateBytes.bytes());
            ByteString thumbprint = ByteString.of((byte[])DigestUtil.sha1((byte[])endpoint.getServerCertificate().bytesOrEmpty()));
            keyPair = (KeyPair)this.server.getConfig().getCertificateManager().getKeyPair(thumbprint).orElseThrow(() -> new UaException(0x80890000L));
            serverCertificate = (X509Certificate)this.server.getConfig().getCertificateManager().getCertificate(thumbprint).orElseThrow(() -> new UaException(0x80890000L));
            serverCertificateChain = this.server.getConfig().getCertificateManager().getCertificateChain(thumbprint).map(Lists::newArrayList).orElseThrow(() -> new UaException(0x80890000L));
        }
        return new SecurityConfiguration(securityPolicy, securityMode, keyPair, serverCertificate, serverCertificateChain, clientCertificate, clientCertificateChain);
    }

    private static boolean endpointMatchesUrl(EndpointDescription endpoint, String requestedEndpointUrl) {
        String endpointHost = EndpointUtil.getHost((String)Strings.nullToEmpty((String)endpoint.getEndpointUrl()));
        String requestedHost = EndpointUtil.getHost((String)Strings.nullToEmpty((String)requestedEndpointUrl));
        return Strings.nullToEmpty((String)endpointHost).equalsIgnoreCase(Strings.nullToEmpty((String)requestedHost));
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void onActivateSession(ServiceRequest serviceRequest) throws UaException {
        ActivateSessionRequest request = (ActivateSessionRequest)serviceRequest.getRequest();
        long secureChannelId = serviceRequest.getSecureChannelId();
        NodeId authToken = request.getRequestHeader().getAuthenticationToken();
        List clientSoftwareCertificates = ConversionUtil.l((Object[])request.getClientSoftwareCertificates());
        Session session = this.createdSessions.get(authToken);
        if (session == null) {
            session = this.activeSessions.get(authToken);
            if (session == null) {
                throw new UaException(2149908480L);
            }
            SessionManager.verifyClientSignature(session, request);
            SecurityConfiguration securityConfiguration = session.getSecurityConfiguration();
            if (session.getSecureChannelId() == secureChannelId) {
                UserIdentityToken identityToken = this.decodeIdentityToken(request.getUserIdentityToken(), session.getEndpoint().getUserIdentityTokens());
                Object identityObject = this.validateIdentityToken(session, identityToken, request.getUserTokenSignature());
                Object[] results = new StatusCode[clientSoftwareCertificates.size()];
                Arrays.fill(results, StatusCode.GOOD);
                ByteString serverNonce = NonceUtil.generateNonce((int)32);
                session.setClientAddress(serviceRequest.getClientAddress());
                session.setIdentityObject(identityObject);
                session.setLastNonce(serverNonce);
                session.setLocaleIds(request.getLocaleIds());
                ActivateSessionResponse response = new ActivateSessionResponse(serviceRequest.createResponseHeader(), serverNonce, (StatusCode[])results, new DiagnosticInfo[0]);
                serviceRequest.setResponse((UaResponseMessage)response);
                return;
            } else {
                ByteString clientCertificateBytes = serviceRequest.getClientCertificateBytes();
                UserIdentityToken identityToken = this.decodeIdentityToken(request.getUserIdentityToken(), session.getEndpoint().getUserIdentityTokens());
                Object identityObject = this.validateIdentityToken(session, identityToken, request.getUserTokenSignature());
                boolean sameIdentity = Objects.equal((Object)identityObject, (Object)session.getIdentityObject());
                boolean sameCertificate = Objects.equal((Object)clientCertificateBytes, (Object)securityConfiguration.getClientCertificateBytes());
                if (!sameIdentity || !sameCertificate) throw new UaException(2148728832L);
                SecurityConfiguration newSecurityConfiguration = this.createSecurityConfiguration(serviceRequest.getEndpoint(), clientCertificateBytes);
                session.setEndpoint(serviceRequest.getEndpoint());
                session.setSecureChannelId(secureChannelId);
                session.setSecurityConfiguration(newSecurityConfiguration);
                this.logger.debug("Session id={} is now associated with secureChannelId={}", (Object)session.getSessionId(), (Object)secureChannelId);
                Object[] results = new StatusCode[clientSoftwareCertificates.size()];
                Arrays.fill(results, StatusCode.GOOD);
                ByteString serverNonce = NonceUtil.generateNonce((int)32);
                session.setClientAddress(serviceRequest.getClientAddress());
                session.setLastNonce(serverNonce);
                session.setLocaleIds(request.getLocaleIds());
                ActivateSessionResponse response = new ActivateSessionResponse(serviceRequest.createResponseHeader(), serverNonce, (StatusCode[])results, new DiagnosticInfo[0]);
                serviceRequest.setResponse((UaResponseMessage)response);
            }
            return;
        } else {
            if (secureChannelId != session.getSecureChannelId()) {
                throw new UaException(2148728832L);
            }
            SessionManager.verifyClientSignature(session, request);
            UserIdentityToken identityToken = this.decodeIdentityToken(request.getUserIdentityToken(), session.getEndpoint().getUserIdentityTokens());
            Object identityObject = this.validateIdentityToken(session, identityToken, request.getUserTokenSignature());
            this.createdSessions.remove(authToken);
            this.activeSessions.put(authToken, session);
            Object[] results = new StatusCode[clientSoftwareCertificates.size()];
            Arrays.fill(results, StatusCode.GOOD);
            ByteString serverNonce = NonceUtil.generateNonce((int)32);
            session.setClientAddress(serviceRequest.getClientAddress());
            session.setIdentityObject(identityObject);
            session.setLocaleIds(request.getLocaleIds());
            session.setLastNonce(serverNonce);
            ActivateSessionResponse response = new ActivateSessionResponse(serviceRequest.createResponseHeader(), serverNonce, (StatusCode[])results, new DiagnosticInfo[0]);
            serviceRequest.setResponse((UaResponseMessage)response);
        }
    }

    private static void verifyClientSignature(Session session, ActivateSessionRequest request) throws UaException {
        SecurityConfiguration securityConfiguration = session.getSecurityConfiguration();
        if (securityConfiguration.getSecurityPolicy() != SecurityPolicy.None) {
            SignatureData clientSignature = request.getClientSignature();
            byte[] dataBytes = Bytes.concat((byte[][])new byte[][]{securityConfiguration.getServerCertificateBytes().bytesOrEmpty(), session.getLastNonce().bytesOrEmpty()});
            byte[] signatureBytes = clientSignature.getSignature().bytesOrEmpty();
            SignatureUtil.verify((SecurityAlgorithm)SecurityAlgorithm.fromUri((String)clientSignature.getAlgorithm()), (X509Certificate)securityConfiguration.getClientCertificate(), (byte[])dataBytes, (byte[])signatureBytes);
        }
    }

    @Nonnull
    private UserIdentityToken decodeIdentityToken(@Nullable ExtensionObject identityTokenXo, @Nullable UserTokenPolicy[] tokenPolicies) {
        Object identityToken;
        if (identityTokenXo != null && !identityTokenXo.isNull() && (identityToken = identityTokenXo.decodeOrNull(this.server.getSerializationContext())) instanceof UserIdentityToken) {
            return (UserIdentityToken)identityToken;
        }
        String policyId = ConversionUtil.l((Object[])tokenPolicies).stream().filter(p -> p.getTokenType() == UserTokenType.Anonymous).findFirst().map(UserTokenPolicy::getPolicyId).orElse(null);
        return new AnonymousIdentityToken(policyId);
    }

    private Object validateIdentityToken(Session session, Object tokenObject, SignatureData tokenSignature) throws UaException {
        IdentityValidator identityValidator = this.server.getConfig().getIdentityValidator();
        UserTokenPolicy tokenPolicy = this.validatePolicyId(session, tokenObject);
        if (tokenObject instanceof UserIdentityToken) {
            return identityValidator.validateIdentityToken(session, (UserIdentityToken)tokenObject, tokenPolicy, tokenSignature);
        }
        throw new UaException(0x80200000L);
    }

    private UserTokenPolicy validatePolicyId(Session session, Object tokenObject) throws UaException {
        if (tokenObject instanceof UserIdentityToken) {
            UserIdentityToken token = (UserIdentityToken)tokenObject;
            String policyId = token.getPolicyId();
            List userIdentityTokens = ConversionUtil.l((Object[])session.getEndpoint().getUserIdentityTokens());
            Optional<UserTokenPolicy> policy = userIdentityTokens.stream().filter(t -> Objects.equal((Object)policyId, (Object)t.getPolicyId())).findFirst();
            return policy.orElseThrow(() -> new UaException(0x80200000L, "policy not found: " + policyId));
        }
        throw new UaException(0x80200000L);
    }

    public void onCloseSession(ServiceRequest service) throws UaException {
        long secureChannelId = service.getSecureChannelId();
        NodeId authToken = service.getRequest().getRequestHeader().getAuthenticationToken();
        Session session = this.activeSessions.get(authToken);
        if (session != null) {
            if (session.getSecureChannelId() != secureChannelId) {
                throw new UaException(0x80220000L);
            }
            this.activeSessions.remove(authToken);
            session.onCloseSession(service);
        } else {
            session = this.createdSessions.get(authToken);
            if (session == null) {
                throw new UaException(2149908480L);
            }
            if (session.getSecureChannelId() != secureChannelId) {
                throw new UaException(0x80220000L);
            }
            this.createdSessions.remove(authToken);
            session.onCloseSession(service);
        }
    }

    public void onCancel(ServiceRequest service) throws UaException {
        this.session(service).onCancel(service);
    }

    private SignatureData getServerSignature(SecurityPolicy securityPolicy, KeyPair keyPair, ByteString clientNonce, ByteString clientCertificate) throws UaException {
        if (securityPolicy == SecurityPolicy.None) {
            return new SignatureData(null, null);
        }
        try {
            SecurityAlgorithm algorithm = securityPolicy.getAsymmetricSignatureAlgorithm();
            byte[] data = Bytes.concat((byte[][])new byte[][]{clientCertificate.bytes(), clientNonce.bytes()});
            byte[] signature = SignatureUtil.sign((SecurityAlgorithm)algorithm, (PrivateKey)keyPair.getPrivate(), (ByteBuffer[])new ByteBuffer[]{ByteBuffer.wrap(data)});
            return new SignatureData(algorithm.getUri(), ByteString.of((byte[])signature));
        }
        catch (UaRuntimeException e) {
            throw new UaException(2148728832L);
        }
    }

    public void onRead(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getAttributeServiceSet().onRead(service);
    }

    public void onWrite(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getAttributeServiceSet().onWrite(service);
    }

    public void onHistoryRead(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getAttributeHistoryServiceSet().onHistoryRead(service);
    }

    public void onHistoryUpdate(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getAttributeHistoryServiceSet().onHistoryUpdate(service);
    }

    public void onBrowse(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getViewServiceSet().onBrowse(service);
    }

    public void onBrowseNext(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getViewServiceSet().onBrowseNext(service);
    }

    public void onTranslateBrowsePaths(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getViewServiceSet().onTranslateBrowsePaths(service);
    }

    public void onRegisterNodes(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getViewServiceSet().onRegisterNodes(service);
    }

    public void onUnregisterNodes(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getViewServiceSet().onUnregisterNodes(service);
    }

    public void onAddNodes(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getNodeManagementServiceSet().onAddNodes(service);
    }

    public void onAddReferences(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getNodeManagementServiceSet().onAddReferences(service);
    }

    public void onDeleteNodes(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getNodeManagementServiceSet().onDeleteNodes(service);
    }

    public void onDeleteReferences(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getNodeManagementServiceSet().onDeleteReferences(service);
    }

    public void onCreateSubscription(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSubscriptionServiceSet().onCreateSubscription(service);
    }

    public void onModifySubscription(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSubscriptionServiceSet().onModifySubscription(service);
    }

    public void onSetPublishingMode(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSubscriptionServiceSet().onSetPublishingMode(service);
    }

    public void onPublish(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSubscriptionServiceSet().onPublish(service);
    }

    public void onRepublish(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSubscriptionServiceSet().onRepublish(service);
    }

    public void onTransferSubscriptions(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSubscriptionServiceSet().onTransferSubscriptions(service);
    }

    public void onDeleteSubscriptions(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getSubscriptionServiceSet().onDeleteSubscriptions(service);
    }

    public void onCreateMonitoredItems(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getMonitoredItemServiceSet().onCreateMonitoredItems(service);
    }

    public void onModifyMonitoredItems(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getMonitoredItemServiceSet().onModifyMonitoredItems(service);
    }

    public void onSetMonitoringMode(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getMonitoredItemServiceSet().onSetMonitoringMode(service);
    }

    public void onSetTriggering(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getMonitoredItemServiceSet().onSetTriggering(service);
    }

    public void onDeleteMonitoredItems(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getMonitoredItemServiceSet().onDeleteMonitoredItems(service);
    }

    public void onCall(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getMethodServiceSet().onCall(service);
    }

    public void onQueryFirst(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getQueryServiceSet().onQueryFirst(service);
    }

    public void onQueryNext(ServiceRequest service) throws UaException {
        Session session = this.session(service);
        session.getQueryServiceSet().onQueryNext(service);
    }
}

