/*
 * Decompiled with CFR 0.152.
 */
package elki.outlier.lof;

import elki.Algorithm;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.datastore.DataStoreUtil;
import elki.database.datastore.DoubleDataStore;
import elki.database.datastore.WritableDataStore;
import elki.database.datastore.WritableDoubleDataStore;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDs;
import elki.database.ids.DoubleDBIDList;
import elki.database.ids.DoubleDBIDListIter;
import elki.database.query.QueryBuilder;
import elki.database.query.range.RangeSearcher;
import elki.database.relation.DoubleRelation;
import elki.database.relation.MaterializedDoubleRelation;
import elki.database.relation.Relation;
import elki.distance.Distance;
import elki.distance.minkowski.EuclideanDistance;
import elki.logging.Logging;
import elki.logging.progress.AbstractProgress;
import elki.logging.progress.FiniteProgress;
import elki.math.DoubleMinMax;
import elki.math.MeanVariance;
import elki.outlier.OutlierAlgorithm;
import elki.result.Metadata;
import elki.result.outlier.OutlierResult;
import elki.result.outlier.QuotientOutlierScoreMeta;
import elki.utilities.datastructures.arrays.DoubleIntegerArrayQuickSort;
import elki.utilities.documentation.Description;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.DoubleParameter;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;
import java.util.Arrays;

@Title(value="LOCI: Fast Outlier Detection Using the Local Correlation Integral")
@Description(value="Algorithm to compute outliers based on the Local Correlation Integral")
@Reference(authors="S. Papadimitriou, H. Kitagawa, P. B. Gibbons, C. Faloutsos", title="LOCI: Fast Outlier Detection Using the Local Correlation Integral", booktitle="Proc. 19th IEEE Int. Conf. on Data Engineering (ICDE '03)", url="https://doi.org/10.1109/ICDE.2003.1260802", bibkey="DBLP:conf/icde/PapadimitriouKGF03")
public class LOCI<O>
implements OutlierAlgorithm {
    private static final Logging LOG = Logging.getLogger(LOCI.class);
    private Distance<? super O> distance;
    private double rmax;
    private int nmin = 0;
    private double alpha = 0.5;

    public LOCI(Distance<? super O> distance, double rmax, int nmin, double alpha) {
        this.distance = distance;
        this.rmax = rmax;
        this.nmin = nmin;
        this.alpha = alpha;
    }

    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{this.distance.getInputTypeRestriction()});
    }

    public OutlierResult run(Relation<O> relation) {
        RangeSearcher rangeQuery = new QueryBuilder(relation, this.distance).rangeByDBID();
        DBIDs ids = relation.getDBIDs();
        WritableDataStore interestingDistances = DataStoreUtil.makeStorage((DBIDs)relation.getDBIDs(), (int)9, DoubleIntArrayList.class);
        this.precomputeInterestingRadii(ids, (RangeSearcher<DBIDRef>)rangeQuery, (WritableDataStore<DoubleIntArrayList>)interestingDistances);
        FiniteProgress progressLOCI = LOG.isVerbose() ? new FiniteProgress("LOCI scores", relation.size(), LOG) : null;
        WritableDoubleDataStore mdef_norm = DataStoreUtil.makeDoubleStorage((DBIDs)relation.getDBIDs(), (int)4);
        WritableDoubleDataStore mdef_radius = DataStoreUtil.makeDoubleStorage((DBIDs)relation.getDBIDs(), (int)4);
        DoubleMinMax minmax = new DoubleMinMax();
        MeanVariance mv_n_r_alpha = new MeanVariance();
        DBIDIter iditer = ids.iter();
        while (iditer.valid()) {
            DoubleIntArrayList cdist = (DoubleIntArrayList)interestingDistances.get((DBIDRef)iditer);
            double maxdist = cdist.getDouble(cdist.size() - 1);
            int maxneig = cdist.getInt(cdist.size() - 1);
            double maxmdefnorm = 0.0;
            double maxnormr = 0.0;
            if (maxneig >= this.nmin) {
                DoubleDBIDList maxneighbors = rangeQuery.getRange((Object)iditer, maxdist);
                int size = cdist.size();
                for (int i = 0; i < size; ++i) {
                    double sigma_nhat_r_alpha;
                    double sigmamdef;
                    if (cdist.getInt(i) < this.nmin) continue;
                    double r = cdist.getDouble(i);
                    double alpha_r = this.alpha * r;
                    int n_alphar = cdist.getInt(cdist.find(alpha_r));
                    mv_n_r_alpha.reset();
                    DoubleDBIDListIter neighbor = maxneighbors.iter();
                    while (neighbor.valid() && !(neighbor.doubleValue() > r)) {
                        DoubleIntArrayList cdist2 = (DoubleIntArrayList)interestingDistances.get((DBIDRef)neighbor);
                        int rn_alphar = cdist2.getInt(cdist2.find(alpha_r));
                        mv_n_r_alpha.put((double)rn_alphar);
                        neighbor.advance();
                    }
                    double nhat_r_alpha = mv_n_r_alpha.getMean();
                    double mdef = nhat_r_alpha - (double)n_alphar;
                    double mdefnorm = mdef / (sigmamdef = (sigma_nhat_r_alpha = mv_n_r_alpha.getPopulationStddev()));
                    if (!(mdefnorm > maxmdefnorm)) continue;
                    maxmdefnorm = mdefnorm;
                    maxnormr = r;
                }
            } else {
                maxmdefnorm = Double.POSITIVE_INFINITY;
                maxnormr = maxdist;
            }
            mdef_norm.putDouble((DBIDRef)iditer, maxmdefnorm);
            mdef_radius.putDouble((DBIDRef)iditer, maxnormr);
            minmax.put(maxmdefnorm);
            LOG.incrementProcessed((AbstractProgress)progressLOCI);
            iditer.advance();
        }
        LOG.ensureCompleted(progressLOCI);
        MaterializedDoubleRelation scoreResult = new MaterializedDoubleRelation("LOCI normalized MDEF", relation.getDBIDs(), (DoubleDataStore)mdef_norm);
        QuotientOutlierScoreMeta scoreMeta = new QuotientOutlierScoreMeta(minmax.getMin(), minmax.getMax(), 0.0, Double.POSITIVE_INFINITY, 0.0);
        OutlierResult result = new OutlierResult(scoreMeta, (DoubleRelation)scoreResult);
        Metadata.hierarchyOf((Object)result).addChild((Object)new MaterializedDoubleRelation("LOCI MDEF Radius", relation.getDBIDs(), (DoubleDataStore)mdef_radius));
        return result;
    }

    protected void precomputeInterestingRadii(DBIDs ids, RangeSearcher<DBIDRef> rangeQuery, WritableDataStore<DoubleIntArrayList> interestingDistances) {
        FiniteProgress progressPreproc = LOG.isVerbose() ? new FiniteProgress("LOCI preprocessing", ids.size(), LOG) : null;
        DBIDIter iditer = ids.iter();
        while (iditer.valid()) {
            DoubleDBIDList neighbors = rangeQuery.getRange((Object)iditer, this.rmax);
            DoubleIntArrayList cdist = new DoubleIntArrayList(neighbors.size() << 1);
            int i = 0;
            DoubleDBIDListIter ni = neighbors.iter();
            while (ni.valid()) {
                double ri;
                double curdist = ni.doubleValue();
                ++i;
                ni.advance();
                if (ni.valid() && curdist == ni.doubleValue()) continue;
                cdist.append(curdist, i);
                if (this.alpha == 1.0 || !((ri = curdist / this.alpha) <= this.rmax)) continue;
                cdist.append(ri, Integer.MIN_VALUE);
            }
            cdist.sort();
            int lastk = 0;
            int size = cdist.size();
            for (int i2 = 0; i2 < size; ++i2) {
                int k = cdist.getInt(i2);
                if (k == Integer.MIN_VALUE) {
                    cdist.setValue(i2, lastk);
                    continue;
                }
                lastk = k;
            }
            interestingDistances.put((DBIDRef)iditer, (Object)cdist);
            LOG.incrementProcessed((AbstractProgress)progressPreproc);
            iditer.advance();
        }
        LOG.ensureCompleted(progressPreproc);
    }

    public static class Par<O>
    implements Parameterizer {
        public static final OptionID RMAX_ID = new OptionID("loci.rmax", "The maximum radius of the neighborhood to be considered.");
        public static final OptionID NMIN_ID = new OptionID("loci.nmin", "Minimum neighborhood size to be considered.");
        public static final OptionID ALPHA_ID = new OptionID("loci.alpha", "Scaling factor for averaging neighborhood");
        protected Distance<? super O> distance;
        protected double rmax;
        protected int nmin = 0;
        protected double alpha = 0.5;

        public void configure(Parameterization config) {
            new ObjectParameter(Algorithm.Utils.DISTANCE_FUNCTION_ID, Distance.class, EuclideanDistance.class).grab(config, x -> {
                this.distance = x;
            });
            new DoubleParameter(RMAX_ID).grab(config, x -> {
                this.rmax = x;
            });
            new IntParameter(NMIN_ID, 20).grab(config, x -> {
                this.nmin = x;
            });
            new DoubleParameter(ALPHA_ID, 0.5).grab(config, x -> {
                this.alpha = x;
            });
        }

        public LOCI<O> make() {
            return new LOCI<O>(this.distance, this.rmax, this.nmin, this.alpha);
        }
    }

    private static class DoubleIntArrayList {
        double[] keys;
        int[] vals;
        int size = 0;

        public DoubleIntArrayList(int alloc) {
            this.keys = new double[alloc];
            this.vals = new int[alloc];
            this.size = 0;
        }

        public int size() {
            return this.size;
        }

        public double getDouble(int i) {
            return this.keys[i];
        }

        public int getInt(int i) {
            return this.vals[i];
        }

        public void setValue(int i, int val) {
            this.vals[i] = val;
        }

        public void append(double key, int val) {
            if (this.size == this.keys.length) {
                this.keys = Arrays.copyOf(this.keys, this.size << 1);
                this.vals = Arrays.copyOf(this.vals, this.size << 1);
            }
            this.keys[this.size] = key;
            this.vals[this.size] = val;
            ++this.size;
        }

        public int find(double search) {
            int a = 0;
            int b = this.size - 1;
            while (a <= b) {
                int mid = a + b >>> 1;
                double cur = this.keys[mid];
                if (cur > search) {
                    b = mid - 1;
                    continue;
                }
                a = mid + 1;
            }
            return b;
        }

        public void sort() {
            DoubleIntegerArrayQuickSort.sort((double[])this.keys, (int[])this.vals, (int)this.size);
        }
    }
}

