/*
 * Decompiled with CFR 0.152.
 */
package elki.outlier.distance;

import elki.Algorithm;
import elki.data.NumberVector;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.datastore.DataStoreUtil;
import elki.database.datastore.DoubleDataStore;
import elki.database.datastore.WritableDoubleDataStore;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDUtil;
import elki.database.ids.DBIDs;
import elki.database.ids.DoubleDBIDList;
import elki.database.ids.DoubleDBIDListIter;
import elki.database.ids.ModifiableDoubleDBIDList;
import elki.database.query.QueryBuilder;
import elki.database.query.distance.PrimitiveDistanceQuery;
import elki.database.relation.DoubleRelation;
import elki.database.relation.MaterializedDoubleRelation;
import elki.database.relation.Relation;
import elki.distance.NumberVectorDistance;
import elki.distance.minkowski.EuclideanDistance;
import elki.math.DoubleMinMax;
import elki.outlier.OutlierAlgorithm;
import elki.result.Metadata;
import elki.result.ReferencePointsResult;
import elki.result.outlier.BasicOutlierScoreMeta;
import elki.result.outlier.OutlierResult;
import elki.utilities.documentation.Description;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.exceptions.AbortException;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;
import elki.utilities.referencepoints.GridBasedReferencePoints;
import elki.utilities.referencepoints.ReferencePointsHeuristic;
import java.util.Collection;

@Title(value="An Efficient Reference-based Approach to Outlier Detection in Large Datasets")
@Description(value="Computes kNN distances approximately, using reference points with various reference point strategies.")
@Reference(authors="Y. Pei, O. R. Zaiane, Y. Gao", title="An Efficient Reference-based Approach to Outlier Detection in Large Datasets", booktitle="Proc. 6th IEEE Int. Conf. on Data Mining (ICDM '06)", url="https://doi.org/10.1109/ICDM.2006.17", bibkey="DBLP:conf/icdm/PeiZG06")
public class ReferenceBasedOutlierDetection
implements OutlierAlgorithm {
    protected NumberVectorDistance<? super NumberVector> distance;
    protected int k;
    protected ReferencePointsHeuristic refp;

    public ReferenceBasedOutlierDetection(int k, NumberVectorDistance<? super NumberVector> distance, ReferencePointsHeuristic refp) {
        this.distance = distance;
        this.k = k;
        this.refp = refp;
    }

    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{this.distance.getInputTypeRestriction()});
    }

    public OutlierResult run(Relation<? extends NumberVector> relation) {
        PrimitiveDistanceQuery distq = (PrimitiveDistanceQuery)new QueryBuilder(relation, this.distance).distanceQuery();
        Collection refPoints = this.refp.getReferencePoints(relation);
        if (refPoints.isEmpty()) {
            throw new AbortException("Cannot compute ROS without reference points!");
        }
        DBIDs ids = relation.getDBIDs();
        if (this.k >= ids.size()) {
            throw new AbortException("k must not be chosen larger than the database size!");
        }
        WritableDoubleDataStore rbod_score = DataStoreUtil.makeDoubleStorage((DBIDs)ids, (int)6, (double)Double.NaN);
        for (NumberVector refPoint : refPoints) {
            DoubleDBIDList referenceDists = this.computeDistanceVector(refPoint, relation, (PrimitiveDistanceQuery<? super NumberVector>)distq);
            this.updateDensities(rbod_score, referenceDists);
        }
        DoubleMinMax mm = new DoubleMinMax();
        DBIDIter iditer = relation.iterDBIDs();
        while (iditer.valid()) {
            mm.put(rbod_score.doubleValue((DBIDRef)iditer));
            iditer.advance();
        }
        double scale = mm.getMax() > 0.0 ? 1.0 / mm.getMax() : 1.0;
        mm.reset();
        DBIDIter iditer2 = relation.iterDBIDs();
        while (iditer2.valid()) {
            double score = 1.0 - rbod_score.doubleValue((DBIDRef)iditer2) * scale;
            mm.put(score);
            rbod_score.putDouble((DBIDRef)iditer2, score);
            iditer2.advance();
        }
        MaterializedDoubleRelation scoreResult = new MaterializedDoubleRelation("Reference-points Outlier Scores", relation.getDBIDs(), (DoubleDataStore)rbod_score);
        BasicOutlierScoreMeta scoreMeta = new BasicOutlierScoreMeta(mm.getMin(), mm.getMax(), 0.0, 1.0, 0.0);
        OutlierResult result = new OutlierResult(scoreMeta, (DoubleRelation)scoreResult);
        ReferencePointsResult refresult = new ReferencePointsResult(refPoints);
        Metadata.of((Object)refresult).setLongName("Reference points");
        Metadata.hierarchyOf((Object)result).addChild((Object)refresult);
        return result;
    }

    protected DoubleDBIDList computeDistanceVector(NumberVector refPoint, Relation<? extends NumberVector> database, PrimitiveDistanceQuery<? super NumberVector> distFunc) {
        ModifiableDoubleDBIDList referenceDists = DBIDUtil.newDistanceDBIDList((int)database.size());
        DBIDIter iditer = database.iterDBIDs();
        while (iditer.valid()) {
            referenceDists.add(distFunc.distance((DBIDRef)iditer, (Object)refPoint), (DBIDRef)iditer);
            iditer.advance();
        }
        return referenceDists.sort();
    }

    protected void updateDensities(WritableDoubleDataStore rbod_score, DoubleDBIDList referenceDists) {
        DoubleDBIDListIter it = referenceDists.iter();
        for (int l = 0; l < referenceDists.size(); ++l) {
            double density = this.computeDensity(referenceDists, it, l);
            it.seek(l);
            if (density > rbod_score.doubleValue((DBIDRef)it)) continue;
            rbod_score.putDouble((DBIDRef)it, density);
        }
    }

    protected double computeDensity(DoubleDBIDList referenceDists, DoubleDBIDListIter iter, int index) {
        int size = referenceDists.size();
        double xDist = iter.seek(index).doubleValue();
        int lef = index;
        int rig = index;
        double sum = 0.0;
        double lef_d = --lef >= 0 ? xDist - iter.seek(lef).doubleValue() : Double.POSITIVE_INFINITY;
        double rig_d = ++rig < size ? iter.seek(rig).doubleValue() - xDist : Double.POSITIVE_INFINITY;
        for (int i = 0; i < this.k; ++i) {
            if (lef >= 0 && rig < size) {
                if (lef_d < rig_d) {
                    sum += lef_d;
                    lef_d = --lef >= 0 ? xDist - iter.seek(lef).doubleValue() : Double.POSITIVE_INFINITY;
                    continue;
                }
                sum += rig_d;
                rig_d = ++rig < size ? iter.seek(rig).doubleValue() - xDist : Double.POSITIVE_INFINITY;
                continue;
            }
            if (lef >= 0) {
                sum += lef_d;
                lef_d = --lef >= 0 ? xDist - iter.seek(lef).doubleValue() : Double.POSITIVE_INFINITY;
                continue;
            }
            if (rig < size) {
                sum += rig_d;
                rig_d = ++rig < size ? iter.seek(rig).doubleValue() - xDist : Double.POSITIVE_INFINITY;
                continue;
            }
            throw new IndexOutOfBoundsException("Less than k objects?");
        }
        return (double)this.k / sum;
    }

    public static class Par
    implements Parameterizer {
        public static final OptionID REFP_ID = new OptionID("refod.refp", "The heuristic for finding reference points.");
        public static final OptionID K_ID = new OptionID("refod.k", "The number of nearest neighbors");
        protected NumberVectorDistance<? super NumberVector> distance;
        private int k;
        private ReferencePointsHeuristic refp;

        public void configure(Parameterization config) {
            new ObjectParameter(Algorithm.Utils.DISTANCE_FUNCTION_ID, NumberVectorDistance.class, EuclideanDistance.class).grab(config, x -> {
                this.distance = x;
            });
            ((IntParameter)new IntParameter(K_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ONE_INT)).grab(config, x -> {
                this.k = x;
            });
            new ObjectParameter(REFP_ID, ReferencePointsHeuristic.class, GridBasedReferencePoints.class).grab(config, x -> {
                this.refp = x;
            });
        }

        public ReferenceBasedOutlierDetection make() {
            return new ReferenceBasedOutlierDetection(this.k, this.distance, this.refp);
        }
    }
}

