/*
 * Decompiled with CFR 0.152.
 */
package elki.outlier.density;

import elki.data.NumberVector;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.Database;
import elki.database.datastore.DataStoreUtil;
import elki.database.datastore.DoubleDataStore;
import elki.database.datastore.WritableDoubleDataStore;
import elki.database.ids.ArrayModifiableDBIDs;
import elki.database.ids.DBIDArrayIter;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDUtil;
import elki.database.ids.DBIDs;
import elki.database.relation.DoubleRelation;
import elki.database.relation.MaterializedDoubleRelation;
import elki.database.relation.Relation;
import elki.logging.Logging;
import elki.logging.progress.AbstractProgress;
import elki.logging.progress.FiniteProgress;
import elki.logging.progress.StepProgress;
import elki.math.DoubleMinMax;
import elki.outlier.OutlierAlgorithm;
import elki.result.outlier.BasicOutlierScoreMeta;
import elki.result.outlier.OutlierResult;
import elki.utilities.documentation.Description;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.IntParameter;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.TreeSet;

@Title(value="HySortOD: Hypercube-Based Outlier Detection")
@Description(value="Algorithm that uses an efficient hypercube-ordering-and-searching strategy for fast outlier detection.")
@Reference(authors="Eug\u00eanio F. Cabral, and Robson L.F. Cordeiro", title="Fast and Scalable Outlier Detection with Sorted Hypercubes", booktitle="Proc. 29th ACM Int. Conf. on Information & Knowledge Management (CIKM'20)", url="https://doi.org/10.1145/3340531.3412033")
public class HySortOD
implements OutlierAlgorithm {
    private static final Logging LOG = Logging.getLogger(HySortOD.class);
    private int b;
    private final double l;
    private DensityStrategy strategy;

    public HySortOD(int b, int minSplit) {
        this.b = b;
        this.l = 1.0 / (double)this.b;
        this.strategy = minSplit > 0 ? new TreeStrategy(minSplit) : new NaiveStrategy();
    }

    public OutlierResult run(Database db, Relation<? extends NumberVector> relation) {
        StepProgress stepprog = LOG.isVerbose() ? new StepProgress(3) : null;
        LOG.beginStep(stepprog, 1, "Ordering of hypercubes lexicographically according to their coordinates.");
        List<Hypercube> H = this.getSortedHypercubes(relation);
        LOG.beginStep(stepprog, 2, "Obtaining densities per hypercube.");
        int[] W = this.strategy.buildIndex(H).getDensities();
        WritableDoubleDataStore scores = DataStoreUtil.makeDoubleStorage((DBIDs)relation.getDBIDs(), (int)30);
        DoubleMinMax minmax = new DoubleMinMax();
        LOG.beginStep(stepprog, 3, "Computing hypercube scores");
        FiniteProgress prog = LOG.isVerbose() ? new FiniteProgress("HySortOD scores", relation.size(), LOG) : null;
        for (int hypercube = 0; hypercube < H.size(); ++hypercube) {
            double hypercubeScore = this.score(W[hypercube]);
            minmax.put(hypercubeScore);
            DBIDIter iter = H.get(hypercube).getInstances().iter();
            while (iter.valid()) {
                scores.putDouble((DBIDRef)iter, hypercubeScore);
                LOG.incrementProcessed((AbstractProgress)prog);
                iter.advance();
            }
        }
        LOG.ensureCompleted(prog);
        BasicOutlierScoreMeta meta = new BasicOutlierScoreMeta(minmax.getMin(), minmax.getMax(), 0.0, Double.POSITIVE_INFINITY);
        MaterializedDoubleRelation rel = new MaterializedDoubleRelation("HySortOD", relation.getDBIDs(), (DoubleDataStore)scores);
        return new OutlierResult(meta, (DoubleRelation)rel);
    }

    private List<Hypercube> getSortedHypercubes(Relation<? extends NumberVector> relation) {
        TreeSet<Hypercube> sorted = new TreeSet<Hypercube>();
        DBIDArrayIter iter = (DBIDArrayIter)relation.iterDBIDs();
        while (iter.valid()) {
            Hypercube h = new Hypercube((NumberVector)relation.get((DBIDRef)iter), this.l);
            if (!sorted.contains(h)) {
                h.add((DBIDRef)iter);
                sorted.add(h);
            } else {
                sorted.ceiling(h).add((DBIDRef)iter);
            }
            iter.advance();
        }
        return new ArrayList<Hypercube>(sorted);
    }

    private double score(int density) {
        return 1.0 - (double)density / this.strategy.getMaxDensity();
    }

    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{TypeUtil.NUMBER_VECTOR_FIELD});
    }

    public static class Par
    implements Parameterizer {
        public static final OptionID B_ID = new OptionID("hysortod.b", "Number of bins to use.");
        public static final OptionID MIN_SPLIT_ID = new OptionID("hysortod.minsplit", "Predefined threshold to use.");
        protected int b = 5;
        protected int minSplit = 100;

        public void configure(Parameterization config) {
            ((IntParameter)new IntParameter(B_ID, 5).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ONE_INT)).grab(config, x -> {
                this.b = x;
            });
            ((IntParameter)new IntParameter(MIN_SPLIT_ID, 100).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ZERO_INT)).grab(config, x -> {
                this.minSplit = x;
            });
        }

        public HySortOD make() {
            return new HySortOD(this.b, this.minSplit);
        }
    }

    private static class TreeStrategy
    extends DensityStrategy {
        Node root;
        final int minSplit;

        public TreeStrategy(int minSplit) {
            this.minSplit = minSplit;
        }

        @Override
        DensityStrategy buildIndex(List<Hypercube> H) {
            this.H = H;
            this.Wmax = 0;
            this.root = new Node(-1, 0, H.size() - 1);
            this.buildIndex(this.root, 0);
            return this;
        }

        private void buildIndex(Node parent, int col) {
            Node child;
            int i;
            if (parent.end - parent.begin < this.minSplit) {
                return;
            }
            int value = ((Hypercube)this.H.get(parent.begin)).getCoordAt(col);
            int begin = parent.begin;
            int end = -1;
            for (i = parent.begin; i <= parent.end; ++i) {
                if (((Hypercube)this.H.get(i)).getCoordAt(col) == value) continue;
                end = i - 1;
                child = new Node(value, begin, end);
                parent.add(child);
                this.buildIndex(child, col + 1);
                begin = i;
                value = ((Hypercube)this.H.get(i)).getCoordAt(col);
            }
            end = i - 1;
            child = new Node(value, begin, end);
            parent.add(child);
            this.buildIndex(child, col + 1);
        }

        @Override
        int[] getDensities() {
            int n = this.H.size();
            int[] W = new int[n];
            for (int i = 0; i < n; ++i) {
                W[i] = this.density(i, this.root, 0);
                this.Wmax = Math.max(this.Wmax, W[i]);
            }
            return W;
        }

        private int density(int i, Node parent, int col) {
            if (parent.children == null) {
                int density = 0;
                for (int k = parent.begin; k <= parent.end; ++k) {
                    if (!this.isImmediate((Hypercube)this.H.get(i), (Hypercube)this.H.get(k))) continue;
                    density += ((Hypercube)this.H.get(k)).getDensity();
                }
                return density;
            }
            int midVal = ((Hypercube)this.H.get(i)).getCoordAt(col);
            int nextCol = Math.min(col + 1, ((Hypercube)this.H.get(i)).getNumDimensions() - 1);
            Node lftNode = (Node)parent.children.get(midVal - 1);
            Node midNode = (Node)parent.children.get(midVal);
            Node rgtNode = (Node)parent.children.get(midVal + 1);
            return (lftNode != null ? this.density(i, lftNode, nextCol) : 0) + (midNode != null ? this.density(i, midNode, nextCol) : 0) + (rgtNode != null ? this.density(i, rgtNode, nextCol) : 0);
        }

        private static class Node {
            public final int value;
            public final int begin;
            public final int end;
            public Int2ObjectOpenHashMap<Node> children;

            public Node(int value, int begin, int end) {
                this.value = value;
                this.begin = begin;
                this.end = end;
            }

            public String toString() {
                return "(" + this.begin + "," + this.end + ")";
            }

            public void add(Node node) {
                if (node != null) {
                    if (this.children == null) {
                        this.children = new Int2ObjectOpenHashMap();
                    }
                    this.children.put(node.value, (Object)node);
                }
            }
        }
    }

    private static class NaiveStrategy
    extends DensityStrategy {
        private NaiveStrategy() {
        }

        @Override
        DensityStrategy buildIndex(List<Hypercube> H) {
            this.H = H;
            this.Wmax = 0;
            return this;
        }

        @Override
        public int[] getDensities() {
            int n = this.H.size();
            int[] W = new int[n];
            for (int i = 0; i < n; ++i) {
                int k;
                W[i] = ((Hypercube)this.H.get(i)).getDensity();
                for (k = i - 1; k >= 0 && this.isProspective((Hypercube)this.H.get(i), (Hypercube)this.H.get(k), 0); --k) {
                    if (!this.isImmediate((Hypercube)this.H.get(i), (Hypercube)this.H.get(k))) continue;
                    int n2 = i;
                    W[n2] = W[n2] + ((Hypercube)this.H.get(k)).getDensity();
                }
                for (k = i + 1; k < n && this.isProspective((Hypercube)this.H.get(i), (Hypercube)this.H.get(k), 0); ++k) {
                    if (!this.isImmediate((Hypercube)this.H.get(i), (Hypercube)this.H.get(k))) continue;
                    int n3 = i;
                    W[n3] = W[n3] + ((Hypercube)this.H.get(k)).getDensity();
                }
                this.Wmax = Math.max(this.Wmax, W[i]);
            }
            return W;
        }
    }

    private static abstract class DensityStrategy {
        int Wmax;
        List<Hypercube> H;

        private DensityStrategy() {
        }

        abstract DensityStrategy buildIndex(List<Hypercube> var1);

        abstract int[] getDensities();

        double getMaxDensity() {
            return this.Wmax;
        }

        protected boolean isImmediate(Hypercube hi, Hypercube hk) {
            int[] p = hi.getCoords();
            int[] q = hk.getCoords();
            for (int j = p.length - 1; j >= 0; --j) {
                if (Math.abs(p[j] - q[j]) <= 1) continue;
                return false;
            }
            return true;
        }

        protected boolean isProspective(Hypercube hi, Hypercube hk, int col) {
            return Math.abs(hi.getCoordAt(col) - hk.getCoordAt(col)) <= 1;
        }
    }

    private static class Hypercube
    implements Comparable<Hypercube> {
        final int[] coords;
        ArrayModifiableDBIDs instances;

        public Hypercube(NumberVector values, double length) {
            this.coords = new int[values.getDimensionality()];
            int[] coords = this.coords;
            for (int d = 0; d < coords.length; ++d) {
                coords[d] = (int)Math.floor(values.doubleValue(d) / length);
            }
        }

        @Override
        public int compareTo(Hypercube o) {
            for (int i = 0; i < this.coords.length; ++i) {
                int d = this.coords[i] - o.coords[i];
                if (d == 0) continue;
                return d;
            }
            return 0;
        }

        public boolean equals(Object obj) {
            return obj != null && this.getClass() == obj.getClass() && Arrays.equals(this.coords, ((Hypercube)obj).coords);
        }

        public int hashCode() {
            return Arrays.hashCode(this.coords) ^ this.instances.hashCode();
        }

        public String toString() {
            StringBuilder str = new StringBuilder(this.coords.length * 10 + 2).append("(").append(this.coords[0]);
            for (int i = 1; i < this.coords.length; ++i) {
                str.append(", ").append(this.coords[i]);
            }
            return str.append(")").toString();
        }

        public int getCoordAt(int j) {
            return this.coords[j];
        }

        public int[] getCoords() {
            return this.coords;
        }

        public int getNumDimensions() {
            return this.coords.length;
        }

        public DBIDs getInstances() {
            return this.instances;
        }

        public void add(DBIDRef instance) {
            if (this.instances == null) {
                this.instances = DBIDUtil.newArray();
            }
            this.instances.add(instance);
        }

        public int getDensity() {
            return this.instances.size();
        }
    }
}

