/*
 * Decompiled with CFR 0.152.
 */
package com.agentsflex.store.chroma;

import com.agentsflex.core.document.Document;
import com.agentsflex.core.llm.client.HttpClient;
import com.agentsflex.core.store.DocumentStore;
import com.agentsflex.core.store.SearchWrapper;
import com.agentsflex.core.store.StoreOptions;
import com.agentsflex.core.store.StoreResult;
import com.agentsflex.core.store.condition.ExpressionAdaptor;
import com.agentsflex.store.chroma.ChromaExpressionAdaptor;
import com.agentsflex.store.chroma.ChromaVectorStoreConfig;
import com.google.gson.Gson;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChromaVectorStore
extends DocumentStore {
    private static final Logger logger = LoggerFactory.getLogger(ChromaVectorStore.class);
    private final String baseUrl;
    private final String collectionName;
    private final String tenant;
    private final String database;
    private final ChromaVectorStoreConfig config;
    private final ExpressionAdaptor expressionAdaptor;
    private final HttpClient httpClient;
    private final int MAX_RETRIES = 3;
    private final long RETRY_INTERVAL_MS = 1000L;
    private static final String BASE_API = "/api/v2";

    public ChromaVectorStore(ChromaVectorStoreConfig config) {
        Objects.requireNonNull(config, "ChromaVectorStoreConfig cannot be null");
        this.baseUrl = config.getBaseUrl();
        this.tenant = config.getTenant();
        this.database = config.getDatabase();
        this.collectionName = config.getCollectionName();
        this.config = config;
        this.expressionAdaptor = ChromaExpressionAdaptor.DEFAULT;
        this.httpClient = this.createHttpClient();
        this.validateConfig();
        if (config.isAutoCreateCollection()) {
            try {
                this.ensureTenantAndDatabaseExists();
                this.ensureCollectionExists();
            }
            catch (Exception e) {
                logger.warn("Failed to ensure collection exists: {}. Will retry on first operation.", (Object)e.getMessage());
            }
        }
    }

    private HttpClient createHttpClient() {
        HttpClient client = new HttpClient();
        return client;
    }

    private void validateConfig() {
        if (this.baseUrl == null || this.baseUrl.isEmpty()) {
            throw new IllegalArgumentException("Base URL cannot be empty");
        }
        if (!this.baseUrl.startsWith("http://") && !this.baseUrl.startsWith("https://")) {
            throw new IllegalArgumentException("Base URL must start with http:// or https://");
        }
    }

    private void ensureTenantAndDatabaseExists() {
        try {
            if (this.tenant != null && !this.tenant.isEmpty()) {
                this.ensureTenantExists();
                if (this.database != null && !this.database.isEmpty()) {
                    this.ensureDatabaseExists();
                }
            }
        }
        catch (Exception e) {
            logger.error("Error ensuring tenant and database exist", (Throwable)e);
        }
    }

    private void ensureTenantExists() throws IOException {
        String tenantUrl = this.baseUrl + BASE_API + "/tenants/" + this.tenant;
        Map<String, String> headers = this.createHeaders();
        try {
            String responseBody = this.executeWithRetry(() -> this.httpClient.get(tenantUrl, headers));
            logger.debug("Successfully verified tenant '{}' exists", (Object)this.tenant);
        }
        catch (IOException e) {
            logger.info("Creating tenant '{}' as it does not exist", (Object)this.tenant);
            HashMap<String, Object> requestBody = new HashMap<String, Object>();
            requestBody.put("name", this.tenant);
            String createTenantUrl = this.baseUrl + BASE_API + "/tenants";
            String jsonRequestBody = this.safeJsonSerialize(requestBody);
            String responseBody = this.executeWithRetry(() -> this.httpClient.post(createTenantUrl, headers, jsonRequestBody));
            logger.info("Successfully created tenant '{}'", (Object)this.tenant);
        }
    }

    private void ensureDatabaseExists() throws IOException {
        if (this.tenant == null || this.tenant.isEmpty()) {
            throw new IllegalStateException("Cannot create database without tenant");
        }
        String databaseUrl = this.baseUrl + BASE_API + "/tenants/" + this.tenant + "/databases/" + this.database;
        Map<String, String> headers = this.createHeaders();
        try {
            String responseBody = this.executeWithRetry(() -> this.httpClient.get(databaseUrl, headers));
            logger.debug("Successfully verified database '{}' exists in tenant '{}'", (Object)this.database, (Object)this.tenant);
        }
        catch (IOException e) {
            logger.info("Creating database '{}' in tenant '{}' as it does not exist", (Object)this.database, (Object)this.tenant);
            HashMap<String, Object> requestBody = new HashMap<String, Object>();
            requestBody.put("name", this.database);
            String createDatabaseUrl = this.baseUrl + BASE_API + "/tenants/" + this.tenant + "/databases";
            String jsonRequestBody = this.safeJsonSerialize(requestBody);
            String responseBody = this.executeWithRetry(() -> this.httpClient.post(createDatabaseUrl, headers, jsonRequestBody));
            logger.info("Successfully created database '{}' in tenant '{}'", (Object)this.database, (Object)this.tenant);
        }
    }

    private String getCollectionId(String collectionName) throws IOException {
        Map<String, String> headers;
        String collectionsUrl = this.buildCollectionsUrl();
        String responseBody = this.executeWithRetry(() -> this.lambda$getCollectionId$4(collectionsUrl, headers = this.createHeaders()));
        if (responseBody == null) {
            throw new IOException("Failed to get collections, no response");
        }
        Object responseObj = this.parseJsonResponse(responseBody);
        List<Map> collections = new ArrayList();
        if (responseObj instanceof Map) {
            Map responseMap = (Map)responseObj;
            if (responseMap.containsKey("collections") && responseMap.get("collections") instanceof List) {
                collections = (List)responseMap.get("collections");
            }
        } else if (responseObj instanceof List) {
            List rawCollections = (List)responseObj;
            for (Object item : rawCollections) {
                if (!(item instanceof Map)) continue;
                collections.add((Map)item);
            }
        }
        for (Map collection : collections) {
            if (!collection.containsKey("name") || !collectionName.equals(collection.get("name"))) continue;
            return collection.get("id").toString();
        }
        throw new IOException("Collection not found: " + collectionName);
    }

    private void createCollection() throws IOException {
        String createCollectionUrl = this.buildCollectionsUrl();
        Map<String, String> headers = this.createHeaders();
        HashMap<String, Object> requestBody = new HashMap<String, Object>();
        requestBody.put("name", this.collectionName);
        String jsonRequestBody = this.safeJsonSerialize(requestBody);
        String responseBody = this.executeWithRetry(() -> this.httpClient.post(createCollectionUrl, headers, jsonRequestBody));
        if (responseBody == null) {
            throw new IOException("Failed to create collection: no response");
        }
        try {
            Object responseObj = this.parseJsonResponse(responseBody);
            Map responseMap = null;
            if (responseObj instanceof Map) {
                responseMap = (Map)responseObj;
            }
            if (responseMap.containsKey("error")) {
                throw new IOException("Failed to create collection: " + responseMap.get("error"));
            }
            logger.info("Collection '{}' created successfully", (Object)this.collectionName);
        }
        catch (Exception e) {
            throw new IOException("Failed to process collection creation response: " + e.getMessage(), e);
        }
    }

    public StoreResult storeInternal(List<Document> documents, StoreOptions options) {
        Objects.requireNonNull(documents, "Documents cannot be null");
        if (documents.isEmpty()) {
            logger.debug("No documents to store");
            return StoreResult.success();
        }
        try {
            this.ensureCollectionExists();
            String collectionName = this.getCollectionName(options);
            ArrayList<String> ids = new ArrayList<String>();
            ArrayList embeddings = new ArrayList();
            ArrayList metadatas = new ArrayList();
            ArrayList<String> documentsContent = new ArrayList<String>();
            for (Document doc : documents) {
                ids.add(String.valueOf(doc.getId()));
                if (doc.getVector() != null) {
                    List embedding = Arrays.stream(doc.getVector()).boxed().collect(Collectors.toList());
                    embeddings.add(embedding);
                } else {
                    embeddings.add(null);
                }
                HashMap metadata = doc.getMetadataMap() != null ? new HashMap(doc.getMetadataMap()) : new HashMap();
                metadatas.add(metadata);
                documentsContent.add(doc.getContent());
            }
            HashMap<String, Object> requestBody = new HashMap<String, Object>();
            requestBody.put("ids", ids);
            requestBody.put("embeddings", embeddings);
            requestBody.put("metadatas", metadatas);
            requestBody.put("documents", documentsContent);
            String collectionId = this.getCollectionId(collectionName);
            String collectionUrl = this.buildCollectionUrl(collectionId, "add");
            Map<String, String> headers = this.createHeaders();
            String jsonRequestBody = this.safeJsonSerialize(requestBody);
            logger.debug("Storing {} documents to collection '{}'", (Object)documents.size(), (Object)collectionName);
            String responseBody = this.executeWithRetry(() -> this.httpClient.post(collectionUrl, headers, jsonRequestBody));
            if (responseBody == null) {
                logger.error("Error storing documents: no response");
                return StoreResult.fail();
            }
            Object responseObj = this.parseJsonResponse(responseBody);
            Map responseMap = null;
            if (responseObj instanceof Map) {
                responseMap = (Map)responseObj;
            }
            if (responseMap.containsKey("error")) {
                String errorMsg = "Error storing documents: " + responseMap.get("error");
                logger.error(errorMsg);
                return StoreResult.fail();
            }
            logger.debug("Successfully stored {} documents", (Object)documents.size());
            return StoreResult.successWithIds(documents);
        }
        catch (Exception e) {
            logger.error("Error storing documents to Chroma", (Throwable)e);
            return StoreResult.fail();
        }
    }

    public StoreResult deleteInternal(Collection<?> ids, StoreOptions options) {
        Objects.requireNonNull(ids, "IDs cannot be null");
        if (ids.isEmpty()) {
            logger.debug("No IDs to delete");
            return StoreResult.success();
        }
        try {
            this.ensureCollectionExists();
            String collectionName = this.getCollectionName(options);
            List stringIds = ids.stream().map(Object::toString).collect(Collectors.toList());
            HashMap<String, Object> requestBody = new HashMap<String, Object>();
            requestBody.put("ids", stringIds);
            String collectionId = this.getCollectionId(collectionName);
            String collectionUrl = this.buildCollectionUrl(collectionId, "delete");
            Map<String, String> headers = this.createHeaders();
            String jsonRequestBody = this.safeJsonSerialize(requestBody);
            logger.debug("Deleting {} documents from collection '{}'", (Object)ids.size(), (Object)collectionName);
            String responseBody = this.executeWithRetry(() -> this.httpClient.post(collectionUrl, headers, jsonRequestBody));
            if (responseBody == null) {
                logger.error("Error deleting documents: no response");
                return StoreResult.fail();
            }
            Object responseObj = this.parseJsonResponse(responseBody);
            Map responseMap = null;
            if (responseObj instanceof Map) {
                responseMap = (Map)responseObj;
            }
            if (responseMap.containsKey("error")) {
                String errorMsg = "Error deleting documents: " + responseMap.get("error");
                logger.error(errorMsg);
                return StoreResult.fail();
            }
            logger.debug("Successfully deleted {} documents", (Object)ids.size());
            return StoreResult.success();
        }
        catch (Exception e) {
            logger.error("Error deleting documents from Chroma", (Throwable)e);
            return StoreResult.fail();
        }
    }

    public StoreResult updateInternal(List<Document> documents, StoreOptions options) {
        Objects.requireNonNull(documents, "Documents cannot be null");
        if (documents.isEmpty()) {
            logger.debug("No documents to update");
            return StoreResult.success();
        }
        try {
            StoreResult storeResult;
            List ids = documents.stream().map(Document::getId).collect(Collectors.toList());
            StoreResult deleteResult = this.deleteInternal(ids, options);
            if (!deleteResult.isSuccess()) {
                logger.warn("Delete failed during update operation: {}", (Object)deleteResult.toString());
            }
            if ((storeResult = this.storeInternal(documents, options)).isSuccess()) {
                logger.debug("Successfully updated {} documents", (Object)documents.size());
            }
            return storeResult;
        }
        catch (Exception e) {
            logger.error("Error updating documents in Chroma", (Throwable)e);
            return StoreResult.fail();
        }
    }

    public List<Document> searchInternal(SearchWrapper wrapper, StoreOptions options) {
        Objects.requireNonNull(wrapper, "SearchWrapper cannot be null");
        try {
            String jsonRequestBody;
            Map<String, String> headers;
            String collectionId;
            String collectionUrl;
            String responseBody;
            this.ensureCollectionExists();
            String collectionName = this.getCollectionName(options);
            int limit = wrapper.getMaxResults() > 0 ? wrapper.getMaxResults() : 10;
            HashMap<String, Object> requestBody = new HashMap<String, Object>();
            if (wrapper.getVector() == null && wrapper.getText() == null) {
                throw new IllegalArgumentException("Either vector or text must be provided for search");
            }
            if (wrapper.getVector() != null) {
                List queryEmbedding = Arrays.stream(wrapper.getVector()).boxed().collect(Collectors.toList());
                requestBody.put("query_embeddings", Collections.singletonList(queryEmbedding));
                logger.debug("Performing vector search with dimension: {}", (Object)queryEmbedding.size());
            } else if (wrapper.getText() != null) {
                requestBody.put("query_texts", Collections.singletonList(wrapper.getText()));
                logger.debug("Performing text search: {}", (Object)this.sanitizeLogString(wrapper.getText(), 100));
            }
            requestBody.put("n_results", limit);
            if (wrapper.getCondition() != null) {
                try {
                    String whereClause = this.expressionAdaptor.toCondition(wrapper.getCondition());
                    Object whereObj = this.parseJsonResponse(whereClause);
                    Map whereMap = null;
                    if (whereObj instanceof Map) {
                        whereMap = (Map)whereObj;
                    }
                    requestBody.put("where", whereMap);
                    logger.debug("Search with filter condition: {}", (Object)whereClause);
                }
                catch (Exception e) {
                    logger.warn("Failed to parse filter condition: {}, ignoring condition", (Object)e.getMessage());
                }
            }
            if ((responseBody = this.executeWithRetry(() -> this.lambda$searchInternal$8(collectionUrl = this.buildCollectionUrl(collectionId = this.getCollectionId(collectionName), "query"), headers = this.createHeaders(), jsonRequestBody = this.safeJsonSerialize(requestBody)))) == null) {
                logger.error("Error searching documents: no response");
                return Collections.emptyList();
            }
            Object responseObj = this.parseJsonResponse(responseBody);
            Map responseMap = null;
            if (responseObj instanceof Map) {
                responseMap = (Map)responseObj;
            }
            if (responseMap.containsKey("error")) {
                logger.error("Error searching documents: {}", responseMap.get("error"));
                return Collections.emptyList();
            }
            return this.parseSearchResults(responseMap);
        }
        catch (Exception e) {
            logger.error("Error searching documents in Chroma", (Throwable)e);
            return Collections.emptyList();
        }
    }

    public List<Document> searchInternal(double[] vector, int topK, StoreOptions options) {
        Objects.requireNonNull(vector, "Vector cannot be null");
        if (topK <= 0) {
            topK = 10;
        }
        try {
            this.ensureCollectionExists();
            String collectionName = this.getCollectionName(options);
            HashMap<String, Object> requestBody = new HashMap<String, Object>();
            List queryEmbedding = Arrays.stream(vector).boxed().collect(Collectors.toList());
            requestBody.put("query_embeddings", Collections.singletonList(queryEmbedding));
            requestBody.put("n_results", topK);
            String collectionId = this.getCollectionId(collectionName);
            String collectionUrl = this.buildCollectionUrl(collectionId, "query");
            Map<String, String> headers = this.createHeaders();
            String jsonRequestBody = this.safeJsonSerialize(requestBody);
            logger.debug("Performing direct vector search with dimension: {}", (Object)vector.length);
            String responseBody = this.executeWithRetry(() -> this.httpClient.post(collectionUrl, headers, jsonRequestBody));
            if (responseBody == null) {
                logger.error("Error searching documents: no response");
                return Collections.emptyList();
            }
            Object responseObj = this.parseJsonResponse(responseBody);
            Map responseMap = null;
            if (responseObj instanceof Map) {
                responseMap = (Map)responseObj;
            }
            if (responseMap.containsKey("error")) {
                logger.error("Error searching documents: {}", responseMap.get("error"));
                return Collections.emptyList();
            }
            return this.parseSearchResults(responseMap);
        }
        catch (Exception e) {
            logger.error("Error searching documents in Chroma", (Throwable)e);
            return Collections.emptyList();
        }
    }

    private List<Document> parseSearchResults(Map<String, Object> responseMap) {
        try {
            List ids = this.extractResultsFromNestedList(responseMap, "ids");
            List documents = this.extractResultsFromNestedList(responseMap, "documents");
            List metadatas = this.extractResultsFromNestedList(responseMap, "metadatas");
            List embeddings = this.extractResultsFromNestedList(responseMap, "embeddings");
            List distances = this.extractResultsFromNestedList(responseMap, "distances");
            if (ids == null || ids.isEmpty()) {
                logger.debug("No documents found in search results");
                return Collections.emptyList();
            }
            ArrayList<Document> resultDocs = new ArrayList<Document>();
            for (int i = 0; i < ids.size(); ++i) {
                Document doc = new Document();
                doc.setId(ids.get(i));
                if (documents != null && i < documents.size()) {
                    doc.setContent((String)documents.get(i));
                }
                if (metadatas != null && i < metadatas.size()) {
                    doc.setMetadataMap((Map)metadatas.get(i));
                }
                if (embeddings != null && i < embeddings.size() && embeddings.get(i) != null) {
                    double[] vector = ((List)embeddings.get(i)).stream().mapToDouble(Double::doubleValue).toArray();
                    doc.setVector(vector);
                }
                if (distances != null && i < distances.size()) {
                    double score = 1.0 - (Double)distances.get(i);
                    score = Math.max(0.0, Math.min(1.0, score));
                    doc.setScore(Double.valueOf(score));
                }
                resultDocs.add(doc);
            }
            logger.debug("Found {} documents in search results", (Object)resultDocs.size());
            return resultDocs;
        }
        catch (Exception e) {
            logger.error("Failed to parse search results", (Throwable)e);
            return Collections.emptyList();
        }
    }

    private <T> List<T> extractResultsFromNestedList(Map<String, Object> responseMap, String key) {
        try {
            if (!responseMap.containsKey(key)) {
                return null;
            }
            List outerList = (List)responseMap.get(key);
            if (outerList == null || outerList.isEmpty()) {
                return null;
            }
            return (List)outerList.get(0);
        }
        catch (Exception e) {
            logger.warn("Failed to extract '{}' from response: {}", (Object)key, (Object)e.getMessage());
            return null;
        }
    }

    private Map<String, String> createHeaders() {
        HashMap<String, String> headers = new HashMap<String, String>();
        headers.put("Content-Type", "application/json");
        if (this.config.getApiKey() != null && !this.config.getApiKey().isEmpty()) {
            headers.put("X-Chroma-Token", this.config.getApiKey());
        }
        if (this.tenant != null && !this.tenant.isEmpty()) {
            headers.put("X-Chroma-Tenant", this.tenant);
        }
        if (this.database != null && !this.database.isEmpty()) {
            headers.put("X-Chroma-Database", this.database);
        }
        return headers;
    }

    private <T> T executeWithRetry(HttpOperation<T> operation) throws IOException {
        int attempts = 0;
        IOException lastException = null;
        while (attempts < 3) {
            try {
                ++attempts;
                return operation.execute();
            }
            catch (IOException e) {
                lastException = e;
                if (attempts >= 3) {
                    throw new IOException("Operation failed after 3 attempts: " + e.getMessage(), e);
                }
                logger.warn("Operation failed (attempt {} of {}), retrying in {}ms: {}", new Object[]{attempts, 3, 1000L, e.getMessage()});
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    throw new IOException("Retry interrupted", ie);
                }
            }
        }
        throw lastException != null ? lastException : new IOException("Operation failed without exception");
    }

    private String safeJsonSerialize(Map<String, Object> map) {
        try {
            return new Gson().toJson(map);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to serialize request body to JSON", e);
        }
    }

    private Object parseJsonResponse(String json) {
        try {
            if (json == null || json.trim().isEmpty()) {
                return null;
            }
            if (json.trim().startsWith("[")) {
                return new Gson().fromJson(json, List.class);
            }
            return new Gson().fromJson(json, Map.class);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to parse JSON response: " + json, e);
        }
    }

    private String sanitizeLogString(String input, int maxLength) {
        if (input == null) {
            return null;
        }
        String sanitized = input.replaceAll("[\n\r]", " ");
        return sanitized.length() > maxLength ? sanitized.substring(0, maxLength) + "..." : sanitized;
    }

    private String getCollectionName(StoreOptions options) {
        return options != null ? options.getCollectionNameOrDefault(this.collectionName) : this.collectionName;
    }

    private String buildCollectionUrl(String collectionId, String operation) {
        StringBuilder urlBuilder = new StringBuilder(this.baseUrl).append(BASE_API);
        if (this.tenant != null && !this.tenant.isEmpty()) {
            urlBuilder.append("/tenants/").append(this.tenant);
            if (this.database != null && !this.database.isEmpty()) {
                urlBuilder.append("/databases/").append(this.database);
            }
        }
        urlBuilder.append("/collections/").append(collectionId).append("/").append(operation);
        return urlBuilder.toString();
    }

    public void close() {
        logger.info("Chroma client closed");
    }

    private void ensureCollectionExists() throws IOException {
        try {
            this.getCollectionId(this.collectionName);
            logger.debug("Collection '{}' exists", (Object)this.collectionName);
        }
        catch (IOException e) {
            logger.info("Collection '{}' does not exist, creating...", (Object)this.collectionName);
            this.createCollection();
            logger.info("Collection '{}' created successfully", (Object)this.collectionName);
        }
    }

    private String buildCollectionsUrl() {
        StringBuilder urlBuilder = new StringBuilder(this.baseUrl).append(BASE_API);
        if (this.tenant != null && !this.tenant.isEmpty()) {
            urlBuilder.append("/tenants/").append(this.tenant);
            if (this.database != null && !this.database.isEmpty()) {
                urlBuilder.append("/databases/").append(this.database);
            }
        }
        urlBuilder.append("/collections");
        return urlBuilder.toString();
    }

    private /* synthetic */ String lambda$searchInternal$8(String collectionUrl, Map headers, String jsonRequestBody) throws IOException {
        return this.httpClient.post(collectionUrl, headers, jsonRequestBody);
    }

    private /* synthetic */ String lambda$getCollectionId$4(String collectionsUrl, Map headers) throws IOException {
        return this.httpClient.get(collectionsUrl, headers);
    }

    private static interface HttpOperation<T> {
        public T execute() throws IOException;
    }
}

