/*
 * Decompiled with CFR 0.152.
 */
package de.siegmar.fastcsv.writer;

import de.siegmar.fastcsv.writer.LineDelimiter;
import de.siegmar.fastcsv.writer.QuoteStrategy;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.UncheckedIOException;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.Objects;
import java.util.StringJoiner;

public final class CsvWriter
implements Closeable {
    private static final char CR = '\r';
    private static final char LF = '\n';
    private final Writer writer;
    private final char fieldSeparator;
    private final char quoteCharacter;
    private final char commentCharacter;
    private final QuoteStrategy quoteStrategy;
    private final String lineDelimiter;
    private final boolean syncWriter;

    CsvWriter(Writer writer, char fieldSeparator, char quoteCharacter, char commentCharacter, QuoteStrategy quoteStrategy, LineDelimiter lineDelimiter, boolean syncWriter) {
        if (fieldSeparator == '\r' || fieldSeparator == '\n') {
            throw new IllegalArgumentException("fieldSeparator must not be a newline char");
        }
        if (quoteCharacter == '\r' || quoteCharacter == '\n') {
            throw new IllegalArgumentException("quoteCharacter must not be a newline char");
        }
        if (commentCharacter == '\r' || commentCharacter == '\n') {
            throw new IllegalArgumentException("commentCharacter must not be a newline char");
        }
        if (!this.allDiffers(fieldSeparator, quoteCharacter, commentCharacter)) {
            throw new IllegalArgumentException(String.format("Control characters must differ (fieldSeparator=%s, quoteCharacter=%s, commentCharacter=%s)", Character.valueOf(fieldSeparator), Character.valueOf(quoteCharacter), Character.valueOf(commentCharacter)));
        }
        this.writer = writer;
        this.fieldSeparator = fieldSeparator;
        this.quoteCharacter = quoteCharacter;
        this.commentCharacter = commentCharacter;
        this.quoteStrategy = Objects.requireNonNull(quoteStrategy);
        this.lineDelimiter = Objects.requireNonNull(lineDelimiter).toString();
        this.syncWriter = syncWriter;
    }

    private boolean allDiffers(char ... chars) {
        for (int i = 0; i < chars.length - 1; ++i) {
            if (chars[i] != chars[i + 1]) continue;
            return false;
        }
        return true;
    }

    public static CsvWriterBuilder builder() {
        return new CsvWriterBuilder();
    }

    public CsvWriter writeRow(Iterable<String> values) {
        try {
            boolean firstField = true;
            for (String value : values) {
                if (!firstField) {
                    this.writer.write(this.fieldSeparator);
                }
                this.writeInternal(value, firstField);
                firstField = false;
            }
            this.endRow();
            return this;
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public CsvWriter writeRow(String ... values) {
        try {
            for (int i = 0; i < values.length; ++i) {
                if (i > 0) {
                    this.writer.write(this.fieldSeparator);
                }
                this.writeInternal(values[i], i == 0);
            }
            this.endRow();
            return this;
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void writeInternal(String value, boolean firstField) throws IOException {
        if (value == null) {
            if (this.quoteStrategy == QuoteStrategy.ALWAYS) {
                this.writer.write(this.quoteCharacter);
                this.writer.write(this.quoteCharacter);
            }
            return;
        }
        if (value.isEmpty()) {
            if (this.quoteStrategy == QuoteStrategy.ALWAYS || this.quoteStrategy == QuoteStrategy.EMPTY) {
                this.writer.write(this.quoteCharacter);
                this.writer.write(this.quoteCharacter);
            }
            return;
        }
        int length = value.length();
        boolean needsQuotes = this.quoteStrategy == QuoteStrategy.ALWAYS;
        int nextDelimPos = -1;
        for (int i = 0; i < length; ++i) {
            char c = value.charAt(i);
            if (c == this.quoteCharacter) {
                needsQuotes = true;
                nextDelimPos = i;
                break;
            }
            if (needsQuotes || c != this.fieldSeparator && c != '\n' && c != '\r' && (!firstField || i != 0 || c != this.commentCharacter)) continue;
            needsQuotes = true;
        }
        if (needsQuotes) {
            this.writer.write(this.quoteCharacter);
        }
        if (nextDelimPos > -1) {
            this.writeEscaped(value, length, nextDelimPos);
        } else {
            this.writer.write(value, 0, length);
        }
        if (needsQuotes) {
            this.writer.write(this.quoteCharacter);
        }
    }

    private void writeEscaped(String value, int length, int nextDelimPos) throws IOException {
        int startPos = 0;
        block0: do {
            int len = nextDelimPos - startPos + 1;
            this.writer.write(value, startPos, len);
            this.writer.write(this.quoteCharacter);
            nextDelimPos = -1;
            for (int i = startPos += len; i < length; ++i) {
                if (value.charAt(i) != this.quoteCharacter) continue;
                nextDelimPos = i;
                continue block0;
            }
        } while (nextDelimPos > -1);
        if (length > startPos) {
            this.writer.write(value, startPos, length - startPos);
        }
    }

    public CsvWriter writeComment(String comment) {
        try {
            this.writer.write(this.commentCharacter);
            if (comment != null && !comment.isEmpty()) {
                this.writeCommentInternal(comment);
            }
            this.endRow();
            return this;
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void writeCommentInternal(String comment) throws IOException {
        int length = comment.length();
        int startPos = 0;
        boolean lastCharWasCR = false;
        for (int i = 0; i < comment.length(); ++i) {
            int len;
            char c = comment.charAt(i);
            if (c == '\r') {
                len = i - startPos;
                this.writer.write(comment, startPos, len);
                this.writer.write(this.lineDelimiter, 0, this.lineDelimiter.length());
                this.writer.write(this.commentCharacter);
                startPos += len + 1;
                lastCharWasCR = true;
                continue;
            }
            if (c == '\n') {
                if (lastCharWasCR) {
                    lastCharWasCR = false;
                    ++startPos;
                    continue;
                }
                len = i - startPos;
                this.writer.write(comment, startPos, len);
                this.writer.write(this.lineDelimiter, 0, this.lineDelimiter.length());
                this.writer.write(this.commentCharacter);
                startPos += len + 1;
                continue;
            }
            lastCharWasCR = false;
        }
        if (length > startPos) {
            this.writer.write(comment, startPos, length - startPos);
        }
    }

    private void endRow() throws IOException {
        this.writer.write(this.lineDelimiter, 0, this.lineDelimiter.length());
        if (this.syncWriter) {
            this.writer.flush();
        }
    }

    @Override
    public void close() throws IOException {
        this.writer.close();
    }

    public String toString() {
        return new StringJoiner(", ", CsvWriter.class.getSimpleName() + "[", "]").add("fieldSeparator=" + this.fieldSeparator).add("quoteCharacter=" + this.quoteCharacter).add("commentCharacter=" + this.commentCharacter).add("quoteStrategy=" + (Object)((Object)this.quoteStrategy)).add("lineDelimiter='" + this.lineDelimiter + "'").toString();
    }

    static final class FastBufferedWriter
    extends Writer {
        private final Writer writer;
        private final char[] buf;
        private int pos;

        FastBufferedWriter(Writer writer, int bufferSize) {
            this.writer = writer;
            this.buf = new char[bufferSize];
        }

        @Override
        public void write(int c) throws IOException {
            if (this.pos == this.buf.length) {
                this.flush();
            }
            this.buf[this.pos++] = (char)c;
        }

        @Override
        public void write(char[] cbuf, int off, int len) {
            throw new IllegalStateException("Not implemented");
        }

        @Override
        public void write(String str, int off, int len) throws IOException {
            if (this.pos + len >= this.buf.length) {
                this.flush();
                if (len >= this.buf.length) {
                    char[] tmp = new char[len];
                    str.getChars(off, off + len, tmp, 0);
                    this.writer.write(tmp, 0, len);
                    return;
                }
            }
            str.getChars(off, off + len, this.buf, this.pos);
            this.pos += len;
        }

        @Override
        public void flush() throws IOException {
            this.writer.write(this.buf, 0, this.pos);
            this.pos = 0;
        }

        @Override
        public void close() throws IOException {
            this.flush();
            this.writer.close();
        }
    }

    public static final class CsvWriterBuilder {
        private static final int DEFAULT_BUFFER_SIZE = 8192;
        private char fieldSeparator = (char)44;
        private char quoteCharacter = (char)34;
        private char commentCharacter = (char)35;
        private QuoteStrategy quoteStrategy = QuoteStrategy.REQUIRED;
        private LineDelimiter lineDelimiter = LineDelimiter.CRLF;
        private int bufferSize = 8192;

        CsvWriterBuilder() {
        }

        public CsvWriterBuilder fieldSeparator(char fieldSeparator) {
            this.fieldSeparator = fieldSeparator;
            return this;
        }

        public CsvWriterBuilder quoteCharacter(char quoteCharacter) {
            this.quoteCharacter = quoteCharacter;
            return this;
        }

        public CsvWriterBuilder commentCharacter(char commentCharacter) {
            this.commentCharacter = commentCharacter;
            return this;
        }

        public CsvWriterBuilder quoteStrategy(QuoteStrategy quoteStrategy) {
            this.quoteStrategy = quoteStrategy;
            return this;
        }

        public CsvWriterBuilder lineDelimiter(LineDelimiter lineDelimiter) {
            this.lineDelimiter = lineDelimiter;
            return this;
        }

        public CsvWriterBuilder bufferSize(int bufferSize) {
            if (bufferSize < 0) {
                throw new IllegalArgumentException("buffer size must be >= 0");
            }
            this.bufferSize = bufferSize;
            return this;
        }

        public CsvWriter build(Writer writer) {
            Objects.requireNonNull(writer, "writer must not be null");
            return this.newWriter(writer, true);
        }

        public CsvWriter build(Path path, OpenOption ... openOptions) throws IOException {
            return this.build(path, StandardCharsets.UTF_8, openOptions);
        }

        public CsvWriter build(Path path, Charset charset, OpenOption ... openOptions) throws IOException {
            Objects.requireNonNull(path, "path must not be null");
            Objects.requireNonNull(charset, "charset must not be null");
            return this.newWriter(new OutputStreamWriter(Files.newOutputStream(path, openOptions), charset), false);
        }

        private CsvWriter newWriter(Writer writer, boolean syncWriter) {
            if (this.bufferSize > 0) {
                return new CsvWriter(new FastBufferedWriter(writer, this.bufferSize), this.fieldSeparator, this.quoteCharacter, this.commentCharacter, this.quoteStrategy, this.lineDelimiter, syncWriter);
            }
            return new CsvWriter(writer, this.fieldSeparator, this.quoteCharacter, this.commentCharacter, this.quoteStrategy, this.lineDelimiter, false);
        }

        public String toString() {
            return new StringJoiner(", ", CsvWriterBuilder.class.getSimpleName() + "[", "]").add("fieldSeparator=" + this.fieldSeparator).add("quoteCharacter=" + this.quoteCharacter).add("commentCharacter=" + this.commentCharacter).add("quoteStrategy=" + (Object)((Object)this.quoteStrategy)).add("lineDelimiter=" + (Object)((Object)this.lineDelimiter)).add("bufferSize=" + this.bufferSize).toString();
        }
    }
}

