/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import org.apache.kafka.clients.admin.AlterConfigOp;
import org.apache.kafka.clients.admin.ConfigEntry;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.config.ConfigResource;
import org.apache.kafka.common.metadata.ConfigRecord;
import org.apache.kafka.common.metadata.MetadataRecordType;
import org.apache.kafka.common.protocol.ApiMessage;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.ApiError;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.controller.ConfigurationValidator;
import org.apache.kafka.controller.ControllerResult;
import org.apache.kafka.controller.ResultOrError;
import org.apache.kafka.metadata.KafkaConfigSchema;
import org.apache.kafka.server.common.ApiMessageAndVersion;
import org.apache.kafka.server.policy.AlterConfigPolicy;
import org.apache.kafka.timeline.SnapshotRegistry;
import org.apache.kafka.timeline.TimelineHashMap;
import org.slf4j.Logger;

public class ConfigurationControlManager {
    public static final ConfigResource DEFAULT_NODE = new ConfigResource(ConfigResource.Type.BROKER, "");
    private final Logger log;
    private final SnapshotRegistry snapshotRegistry;
    private final KafkaConfigSchema configSchema;
    private final Consumer<ConfigResource> existenceChecker;
    private final Optional<AlterConfigPolicy> alterConfigPolicy;
    private final ConfigurationValidator validator;
    private final TimelineHashMap<ConfigResource, TimelineHashMap<String, String>> configData;
    private final Map<String, Object> staticConfig;
    private final ConfigResource currentController;

    private ConfigurationControlManager(LogContext logContext, SnapshotRegistry snapshotRegistry, KafkaConfigSchema configSchema, Consumer<ConfigResource> existenceChecker, Optional<AlterConfigPolicy> alterConfigPolicy, ConfigurationValidator validator, Map<String, Object> staticConfig, int nodeId) {
        this.log = logContext.logger(ConfigurationControlManager.class);
        this.snapshotRegistry = snapshotRegistry;
        this.configSchema = configSchema;
        this.existenceChecker = existenceChecker;
        this.alterConfigPolicy = alterConfigPolicy;
        this.validator = validator;
        this.configData = new TimelineHashMap(snapshotRegistry, 0);
        this.staticConfig = Collections.unmodifiableMap(new HashMap<String, Object>(staticConfig));
        this.currentController = new ConfigResource(ConfigResource.Type.BROKER, Integer.toString(nodeId));
    }

    SnapshotRegistry snapshotRegistry() {
        return this.snapshotRegistry;
    }

    ControllerResult<Map<ConfigResource, ApiError>> incrementalAlterConfigs(Map<ConfigResource, Map<String, Map.Entry<AlterConfigOp.OpType, String>>> configChanges, boolean newlyCreatedResource) {
        ArrayList<ApiMessageAndVersion> outputRecords = new ArrayList<ApiMessageAndVersion>();
        HashMap<ConfigResource, ApiError> outputResults = new HashMap<ConfigResource, ApiError>();
        for (Map.Entry<ConfigResource, Map<String, Map.Entry<AlterConfigOp.OpType, String>>> resourceEntry : configChanges.entrySet()) {
            this.incrementalAlterConfigResource(resourceEntry.getKey(), resourceEntry.getValue(), newlyCreatedResource, outputRecords, outputResults);
        }
        return ControllerResult.atomicOf(outputRecords, outputResults);
    }

    private void incrementalAlterConfigResource(ConfigResource configResource, Map<String, Map.Entry<AlterConfigOp.OpType, String>> keysToOps, boolean newlyCreatedResource, List<ApiMessageAndVersion> outputRecords, Map<ConfigResource, ApiError> outputResults) {
        ArrayList<ApiMessageAndVersion> newRecords = new ArrayList<ApiMessageAndVersion>();
        for (Map.Entry<String, Map.Entry<AlterConfigOp.OpType, String>> keysToOpsEntry : keysToOps.entrySet()) {
            String key = keysToOpsEntry.getKey();
            String currentValue = null;
            TimelineHashMap<String, String> currentConfigs = this.configData.get(configResource);
            if (currentConfigs != null) {
                currentValue = currentConfigs.get(key);
            }
            String newValue = currentValue;
            Map.Entry<AlterConfigOp.OpType, String> opTypeAndNewValue = keysToOpsEntry.getValue();
            AlterConfigOp.OpType opType = opTypeAndNewValue.getKey();
            String opValue = opTypeAndNewValue.getValue();
            switch (opType) {
                case SET: {
                    newValue = opValue;
                    break;
                }
                case DELETE: {
                    newValue = null;
                    break;
                }
                case APPEND: 
                case SUBTRACT: {
                    if (!this.configSchema.isSplittable(configResource.type(), key)) {
                        outputResults.put(configResource, new ApiError(Errors.INVALID_CONFIG, "Can't " + opType + " to key " + key + " because its type is not LIST."));
                        return;
                    }
                    List<String> oldValueList = this.getParts(newValue, key, configResource);
                    if (opType == AlterConfigOp.OpType.APPEND) {
                        for (String value : opValue.split(",")) {
                            if (oldValueList.contains(value)) continue;
                            oldValueList.add(value);
                        }
                    } else {
                        for (String value : opValue.split(",")) {
                            oldValueList.remove(value);
                        }
                    }
                    newValue = String.join((CharSequence)",", oldValueList);
                }
            }
            if (Objects.equals(currentValue, newValue) && !configResource.type().equals((Object)ConfigResource.Type.BROKER)) continue;
            newRecords.add(new ApiMessageAndVersion((ApiMessage)new ConfigRecord().setResourceType(configResource.type().id()).setResourceName(configResource.name()).setName(key).setValue(newValue), MetadataRecordType.CONFIG_RECORD.highestSupportedVersion()));
        }
        ApiError error = this.validateAlterConfig(configResource, newRecords, Collections.emptyList(), newlyCreatedResource);
        if (error.isFailure()) {
            outputResults.put(configResource, error);
            return;
        }
        outputRecords.addAll(newRecords);
        outputResults.put(configResource, ApiError.NONE);
    }

    private ApiError validateAlterConfig(ConfigResource configResource, List<ApiMessageAndVersion> recordsExplicitlyAltered, List<ApiMessageAndVersion> recordsImplicitlyDeleted, boolean newlyCreatedResource) {
        ConfigRecord configRecord;
        HashMap<String, String> allConfigs = new HashMap<String, String>();
        HashMap<String, String> alteredConfigsForAlterConfigPolicyCheck = new HashMap<String, String>();
        TimelineHashMap<String, String> existingConfigs = this.configData.get(configResource);
        if (existingConfigs != null) {
            allConfigs.putAll(existingConfigs);
        }
        for (ApiMessageAndVersion newRecord : recordsExplicitlyAltered) {
            configRecord = (ConfigRecord)newRecord.message();
            if (configRecord.value() == null) {
                allConfigs.remove(configRecord.name());
            } else {
                allConfigs.put(configRecord.name(), configRecord.value());
            }
            alteredConfigsForAlterConfigPolicyCheck.put(configRecord.name(), configRecord.value());
        }
        for (ApiMessageAndVersion recordImplicitlyDeleted : recordsImplicitlyDeleted) {
            configRecord = (ConfigRecord)recordImplicitlyDeleted.message();
            allConfigs.remove(configRecord.name());
        }
        try {
            this.validator.validate(configResource, allConfigs);
            if (!newlyCreatedResource) {
                this.existenceChecker.accept(configResource);
            }
            if (this.alterConfigPolicy.isPresent()) {
                this.alterConfigPolicy.get().validate(new AlterConfigPolicy.RequestMetadata(configResource, alteredConfigsForAlterConfigPolicyCheck));
            }
        }
        catch (ConfigException e) {
            return new ApiError(Errors.INVALID_CONFIG, e.getMessage());
        }
        catch (Throwable e) {
            ApiError apiError = ApiError.fromThrowable((Throwable)e);
            if (apiError.error() == Errors.UNKNOWN_SERVER_ERROR) {
                this.log.error("Unknown server error validating Alter Configs", e);
            }
            return apiError;
        }
        return ApiError.NONE;
    }

    ControllerResult<Map<ConfigResource, ApiError>> legacyAlterConfigs(Map<ConfigResource, Map<String, String>> newConfigs, boolean newlyCreatedResource) {
        ArrayList<ApiMessageAndVersion> outputRecords = new ArrayList<ApiMessageAndVersion>();
        HashMap<ConfigResource, ApiError> outputResults = new HashMap<ConfigResource, ApiError>();
        for (Map.Entry<ConfigResource, Map<String, String>> resourceEntry : newConfigs.entrySet()) {
            this.legacyAlterConfigResource(resourceEntry.getKey(), resourceEntry.getValue(), newlyCreatedResource, outputRecords, outputResults);
        }
        return ControllerResult.atomicOf(outputRecords, outputResults);
    }

    private void legacyAlterConfigResource(ConfigResource configResource, Map<String, String> newConfigs, boolean newlyCreatedResource, List<ApiMessageAndVersion> outputRecords, Map<ConfigResource, ApiError> outputResults) {
        ArrayList<ApiMessageAndVersion> recordsExplicitlyAltered = new ArrayList<ApiMessageAndVersion>();
        Map currentConfigs = this.configData.get(configResource);
        if (currentConfigs == null) {
            currentConfigs = Collections.emptyMap();
        }
        for (Map.Entry<String, String> entry : newConfigs.entrySet()) {
            String key = entry.getKey();
            String newValue = entry.getValue();
            String currentValue = (String)currentConfigs.get(key);
            if (Objects.equals(currentValue, newValue) && !configResource.type().equals((Object)ConfigResource.Type.BROKER)) continue;
            recordsExplicitlyAltered.add(new ApiMessageAndVersion((ApiMessage)new ConfigRecord().setResourceType(configResource.type().id()).setResourceName(configResource.name()).setName(key).setValue(newValue), MetadataRecordType.CONFIG_RECORD.highestSupportedVersion()));
        }
        ArrayList<ApiMessageAndVersion> recordsImplicitlyDeleted = new ArrayList<ApiMessageAndVersion>();
        for (String key : currentConfigs.keySet()) {
            if (newConfigs.containsKey(key)) continue;
            recordsImplicitlyDeleted.add(new ApiMessageAndVersion((ApiMessage)new ConfigRecord().setResourceType(configResource.type().id()).setResourceName(configResource.name()).setName(key).setValue(null), MetadataRecordType.CONFIG_RECORD.highestSupportedVersion()));
        }
        ApiError apiError = this.validateAlterConfig(configResource, recordsExplicitlyAltered, recordsImplicitlyDeleted, newlyCreatedResource);
        if (apiError.isFailure()) {
            outputResults.put(configResource, apiError);
            return;
        }
        outputRecords.addAll(recordsExplicitlyAltered);
        outputRecords.addAll(recordsImplicitlyDeleted);
        outputResults.put(configResource, ApiError.NONE);
    }

    private List<String> getParts(String value, String key, ConfigResource configResource) {
        String[] splitValues;
        if (value == null) {
            value = this.configSchema.getDefault(configResource.type(), key);
        }
        ArrayList<String> parts = new ArrayList<String>();
        if (value == null) {
            return parts;
        }
        for (String splitValue : splitValues = value.split(",")) {
            if (splitValue.isEmpty()) continue;
            parts.add(splitValue);
        }
        return parts;
    }

    public void replay(ConfigRecord record) {
        ConfigResource.Type type = ConfigResource.Type.forId((byte)record.resourceType());
        ConfigResource configResource = new ConfigResource(type, record.resourceName());
        TimelineHashMap<String, String> configs = this.configData.get(configResource);
        if (configs == null) {
            configs = new TimelineHashMap(this.snapshotRegistry, 0);
            this.configData.put(configResource, configs);
        }
        if (record.value() == null) {
            configs.remove(record.name());
        } else {
            configs.put(record.name(), record.value());
        }
        if (configs.isEmpty()) {
            this.configData.remove(configResource);
        }
        if (this.configSchema.isSensitive(record)) {
            this.log.info("{}: set configuration {} to {}", new Object[]{configResource, record.name(), "[hidden]"});
        } else {
            this.log.info("{}: set configuration {} to {}", new Object[]{configResource, record.name(), record.value()});
        }
    }

    Map<String, String> getConfigs(ConfigResource configResource) {
        Map map = this.configData.get(configResource);
        if (map == null) {
            return Collections.emptyMap();
        }
        return Collections.unmodifiableMap(new HashMap(map));
    }

    public Map<ConfigResource, ResultOrError<Map<String, String>>> describeConfigs(long lastCommittedOffset, Map<ConfigResource, Collection<String>> resources) {
        HashMap<ConfigResource, ResultOrError<Map<String, String>>> results = new HashMap<ConfigResource, ResultOrError<Map<String, String>>>();
        for (Map.Entry<ConfigResource, Collection<String>> resourceEntry : resources.entrySet()) {
            ConfigResource resource = resourceEntry.getKey();
            try {
                this.validator.validate(resource);
            }
            catch (Throwable e) {
                results.put(resource, new ResultOrError<ApiError>(ApiError.fromThrowable((Throwable)e)));
                continue;
            }
            HashMap<String, String> foundConfigs = new HashMap<String, String>();
            TimelineHashMap<String, String> configs = this.configData.get(resource, lastCommittedOffset);
            if (configs != null) {
                Collection<String> targetConfigs = resourceEntry.getValue();
                if (targetConfigs.isEmpty()) {
                    for (Map.Entry<String, String> entry : configs.entrySet(lastCommittedOffset)) {
                        foundConfigs.put(entry.getKey(), entry.getValue());
                    }
                } else {
                    for (String key : targetConfigs) {
                        String value = configs.get(key, lastCommittedOffset);
                        if (value == null) continue;
                        foundConfigs.put(key, value);
                    }
                }
            }
            results.put(resource, new ResultOrError(foundConfigs));
        }
        return results;
    }

    void deleteTopicConfigs(String name) {
        this.configData.remove(new ConfigResource(ConfigResource.Type.TOPIC, name));
    }

    boolean uncleanLeaderElectionEnabledForTopic(String name) {
        return false;
    }

    Map<String, ConfigEntry> computeEffectiveTopicConfigs(Map<String, String> creationConfigs) {
        return this.configSchema.resolveEffectiveTopicConfigs(this.staticConfig, this.clusterConfig(), this.currentControllerConfig(), creationConfigs);
    }

    Map<String, String> clusterConfig() {
        Map result = this.configData.get(DEFAULT_NODE);
        return result == null ? Collections.emptyMap() : result;
    }

    Map<String, String> currentControllerConfig() {
        Map result = this.configData.get(this.currentController);
        return result == null ? Collections.emptyMap() : result;
    }

    ConfigurationControlIterator iterator(long epoch) {
        return new ConfigurationControlIterator(epoch);
    }

    class ConfigurationControlIterator
    implements Iterator<List<ApiMessageAndVersion>> {
        private final long epoch;
        private final Iterator<Map.Entry<ConfigResource, TimelineHashMap<String, String>>> iterator;

        ConfigurationControlIterator(long epoch) {
            this.epoch = epoch;
            this.iterator = ConfigurationControlManager.this.configData.entrySet(epoch).iterator();
        }

        @Override
        public boolean hasNext() {
            return this.iterator.hasNext();
        }

        @Override
        public List<ApiMessageAndVersion> next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            ArrayList<ApiMessageAndVersion> records = new ArrayList<ApiMessageAndVersion>();
            Map.Entry<ConfigResource, TimelineHashMap<String, String>> entry = this.iterator.next();
            ConfigResource resource = entry.getKey();
            for (Map.Entry<String, String> configEntry : entry.getValue().entrySet(this.epoch)) {
                records.add(new ApiMessageAndVersion((ApiMessage)new ConfigRecord().setResourceName(resource.name()).setResourceType(resource.type().id()).setName(configEntry.getKey()).setValue(configEntry.getValue()), MetadataRecordType.CONFIG_RECORD.highestSupportedVersion()));
            }
            return records;
        }
    }

    static class Builder {
        private LogContext logContext = null;
        private SnapshotRegistry snapshotRegistry = null;
        private KafkaConfigSchema configSchema = KafkaConfigSchema.EMPTY;
        private Consumer<ConfigResource> existenceChecker = __ -> {};
        private Optional<AlterConfigPolicy> alterConfigPolicy = Optional.empty();
        private ConfigurationValidator validator = ConfigurationValidator.NO_OP;
        private Map<String, Object> staticConfig = Collections.emptyMap();
        private int nodeId = 0;

        Builder() {
        }

        Builder setLogContext(LogContext logContext) {
            this.logContext = logContext;
            return this;
        }

        Builder setSnapshotRegistry(SnapshotRegistry snapshotRegistry) {
            this.snapshotRegistry = snapshotRegistry;
            return this;
        }

        Builder setKafkaConfigSchema(KafkaConfigSchema configSchema) {
            this.configSchema = configSchema;
            return this;
        }

        Builder setExistenceChecker(Consumer<ConfigResource> existenceChecker) {
            this.existenceChecker = existenceChecker;
            return this;
        }

        Builder setAlterConfigPolicy(Optional<AlterConfigPolicy> alterConfigPolicy) {
            this.alterConfigPolicy = alterConfigPolicy;
            return this;
        }

        Builder setValidator(ConfigurationValidator validator) {
            this.validator = validator;
            return this;
        }

        Builder setStaticConfig(Map<String, Object> staticConfig) {
            this.staticConfig = staticConfig;
            return this;
        }

        Builder setNodeId(int nodeId) {
            this.nodeId = nodeId;
            return this;
        }

        ConfigurationControlManager build() {
            if (this.logContext == null) {
                this.logContext = new LogContext();
            }
            if (this.snapshotRegistry == null) {
                this.snapshotRegistry = new SnapshotRegistry(this.logContext);
            }
            return new ConfigurationControlManager(this.logContext, this.snapshotRegistry, this.configSchema, this.existenceChecker, this.alterConfigPolicy, this.validator, this.staticConfig, this.nodeId);
        }
    }
}

