/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.security.DigestInputStream;
import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.util.Arrays;
import org.apache.commons.logging.Log;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.permission.PermissionStatus;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.FSConstants;
import org.apache.hadoop.hdfs.protocol.LayoutVersion;
import org.apache.hadoop.hdfs.server.common.Util;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.FSImage;
import org.apache.hadoop.hdfs.server.namenode.FSImageCompression;
import org.apache.hadoop.hdfs.server.namenode.FSImageSerialization;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodeFileUnderConstruction;
import org.apache.hadoop.io.MD5Hash;
import org.apache.hadoop.io.Text;

@InterfaceAudience.Private
@InterfaceStability.Evolving
class FSImageFormat {
    private static final Log LOG = FSImage.LOG;

    private FSImageFormat() {
    }

    static class Saver {
        private boolean saved = false;
        private MD5Hash savedDigest;
        private static final byte[] PATH_SEPARATOR = DFSUtil.string2Bytes("/");

        Saver() {
        }

        private void checkSaved() {
            if (!this.saved) {
                throw new IllegalStateException("FSImageSaver has not saved an image");
            }
        }

        private void checkNotSaved() {
            if (this.saved) {
                throw new IllegalStateException("FSImageSaver has already saved an image");
            }
        }

        MD5Hash getSavedDigest() {
            this.checkSaved();
            return this.savedDigest;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void save(File newFile, FSNamesystem sourceNamesystem, FSImageCompression compression) throws IOException {
            this.checkNotSaved();
            FSDirectory fsDir = sourceNamesystem.dir;
            long startTime = Util.now();
            MessageDigest digester = MD5Hash.getDigester();
            FileOutputStream fout = new FileOutputStream(newFile);
            DigestOutputStream fos = new DigestOutputStream(fout, digester);
            DataOutputStream out = new DataOutputStream(fos);
            try {
                out.writeInt(FSConstants.LAYOUT_VERSION);
                out.writeInt(sourceNamesystem.getFSImage().getNamespaceID());
                out.writeLong(fsDir.rootDir.numItemsInTree());
                out.writeLong(sourceNamesystem.getGenerationStamp());
                out = compression.writeHeaderAndWrapStream(fos);
                LOG.info((Object)("Saving image file " + newFile + " using " + compression));
                byte[] byteStore = new byte[32000];
                ByteBuffer strbuf = ByteBuffer.wrap(byteStore);
                FSImageSerialization.saveINode2Image(strbuf, fsDir.rootDir, out);
                Saver.saveImage(strbuf, 0, fsDir.rootDir, out);
                sourceNamesystem.saveFilesUnderConstruction(out);
                sourceNamesystem.saveSecretManagerState(out);
                strbuf = null;
                out.flush();
                fout.getChannel().force(true);
            }
            finally {
                out.close();
            }
            this.saved = true;
            this.savedDigest = new MD5Hash(digester.digest());
            LOG.info((Object)("Image file of size " + newFile.length() + " saved in " + (Util.now() - startTime) / 1000L + " seconds."));
        }

        private static void saveImage(ByteBuffer parentPrefix, int prefixLength, INodeDirectory current, DataOutputStream out) throws IOException {
            int newPrefixLength = prefixLength;
            if (current.getChildrenRaw() == null) {
                return;
            }
            for (INode child : current.getChildren()) {
                parentPrefix.position(prefixLength);
                parentPrefix.put(PATH_SEPARATOR).put(child.getLocalNameBytes());
                FSImageSerialization.saveINode2Image(parentPrefix, child, out);
            }
            for (INode child : current.getChildren()) {
                if (!child.isDirectory()) continue;
                parentPrefix.position(prefixLength);
                parentPrefix.put(PATH_SEPARATOR).put(child.getLocalNameBytes());
                newPrefixLength = parentPrefix.position();
                Saver.saveImage(parentPrefix, newPrefixLength, (INodeDirectory)child, out);
            }
            parentPrefix.position(prefixLength);
        }
    }

    static class Loader {
        private final Configuration conf;
        private boolean loaded = false;
        private int imgVersion;
        private int imgNamespaceID;
        private MD5Hash imgDigest;

        Loader(Configuration conf) {
            this.conf = conf;
        }

        int getLoadedImageVersion() {
            this.checkLoaded();
            return this.imgVersion;
        }

        MD5Hash getLoadedImageMd5() {
            this.checkLoaded();
            return this.imgDigest;
        }

        int getLoadedNamespaceID() {
            this.checkLoaded();
            return this.imgNamespaceID;
        }

        private void checkLoaded() {
            if (!this.loaded) {
                throw new IllegalStateException("Image not yet loaded!");
            }
        }

        private void checkNotLoaded() {
            if (this.loaded) {
                throw new IllegalStateException("Image already loaded!");
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void load(File curFile, FSNamesystem targetNamesystem) throws IOException {
            this.checkNotLoaded();
            assert (curFile != null) : "curFile is null";
            long startTime = Util.now();
            FSDirectory fsDir = targetNamesystem.dir;
            MessageDigest digester = MD5Hash.getDigester();
            DigestInputStream fin = new DigestInputStream(new FileInputStream(curFile), digester);
            DataInputStream in = new DataInputStream(fin);
            try {
                this.imgVersion = in.readInt();
                this.imgNamespaceID = in.readInt();
                long numFiles = this.readNumFiles(in);
                if (this.imgVersion <= -12) {
                    long genstamp = in.readLong();
                    targetNamesystem.setGenerationStamp(genstamp);
                }
                FSImageCompression compression = LayoutVersion.supports(LayoutVersion.Feature.FSIMAGE_COMPRESSION, this.imgVersion) ? FSImageCompression.readCompressionHeader(this.conf, in) : FSImageCompression.createNoopCompression();
                in = compression.unwrapInputStream(fin);
                LOG.info((Object)("Loading image file " + curFile + " using " + compression));
                short replication = targetNamesystem.getDefaultReplication();
                LOG.info((Object)("Number of files = " + numFiles));
                Object parentPath = new byte[][]{new byte[0]};
                INodeDirectory parentINode = fsDir.rootDir;
                for (long i = 0L; i < numFiles; ++i) {
                    long modificationTime = 0L;
                    long atime = 0L;
                    long blockSize = 0L;
                    byte[][] pathComponents = FSImageSerialization.readPathComponents(in);
                    replication = in.readShort();
                    replication = targetNamesystem.adjustReplication(replication);
                    modificationTime = in.readLong();
                    if (LayoutVersion.supports(LayoutVersion.Feature.FILE_ACCESS_TIME, this.imgVersion)) {
                        atime = in.readLong();
                    }
                    if (this.imgVersion <= -8) {
                        blockSize = in.readLong();
                    }
                    int numBlocks = in.readInt();
                    Block[] blocks = null;
                    if (-9 <= this.imgVersion && numBlocks > 0 || this.imgVersion < -9 && numBlocks >= 0) {
                        blocks = new Block[numBlocks];
                        for (int j = 0; j < numBlocks; ++j) {
                            blocks[j] = new Block();
                            if (-14 < this.imgVersion) {
                                blocks[j].set(in.readLong(), in.readLong(), 0L);
                                continue;
                            }
                            blocks[j].readFields(in);
                        }
                    }
                    if (-8 <= this.imgVersion && blockSize == 0L) {
                        if (numBlocks > 1) {
                            blockSize = blocks[0].getNumBytes();
                        } else {
                            long first = numBlocks == 1 ? blocks[0].getNumBytes() : 0L;
                            blockSize = Math.max(targetNamesystem.getDefaultBlockSize(), first);
                        }
                    }
                    long nsQuota = -1L;
                    if (LayoutVersion.supports(LayoutVersion.Feature.NAMESPACE_QUOTA, this.imgVersion) && blocks == null && numBlocks == -1) {
                        nsQuota = in.readLong();
                    }
                    long dsQuota = -1L;
                    if (LayoutVersion.supports(LayoutVersion.Feature.DISKSPACE_QUOTA, this.imgVersion) && blocks == null && numBlocks == -1) {
                        dsQuota = in.readLong();
                    }
                    String symlink = "";
                    if (numBlocks == -2) {
                        symlink = Text.readString((DataInput)in);
                    }
                    PermissionStatus permissions = targetNamesystem.getUpgradePermission();
                    if (this.imgVersion <= -11) {
                        permissions = PermissionStatus.read((DataInput)in);
                    }
                    if (this.isRoot(pathComponents)) {
                        if (nsQuota != -1L || dsQuota != -1L) {
                            fsDir.rootDir.setQuota(nsQuota, dsQuota);
                        }
                        fsDir.rootDir.setModificationTime(modificationTime);
                        fsDir.rootDir.setPermissionStatus(permissions);
                        continue;
                    }
                    if (!this.isParent(pathComponents, (byte[][])parentPath)) {
                        parentINode = null;
                        parentPath = this.getParent(pathComponents);
                    }
                    parentINode = fsDir.addToParent(pathComponents, parentINode, permissions, blocks, symlink, replication, modificationTime, atime, nsQuota, dsQuota, blockSize, false);
                }
                this.loadDatanodes(in);
                this.loadFilesUnderConstruction(in, targetNamesystem);
                this.loadSecretManagerState(in, targetNamesystem);
                int eof = in.read();
                assert (eof == -1) : "Should have reached the end of image file " + curFile;
            }
            finally {
                in.close();
            }
            this.imgDigest = new MD5Hash(digester.digest());
            this.loaded = true;
            LOG.info((Object)("Image file of size " + curFile.length() + " loaded in " + (Util.now() - startTime) / 1000L + " seconds."));
        }

        private void loadDatanodes(DataInputStream in) throws IOException {
            if (this.imgVersion > -3) {
                return;
            }
            if (this.imgVersion <= -12) {
                return;
            }
            int size = in.readInt();
            for (int i = 0; i < size; ++i) {
                FSImageSerialization.DatanodeImage.skipOne(in);
            }
        }

        private void loadFilesUnderConstruction(DataInputStream in, FSNamesystem fs) throws IOException {
            FSDirectory fsDir = fs.dir;
            if (this.imgVersion > -13) {
                return;
            }
            int size = in.readInt();
            LOG.info((Object)("Number of files under construction = " + size));
            for (int i = 0; i < size; ++i) {
                INodeFileUnderConstruction cons = FSImageSerialization.readINodeUnderConstruction(in);
                String path = cons.getLocalName();
                INodeFile old = fsDir.getFileINode(path);
                if (old == null) {
                    throw new IOException("Found lease for non-existent file " + path);
                }
                if (((INode)old).isDirectory()) {
                    throw new IOException("Found lease for directory " + path);
                }
                INodeFile oldnode = old;
                fsDir.replaceNode(path, oldnode, cons);
                fs.leaseManager.addLease(cons.getClientName(), path);
            }
        }

        private void loadSecretManagerState(DataInputStream in, FSNamesystem fs) throws IOException {
            if (!LayoutVersion.supports(LayoutVersion.Feature.DELEGATION_TOKEN, this.imgVersion)) {
                return;
            }
            fs.loadSecretManagerState(in);
        }

        private long readNumFiles(DataInputStream in) throws IOException {
            if (LayoutVersion.supports(LayoutVersion.Feature.NAMESPACE_QUOTA, this.imgVersion)) {
                return in.readLong();
            }
            return in.readInt();
        }

        private boolean isRoot(byte[][] path) {
            return path.length == 1 && path[0] == null;
        }

        private boolean isParent(byte[][] path, byte[][] parent) {
            if (path == null || parent == null) {
                return false;
            }
            if (parent.length == 0 || path.length != parent.length + 1) {
                return false;
            }
            boolean isParent = true;
            for (int i = 0; i < parent.length; ++i) {
                isParent = isParent && Arrays.equals(path[i], parent[i]);
            }
            return isParent;
        }

        String getParent(String path) {
            return path.substring(0, path.lastIndexOf("/"));
        }

        byte[][] getParent(byte[][] path) {
            byte[][] result = new byte[path.length - 1][];
            for (int i = 0; i < result.length; ++i) {
                result[i] = new byte[path[i].length];
                System.arraycopy(path[i], 0, result[i], 0, path[i].length);
            }
            return result;
        }
    }
}

