/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;
import java.util.zip.GZIPInputStream;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.ChecksumFileSystem;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.FsShellPermissions;
import org.apache.hadoop.fs.FsStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.Trash;
import org.apache.hadoop.fs.shell.CommandFormat;
import org.apache.hadoop.fs.shell.Count;
import org.apache.hadoop.io.DataInputBuffer;
import org.apache.hadoop.io.DataOutputBuffer;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.SequenceFile;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.io.WritableComparable;
import org.apache.hadoop.io.compress.CompressionCodec;
import org.apache.hadoop.io.compress.CompressionCodecFactory;
import org.apache.hadoop.ipc.RPC;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.util.ReflectionUtils;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;

@InterfaceAudience.Private
public class FsShell
extends Configured
implements Tool {
    protected FileSystem fs = null;
    private Trash trash = null;
    public static final SimpleDateFormat dateForm = new SimpleDateFormat("yyyy-MM-dd HH:mm");
    protected static final SimpleDateFormat modifFmt = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    static final int BORDER = 2;
    static final String SETREP_SHORT_USAGE = "-setrep [-R] [-w] <rep> <path/file>";
    static final String GET_SHORT_USAGE = "-get [-ignoreCrc] [-crc] <src> <localdst>";
    static final String COPYTOLOCAL_SHORT_USAGE;
    static final String TAIL_USAGE = "-tail [-f] <file>";
    static final String DU_USAGE = "-du [-s] [-h] <paths...>";
    static final String COPYTOLOCAL_PREFIX = "_copyToLocal_";

    public FsShell() {
        this(null);
    }

    public FsShell(Configuration conf) {
        super(conf);
    }

    protected void init() throws IOException {
        this.getConf().setQuietMode(true);
        if (this.fs == null) {
            this.fs = FileSystem.get(this.getConf());
        }
        if (this.trash == null) {
            this.trash = new Trash(this.getConf());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void copyFromStdin(Path dst, FileSystem dstFs) throws IOException {
        if (dstFs.isDirectory(dst)) {
            throw new IOException("When source is stdin, destination must be a file.");
        }
        if (dstFs.exists(dst)) {
            throw new IOException("Target " + dst.toString() + " already exists.");
        }
        FSDataOutputStream out = dstFs.create(dst);
        try {
            IOUtils.copyBytes(System.in, (OutputStream)out, this.getConf(), false);
        }
        finally {
            out.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void printToStdout(InputStream in) throws IOException {
        try {
            IOUtils.copyBytes(in, (OutputStream)System.out, this.getConf(), false);
        }
        finally {
            in.close();
        }
    }

    void copyFromLocal(Path[] srcs, String dstf) throws IOException {
        Path dstPath = new Path(dstf);
        FileSystem dstFs = dstPath.getFileSystem(this.getConf());
        if (srcs.length == 1 && srcs[0].toString().equals("-")) {
            this.copyFromStdin(dstPath, dstFs);
        } else {
            dstFs.copyFromLocalFile(false, false, srcs, dstPath);
        }
    }

    void moveFromLocal(Path[] srcs, String dstf) throws IOException {
        Path dstPath = new Path(dstf);
        FileSystem dstFs = dstPath.getFileSystem(this.getConf());
        dstFs.moveFromLocalFile(srcs, dstPath);
    }

    void moveFromLocal(Path src, String dstf) throws IOException {
        this.moveFromLocal(new Path[]{src}, dstf);
    }

    void copyToLocal(String[] argv, int pos) throws IOException {
        boolean verifyChecksum;
        CommandFormat cf = new CommandFormat("copyToLocal", 2, 2, "crc", "ignoreCrc");
        String srcstr = null;
        String dststr = null;
        try {
            List<String> parameters = cf.parse(argv, pos);
            srcstr = parameters.get(0);
            dststr = parameters.get(1);
        }
        catch (IllegalArgumentException iae) {
            System.err.println("Usage: java FsShell -get [-ignoreCrc] [-crc] <src> <localdst>");
            throw iae;
        }
        boolean copyCrc = cf.getOpt("crc");
        boolean bl = verifyChecksum = !cf.getOpt("ignoreCrc");
        if (dststr.equals("-")) {
            if (copyCrc) {
                System.err.println("-crc option is not valid when destination is stdout.");
            }
            this.cat(srcstr, verifyChecksum);
        } else {
            FileStatus[] srcs;
            File dst = new File(dststr);
            Path srcpath = new Path(srcstr);
            FileSystem srcFS = this.getSrcFileSystem(srcpath, verifyChecksum);
            if (copyCrc && !(srcFS instanceof ChecksumFileSystem)) {
                System.err.println("-crc option is not valid when source file system does not have crc files. Automatically turn the option off.");
                copyCrc = false;
            }
            if (null == (srcs = srcFS.globStatus(srcpath))) {
                throw new IOException(srcpath + ": No such file or directory");
            }
            boolean dstIsDir = dst.isDirectory();
            if (srcs.length > 1 && !dstIsDir) {
                throw new IOException("When copying multiple files, destination should be a directory.");
            }
            for (FileStatus status : srcs) {
                Path p = status.getPath();
                File f = dstIsDir ? new File(dst, p.getName()) : dst;
                this.copyToLocal(srcFS, status, f, copyCrc);
            }
        }
    }

    private FileSystem getSrcFileSystem(Path src, boolean verifyChecksum) throws IOException {
        FileSystem srcFs = src.getFileSystem(this.getConf());
        srcFs.setVerifyChecksum(verifyChecksum);
        return srcFs;
    }

    private void copyToLocal(FileSystem srcFS, FileStatus srcStatus, File dst, boolean copyCrc) throws IOException {
        Path src = srcStatus.getPath();
        if (srcStatus.isFile()) {
            if (dst.exists()) {
                throw new IOException("Target " + dst + " already exists");
            }
            File tmp = FileUtil.createLocalTempFile(dst.getAbsoluteFile(), COPYTOLOCAL_PREFIX, true);
            if (!FileUtil.copy(srcFS, src, tmp, false, srcFS.getConf())) {
                throw new IOException("Failed to copy " + src + " to " + dst);
            }
            if (!tmp.renameTo(dst)) {
                throw new IOException("Failed to rename tmp file " + tmp + " to local destination \"" + dst + "\".");
            }
            if (copyCrc) {
                if (!(srcFS instanceof ChecksumFileSystem)) {
                    throw new IOException("Source file system does not have crc files");
                }
                ChecksumFileSystem csfs = (ChecksumFileSystem)srcFS;
                File dstcs = FileSystem.getLocal(srcFS.getConf()).pathToFile(csfs.getChecksumFile(new Path(dst.getCanonicalPath())));
                FileSystem fs = csfs.getRawFileSystem();
                FileStatus status = csfs.getFileStatus(csfs.getChecksumFile(src));
                this.copyToLocal(fs, status, dstcs, false);
            }
        } else {
            if (srcStatus.isSymlink()) {
                throw new AssertionError((Object)"Symlinks unsupported");
            }
            if (!dst.mkdirs()) {
                throw new IOException("Failed to create local destination \"" + dst + "\".");
            }
            for (FileStatus status : srcFS.listStatus(src)) {
                this.copyToLocal(srcFS, status, new File(dst, status.getPath().getName()), copyCrc);
            }
        }
    }

    void copyMergeToLocal(String srcf, Path dst) throws IOException {
        this.copyMergeToLocal(srcf, dst, false);
    }

    void copyMergeToLocal(String srcf, Path dst, boolean endline) throws IOException {
        Path srcPath = new Path(srcf);
        FileSystem srcFs = srcPath.getFileSystem(this.getConf());
        Path[] srcs = FileUtil.stat2Paths(srcFs.globStatus(srcPath), srcPath);
        for (int i = 0; i < srcs.length; ++i) {
            if (endline) {
                FileUtil.copyMerge(srcFs, srcs[i], FileSystem.getLocal(this.getConf()), dst, false, this.getConf(), "\n");
                continue;
            }
            FileUtil.copyMerge(srcFs, srcs[i], FileSystem.getLocal(this.getConf()), dst, false, this.getConf(), null);
        }
    }

    void moveToLocal(String srcf, Path dst) throws IOException {
        System.err.println("Option '-moveToLocal' is not implemented yet.");
    }

    void cat(String src, boolean verifyChecksum) throws IOException {
        Path srcPattern = new Path(src);
        new DelayedExceptionThrowing(){

            @Override
            void process(Path p, FileSystem srcFs) throws IOException {
                FsShell.this.printToStdout(srcFs.open(p));
            }
        }.globAndProcess(srcPattern, this.getSrcFileSystem(srcPattern, verifyChecksum));
    }

    private InputStream forMagic(Path p, FileSystem srcFs) throws IOException {
        FSDataInputStream i = srcFs.open(p);
        CompressionCodecFactory cf = new CompressionCodecFactory(this.getConf());
        CompressionCodec codec = cf.getCodec(p);
        if (codec != null) {
            return codec.createInputStream(i);
        }
        switch (i.readShort()) {
            case 8075: {
                i.seek(0L);
                return new GZIPInputStream(i);
            }
            case 21317: {
                if (i.readByte() != 81) break;
                i.close();
                return new TextRecordInputStream(srcFs.getFileStatus(p));
            }
        }
        i.seek(0L);
        return i;
    }

    void text(String srcf) throws IOException {
        Path srcPattern = new Path(srcf);
        new DelayedExceptionThrowing(){

            @Override
            void process(Path p, FileSystem srcFs) throws IOException {
                if (srcFs.isDirectory(p)) {
                    throw new IOException("Source must be a file.");
                }
                FsShell.this.printToStdout(FsShell.this.forMagic(p, srcFs));
            }
        }.globAndProcess(srcPattern, srcPattern.getFileSystem(this.getConf()));
    }

    private void setReplication(String[] cmd, int pos) throws IOException {
        CommandFormat c = new CommandFormat("setrep", 2, 2, "R", "w");
        String dst = null;
        short rep = 0;
        try {
            List<String> parameters = c.parse(cmd, pos);
            rep = Short.parseShort(parameters.get(0));
            dst = parameters.get(1);
        }
        catch (NumberFormatException nfe) {
            System.err.println("Illegal replication, a positive integer expected");
            throw nfe;
        }
        catch (IllegalArgumentException iae) {
            System.err.println("Usage: java FsShell -setrep [-R] [-w] <rep> <path/file>");
            throw iae;
        }
        if (rep < 1) {
            System.err.println("Cannot set replication to: " + rep);
            throw new IllegalArgumentException("replication must be >= 1");
        }
        ArrayList<Path> waitList = c.getOpt("w") ? new ArrayList<Path>() : null;
        this.setReplication(rep, dst, c.getOpt("R"), waitList);
        if (waitList != null) {
            this.waitForReplication(waitList, rep);
        }
    }

    void waitForReplication(List<Path> waitList, int rep) throws IOException {
        for (Path f : waitList) {
            System.out.print("Waiting for " + f + " ...");
            System.out.flush();
            boolean printWarning = false;
            FileStatus status = this.fs.getFileStatus(f);
            long len = status.getLen();
            boolean done = false;
            while (!done) {
                int i;
                BlockLocation[] locations = this.fs.getFileBlockLocations(status, 0L, len);
                for (i = 0; i < locations.length && locations[i].getHosts().length == rep; ++i) {
                    if (printWarning || locations[i].getHosts().length <= rep) continue;
                    System.out.println("\nWARNING: the waiting time may be long for DECREASING the number of replication.");
                    printWarning = true;
                }
                done = i == locations.length;
                if (done) continue;
                System.out.print(".");
                System.out.flush();
                try {
                    Thread.sleep(10000L);
                }
                catch (InterruptedException e) {}
            }
            System.out.println(" done");
        }
    }

    void setReplication(short newRep, String srcf, boolean recursive, List<Path> waitingList) throws IOException {
        Path srcPath = new Path(srcf);
        FileSystem srcFs = srcPath.getFileSystem(this.getConf());
        Path[] srcs = FileUtil.stat2Paths(srcFs.globStatus(srcPath), srcPath);
        for (int i = 0; i < srcs.length; ++i) {
            this.setReplication(newRep, srcFs, srcs[i], recursive, waitingList);
        }
    }

    private void setReplication(short newRep, FileSystem srcFs, Path src, boolean recursive, List<Path> waitingList) throws IOException {
        FileStatus[] items;
        if (srcFs.getFileStatus(src).isFile()) {
            this.setFileReplication(src, srcFs, newRep, waitingList);
            return;
        }
        try {
            items = srcFs.listStatus(src);
        }
        catch (FileNotFoundException fnfe) {
            throw new IOException("Could not get listing for " + src);
        }
        for (int i = 0; i < items.length; ++i) {
            if (items[i].isFile()) {
                this.setFileReplication(items[i].getPath(), srcFs, newRep, waitingList);
                continue;
            }
            if (items[i].isSymlink()) {
                throw new AssertionError((Object)"Symlinks unsupported");
            }
            if (!recursive) continue;
            this.setReplication(newRep, srcFs, items[i].getPath(), recursive, waitingList);
        }
    }

    private void setFileReplication(Path file, FileSystem srcFs, short newRep, List<Path> waitList) throws IOException {
        if (srcFs.setReplication(file, newRep)) {
            if (waitList != null) {
                waitList.add(file);
            }
            System.out.println("Replication " + newRep + " set: " + file);
        } else {
            System.err.println("Could not set replication for: " + file);
        }
    }

    private int ls(String srcf, boolean recursive) throws IOException {
        Path srcPath = new Path(srcf);
        FileSystem srcFs = srcPath.getFileSystem(this.getConf());
        FileStatus[] srcs = srcFs.globStatus(srcPath);
        if (srcs == null || srcs.length == 0) {
            throw new FileNotFoundException("Cannot access " + srcf + ": No such file or directory.");
        }
        boolean printHeader = srcs.length == 1;
        int numOfErrors = 0;
        for (int i = 0; i < srcs.length; ++i) {
            numOfErrors += this.ls(srcs[i], srcFs, recursive, printHeader);
        }
        return numOfErrors == 0 ? 0 : -1;
    }

    private int ls(FileStatus src, FileSystem srcFs, boolean recursive, boolean printHeader) throws IOException {
        FileStatus stat;
        int i;
        String cmd = recursive ? "lsr" : "ls";
        FileStatus[] items = FsShell.shellListStatus(cmd, srcFs, src);
        if (items == null) {
            return 1;
        }
        int numOfErrors = 0;
        if (!recursive && printHeader && items.length != 0) {
            System.out.println("Found " + items.length + " items");
        }
        int maxReplication = 3;
        int maxLen = 10;
        int maxOwner = 0;
        int maxGroup = 0;
        for (i = 0; i < items.length; ++i) {
            stat = items[i];
            int replication = String.valueOf(stat.getReplication()).length();
            int len = String.valueOf(stat.getLen()).length();
            int owner = String.valueOf(stat.getOwner()).length();
            int group = String.valueOf(stat.getGroup()).length();
            if (replication > maxReplication) {
                maxReplication = replication;
            }
            if (len > maxLen) {
                maxLen = len;
            }
            if (owner > maxOwner) {
                maxOwner = owner;
            }
            if (group <= maxGroup) continue;
            maxGroup = group;
        }
        for (i = 0; i < items.length; ++i) {
            stat = items[i];
            Path cur = stat.getPath();
            String mdate = dateForm.format(new Date(stat.getModificationTime()));
            System.out.print((stat.isDirectory() ? "d" : "-") + stat.getPermission() + " ");
            System.out.printf("%" + maxReplication + "s ", stat.isFile() ? Short.valueOf(stat.getReplication()) : "-");
            if (maxOwner > 0) {
                System.out.printf("%-" + maxOwner + "s ", stat.getOwner());
            }
            if (maxGroup > 0) {
                System.out.printf("%-" + maxGroup + "s ", stat.getGroup());
            }
            System.out.printf("%" + maxLen + "d ", stat.getLen());
            System.out.print(mdate + " ");
            System.out.println(cur.toUri().getPath());
            if (!recursive || !stat.isDirectory()) continue;
            numOfErrors += this.ls(stat, srcFs, recursive, printHeader);
        }
        return numOfErrors;
    }

    void df(String path) throws IOException {
        Path srcPath;
        FileSystem srcFs;
        if (path == null) {
            path = "/";
        }
        if (!(srcFs = (srcPath = new Path(path)).getFileSystem(this.getConf())).exists(srcPath)) {
            throw new FileNotFoundException("Cannot access " + srcPath.toString());
        }
        FsStatus stats = srcFs.getStatus(srcPath);
        int PercentUsed = (int)(100.0f * (float)stats.getUsed() / (float)stats.getCapacity());
        System.out.println("Filesystem\t\tSize\tUsed\tAvail\tUse%");
        System.out.printf("%s\t\t%d\t%d\t%d\t%d%%\n", path, stats.getCapacity(), stats.getUsed(), stats.getRemaining(), PercentUsed);
    }

    void du(String[] cmd, int pos) throws IOException {
        List<String> params;
        CommandFormat c = new CommandFormat("du", 0, Integer.MAX_VALUE, "h", "s");
        try {
            params = c.parse(cmd, pos);
        }
        catch (IllegalArgumentException iae) {
            System.err.println("Usage: java FsShell -du [-s] [-h] <paths...>");
            throw iae;
        }
        boolean humanReadable = c.getOpt("h");
        boolean summary = c.getOpt("s");
        if (params.isEmpty()) {
            params.add(".");
        }
        ArrayList<UsagePair> usages = new ArrayList<UsagePair>();
        for (String src : params) {
            FileStatus[] statusToPrint;
            Path srcPath = new Path(src);
            FileSystem srcFs = srcPath.getFileSystem(this.getConf());
            FileStatus[] globStatus = srcFs.globStatus(srcPath);
            if (summary) {
                statusToPrint = globStatus;
            } else {
                Path[] statPaths = FileUtil.stat2Paths(globStatus, srcPath);
                try {
                    statusToPrint = srcFs.listStatus(statPaths);
                }
                catch (FileNotFoundException fnfe) {
                    statusToPrint = null;
                }
            }
            if (statusToPrint == null || statusToPrint.length == 0 && !srcFs.exists(srcPath)) {
                throw new FileNotFoundException("Cannot access " + src + ": No such file or directory.");
            }
            if (!summary) {
                System.out.println("Found " + statusToPrint.length + " items");
            }
            for (FileStatus stat : statusToPrint) {
                long length = summary || stat.isDirectory() ? srcFs.getContentSummary(stat.getPath()).getLength() : stat.getLen();
                usages.add(new UsagePair(String.valueOf(stat.getPath()), length));
            }
        }
        this.printUsageSummary(usages, humanReadable);
    }

    void dus(String[] cmd, int pos) throws IOException {
        String[] newcmd = new String[cmd.length + 1];
        System.arraycopy(cmd, 0, newcmd, 0, cmd.length);
        newcmd[cmd.length] = "-s";
        this.du(newcmd, pos);
    }

    private void printUsageSummary(List<UsagePair> usages, boolean humanReadable) {
        String toPrint;
        int maxColumnWidth = 0;
        for (UsagePair usage : usages) {
            toPrint = humanReadable ? StringUtils.humanReadableInt(usage.bytes) : String.valueOf(usage.bytes);
            if (toPrint.length() <= maxColumnWidth) continue;
            maxColumnWidth = toPrint.length();
        }
        for (UsagePair usage : usages) {
            toPrint = humanReadable ? StringUtils.humanReadableInt(usage.bytes) : String.valueOf(usage.bytes);
            System.out.printf("%-" + (maxColumnWidth + 2) + "s", toPrint);
            System.out.println(usage.path);
        }
    }

    void mkdir(String src) throws IOException {
        Path f = new Path(src);
        FileSystem srcFs = f.getFileSystem(this.getConf());
        FileStatus fstatus = null;
        try {
            fstatus = srcFs.getFileStatus(f);
            if (fstatus.isDirectory()) {
                throw new IOException("cannot create directory " + src + ": File exists");
            }
            throw new IOException(src + " exists but " + "is not a directory");
        }
        catch (FileNotFoundException e) {
            if (!srcFs.mkdirs(f)) {
                throw new IOException("failed to create " + src);
            }
            return;
        }
    }

    void touchz(String src) throws IOException {
        Path f = new Path(src);
        FileSystem srcFs = f.getFileSystem(this.getConf());
        if (srcFs.exists(f)) {
            FileStatus st = srcFs.getFileStatus(f);
            if (st.isDirectory()) {
                throw new IOException(src + " is a directory");
            }
            if (st.getLen() != 0L) {
                throw new IOException(src + " must be a zero-length file");
            }
        }
        FSDataOutputStream out = srcFs.create(f);
        out.close();
    }

    int test(String[] argv, int i) throws IOException {
        if (!argv[i].startsWith("-") || argv[i].length() > 2) {
            throw new IOException("Not a flag: " + argv[i]);
        }
        char flag = argv[i].toCharArray()[1];
        Path f = new Path(argv[++i]);
        FileSystem srcFs = f.getFileSystem(this.getConf());
        switch (flag) {
            case 'e': {
                return srcFs.exists(f) ? 0 : 1;
            }
            case 'z': {
                return srcFs.getFileStatus(f).getLen() == 0L ? 0 : 1;
            }
            case 'd': {
                return srcFs.getFileStatus(f).isDirectory() ? 0 : 1;
            }
        }
        throw new IOException("Unknown flag: " + flag);
    }

    void stat(char[] fmt, String src) throws IOException {
        Path srcPath = new Path(src);
        FileSystem srcFs = srcPath.getFileSystem(this.getConf());
        FileStatus[] glob = srcFs.globStatus(srcPath);
        if (null == glob) {
            throw new IOException("cannot stat `" + src + "': No such file or directory");
        }
        for (FileStatus f : glob) {
            StringBuilder buf = new StringBuilder();
            block10: for (int i = 0; i < fmt.length; ++i) {
                if (fmt[i] != '%') {
                    buf.append(fmt[i]);
                    continue;
                }
                if (i + 1 == fmt.length) break;
                switch (fmt[++i]) {
                    case 'b': {
                        buf.append(f.getLen());
                        continue block10;
                    }
                    case 'F': {
                        buf.append(f.isDirectory() ? "directory" : (f.isFile() ? "regular file" : "symlink"));
                        continue block10;
                    }
                    case 'n': {
                        buf.append(f.getPath().getName());
                        continue block10;
                    }
                    case 'o': {
                        buf.append(f.getBlockSize());
                        continue block10;
                    }
                    case 'r': {
                        buf.append(f.getReplication());
                        continue block10;
                    }
                    case 'y': {
                        buf.append(modifFmt.format(new Date(f.getModificationTime())));
                        continue block10;
                    }
                    case 'Y': {
                        buf.append(f.getModificationTime());
                        continue block10;
                    }
                    default: {
                        buf.append(fmt[i]);
                    }
                }
            }
            System.out.println(buf.toString());
        }
    }

    void rename(String srcf, String dstf) throws IOException {
        URI dstURI;
        Path srcPath = new Path(srcf);
        Path dstPath = new Path(dstf);
        FileSystem fs = srcPath.getFileSystem(this.getConf());
        URI srcURI = fs.getUri();
        if (srcURI.compareTo(dstURI = dstPath.getFileSystem(this.getConf()).getUri()) != 0) {
            throw new IOException("src and destination filesystems do not match.");
        }
        Path[] srcs = FileUtil.stat2Paths(fs.globStatus(srcPath), srcPath);
        Path dst = new Path(dstf);
        if (srcs.length > 1 && !fs.isDirectory(dst)) {
            throw new IOException("When moving multiple files, destination should be a directory.");
        }
        for (int i = 0; i < srcs.length; ++i) {
            if (fs.rename(srcs[i], dst)) continue;
            FileStatus srcFstatus = null;
            FileStatus dstFstatus = null;
            try {
                srcFstatus = fs.getFileStatus(srcs[i]);
            }
            catch (FileNotFoundException e) {
                throw new FileNotFoundException(srcs[i] + ": No such file or directory");
            }
            try {
                dstFstatus = fs.getFileStatus(dst);
            }
            catch (IOException e) {
                // empty catch block
            }
            if (srcFstatus != null && dstFstatus != null && srcFstatus.isDirectory() && !dstFstatus.isDirectory()) {
                throw new IOException("cannot overwrite non directory " + dst + " with directory " + srcs[i]);
            }
            throw new IOException("Failed to rename " + srcs[i] + " to " + dst);
        }
    }

    private int rename(String[] argv, Configuration conf) throws IOException {
        Path dst;
        FileSystem dstFs;
        int i = 0;
        int exitCode = 0;
        String cmd = argv[i++];
        String dest = argv[argv.length - 1];
        if (argv.length > 3 && !(dstFs = (dst = new Path(dest)).getFileSystem(this.getConf())).isDirectory(dst)) {
            throw new IOException("When moving multiple files, destination " + dest + " should be a directory.");
        }
        while (i < argv.length - 1) {
            try {
                this.rename(argv[i], dest);
            }
            catch (RemoteException e) {
                exitCode = -1;
                try {
                    String[] content = e.getLocalizedMessage().split("\n");
                    System.err.println(cmd.substring(1) + ": " + content[0]);
                }
                catch (Exception ex) {
                    System.err.println(cmd.substring(1) + ": " + ex.getLocalizedMessage());
                }
            }
            catch (IOException e) {
                exitCode = -1;
                System.err.println(cmd.substring(1) + ": " + e.getLocalizedMessage());
            }
            ++i;
        }
        return exitCode;
    }

    void copy(String srcf, String dstf, Configuration conf) throws IOException {
        Path srcPath = new Path(srcf);
        FileSystem srcFs = srcPath.getFileSystem(this.getConf());
        Path dstPath = new Path(dstf);
        FileSystem dstFs = dstPath.getFileSystem(this.getConf());
        Path[] srcs = FileUtil.stat2Paths(srcFs.globStatus(srcPath), srcPath);
        if (srcs.length > 1 && !dstFs.isDirectory(dstPath)) {
            throw new IOException("When copying multiple files, destination should be a directory.");
        }
        for (int i = 0; i < srcs.length; ++i) {
            FileUtil.copy(srcFs, srcs[i], dstFs, dstPath, false, conf);
        }
    }

    private int copy(String[] argv, Configuration conf) throws IOException {
        Path dst;
        int i = 0;
        int exitCode = 0;
        String cmd = argv[i++];
        String dest = argv[argv.length - 1];
        if (argv.length > 3 && !this.fs.isDirectory(dst = new Path(dest))) {
            throw new IOException("When copying multiple files, destination " + dest + " should be a directory.");
        }
        while (i < argv.length - 1) {
            try {
                this.copy(argv[i], dest, conf);
            }
            catch (RemoteException e) {
                exitCode = -1;
                try {
                    String[] content = e.getLocalizedMessage().split("\n");
                    System.err.println(cmd.substring(1) + ": " + content[0]);
                }
                catch (Exception ex) {
                    System.err.println(cmd.substring(1) + ": " + ex.getLocalizedMessage());
                }
            }
            catch (IOException e) {
                exitCode = -1;
                System.err.println(cmd.substring(1) + ": " + e.getLocalizedMessage());
            }
            ++i;
        }
        return exitCode;
    }

    void delete(String srcf, final boolean recursive, final boolean skipTrash) throws IOException {
        Path srcPattern = new Path(srcf);
        new DelayedExceptionThrowing(){

            @Override
            void process(Path p, FileSystem srcFs) throws IOException {
                FsShell.this.delete(p, srcFs, recursive, skipTrash);
            }
        }.globAndProcess(srcPattern, srcPattern.getFileSystem(this.getConf()));
    }

    private void delete(Path src, FileSystem srcFs, boolean recursive, boolean skipTrash) throws IOException {
        FileStatus fs = null;
        try {
            fs = srcFs.getFileStatus(src);
        }
        catch (FileNotFoundException fnfe) {
            throw new FileNotFoundException("cannot remove " + src + ": No such file or directory.");
        }
        if (fs.isDirectory() && !recursive) {
            throw new IOException("Cannot remove directory \"" + src + "\", use -rmr instead");
        }
        if (!skipTrash) {
            try {
                Trash trashTmp = new Trash(srcFs, this.getConf());
                if (trashTmp.moveToTrash(src)) {
                    System.out.println("Moved to trash: " + src);
                    return;
                }
            }
            catch (IOException e) {
                Exception cause = (Exception)e.getCause();
                String msg = "";
                if (cause != null) {
                    msg = cause.getLocalizedMessage();
                }
                System.err.println("Problem with Trash." + msg + ". Consider using -skipTrash option");
                throw e;
            }
        }
        if (!srcFs.delete(src, true)) {
            throw new IOException("Delete failed " + src);
        }
        System.out.println("Deleted " + src);
    }

    private void expunge() throws IOException {
        this.trash.expunge();
        this.trash.checkpoint();
    }

    public Path getCurrentTrashDir() {
        return this.trash.getCurrentTrashDir();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void tail(String[] cmd, int pos) throws IOException {
        CommandFormat c = new CommandFormat("tail", 1, 1, "f");
        String src = null;
        Path path = null;
        try {
            List<String> parameters = c.parse(cmd, pos);
            src = parameters.get(0);
        }
        catch (IllegalArgumentException iae) {
            System.err.println("Usage: java FsShell -tail [-f] <file>");
            throw iae;
        }
        boolean foption = c.getOpt("f");
        path = new Path(src);
        FileSystem srcFs = path.getFileSystem(this.getConf());
        FileStatus fileStatus = srcFs.getFileStatus(path);
        if (fileStatus.isDirectory()) {
            throw new IOException("Source must be a file.");
        }
        long fileSize = fileStatus.getLen();
        long offset = fileSize > 1024L ? fileSize - 1024L : 0L;
        while (true) {
            FSDataInputStream in = srcFs.open(path);
            try {
                in.seek(offset);
                IOUtils.copyBytes((InputStream)in, (OutputStream)System.out, 1024);
                offset = in.getPos();
            }
            finally {
                in.close();
            }
            if (!foption) break;
            fileSize = srcFs.getFileStatus(path).getLen();
            offset = fileSize > offset ? offset : fileSize;
            try {
                Thread.sleep(5000L);
            }
            catch (InterruptedException e) {
                break;
            }
        }
    }

    private static FileStatus[] shellListStatus(String cmd, FileSystem srcFs, FileStatus src) {
        if (src.isFile()) {
            FileStatus[] files = new FileStatus[]{src};
            return files;
        }
        Path path = src.getPath();
        try {
            FileStatus[] files = srcFs.listStatus(path);
            return files;
        }
        catch (FileNotFoundException fnfe) {
            System.err.println(cmd + ": could not get listing for '" + path + "'");
        }
        catch (IOException e) {
            System.err.println(cmd + ": could not get get listing for '" + path + "' : " + e.getMessage().split("\n")[0]);
        }
        return null;
    }

    private static int runCmdHandler(CmdHandler handler, FileStatus stat, FileSystem srcFs, boolean recursive) throws IOException {
        int errors = 0;
        handler.run(stat, srcFs);
        if (recursive && stat.isDirectory() && handler.okToContinue()) {
            FileStatus[] files = FsShell.shellListStatus(handler.getName(), srcFs, stat);
            if (files == null) {
                return 1;
            }
            for (FileStatus file : files) {
                errors += FsShell.runCmdHandler(handler, file, srcFs, recursive);
            }
        }
        return errors;
    }

    int runCmdHandler(CmdHandler handler, String[] args, int startIndex, boolean recursive) throws IOException {
        int errors = 0;
        for (int i = startIndex; i < args.length; ++i) {
            Path srcPath = new Path(args[i]);
            FileSystem srcFs = srcPath.getFileSystem(this.getConf());
            Path[] paths = FileUtil.stat2Paths(srcFs.globStatus(srcPath), srcPath);
            if (paths.length == 0) {
                System.err.println(handler.getName() + ": could not get status for '" + args[i] + "'");
                ++errors;
            }
            for (Path path : paths) {
                try {
                    FileStatus file = srcFs.getFileStatus(path);
                    if (file == null) {
                        System.err.println(handler.getName() + ": could not get status for '" + path + "'");
                        ++errors;
                        continue;
                    }
                    errors += FsShell.runCmdHandler(handler, file, srcFs, recursive);
                }
                catch (IOException e) {
                    String msg = e.getMessage() != null ? e.getLocalizedMessage() : (e.getCause().getMessage() != null ? e.getCause().getLocalizedMessage() : "null");
                    System.err.println(handler.getName() + ": could not get status for '" + path + "': " + msg.split("\n")[0]);
                    ++errors;
                }
            }
        }
        return errors > 0 || handler.getErrorCode() != 0 ? 1 : 0;
    }

    @Deprecated
    public static String byteDesc(long len) {
        return StringUtils.byteDesc(len);
    }

    @Deprecated
    public static synchronized String limitDecimalTo2(double d) {
        return StringUtils.limitDecimalTo2(d);
    }

    private void printHelp(String cmd) {
        String summary = "hadoop fs is the command to execute fs commands. The full syntax is: \n\nhadoop fs [-fs <local | file system URI>] [-conf <configuration file>]\n\t[-D <property=value>] [-ls <path>] [-lsr <path>] [-df [<path>]] [-du [-s] [-h] <path>]\n\t[-dus <path>] [-mv <src> <dst>] [-cp <src> <dst>] [-rm [-skipTrash] <src>]\n\t[-rmr [-skipTrash] <src>] [-put <localsrc> ... <dst>] [-copyFromLocal <localsrc> ... <dst>]\n\t[-moveFromLocal <localsrc> ... <dst>] [-get [-ignoreCrc] [-crc] <src> <localdst>\n\t[-getmerge <src> <localdst> [addnl]] [-cat <src>]\n\t[" + COPYTOLOCAL_SHORT_USAGE + "] [-moveToLocal <src> <localdst>]\n\t" + "[-mkdir <path>] [-report] [" + SETREP_SHORT_USAGE + "]\n\t" + "[-touchz <path>] [-test -[ezd] <path>] [-stat [format] <path>]\n\t" + "[-tail [-f] <path>] [-text <path>]\n\t" + "[" + FsShellPermissions.CHMOD_USAGE + "]\n\t" + "[" + FsShellPermissions.CHOWN_USAGE + "]\n\t" + "[" + FsShellPermissions.CHGRP_USAGE + "]\n\t" + "[" + "-count[-q] <path>" + "]\n\t" + "[-help [cmd]]\n";
        String conf = "-conf <configuration file>:  Specify an application configuration file.";
        String D = "-D <property=value>:  Use value for given property.";
        String fs = "-fs [local | <file system URI>]: \tSpecify the file system to use.\n\t\tIf not specified, the current configuration is used, \n\t\ttaken from the following, in increasing precedence: \n\t\t\tcore-default.xml inside the hadoop jar file \n\t\t\tcore-site.xml in $HADOOP_CONF_DIR \n\t\t'local' means use the local file system as your DFS. \n\t\t<file system URI> specifies a particular file system to \n\t\tcontact. This argument is optional but if used must appear\n\t\tappear first on the command line.  Exactly one additional\n\t\targument must be specified. \n";
        String ls = "-ls <path>: \tList the contents that match the specified file pattern. If\n\t\tpath is not specified, the contents of /user/<currentUser>\n\t\twill be listed. Directory entries are of the form \n\t\t\tdirName (full path) <dir> \n\t\tand file entries are of the form \n\t\t\tfileName(full path) <r n> size \n\t\twhere n is the number of replicas specified for the file \n\t\tand size is the size of the file, in bytes.\n";
        String lsr = "-lsr <path>: \tRecursively list the contents that match the specified\n\t\tfile pattern.  Behaves very similarly to hadoop fs -ls,\n\t\texcept that the data is shown for all the entries in the\n\t\tsubtree.\n";
        String df = "-df [<path>]: \tShows the capacity, free and used space of the filesystem.\n\t\tIf the filesystem has multiple partitions, and no path to a particular partition\n\t\tis specified, then the status of the root partitions will be shown.\n";
        String du = "-du [-s] [-h] <path>: \tShow the amount of space, in bytes, used by the files that \n\t\tmatch the specified file pattern. The following flags are optional:\n\t\t  -s   Rather than showing the size of each individual file that\n\t\t       matches the pattern, shows the total (summary) size.\n\t\t  -h   Formats the sizes of files in a human-readable fashion\n\t\t       rather than a number of bytes.\n\n\t\tNote that, even without the -s option, this only shows size summaries\n\t\tone level deep into a directory.\n\t\tThe output is in the form \n\t\t\tsize\tname(full path)\n";
        String dus = "-dus <path>: \tShow the amount of space, in bytes, used by the files that \n\t\tmatch the specified file pattern. This is equivalent to -du -s above.\n";
        String mv = "-mv <src> <dst>:   Move files that match the specified file pattern <src>\n\t\tto a destination <dst>.  When moving multiple files, the \n\t\tdestination must be a directory. \n";
        String cp = "-cp <src> <dst>:   Copy files that match the file pattern <src> to a \n\t\tdestination.  When copying multiple files, the destination\n\t\tmust be a directory. \n";
        String rm = "-rm [-skipTrash] <src>: \tDelete all files that match the specified file pattern.\n\t\tEquivalent to the Unix command \"rm <src>\"\n\t\t-skipTrash option bypasses trash, if enabled, and immediately\ndeletes <src>";
        String rmr = "-rmr [-skipTrash] <src>: \tRemove all directories which match the specified file \n\t\tpattern. Equivalent to the Unix command \"rm -rf <src>\"\n\t\t-skipTrash option bypasses trash, if enabled, and immediately\ndeletes <src>";
        String put = "-put <localsrc> ... <dst>: \tCopy files from the local file system \n\t\tinto fs. \n";
        String copyFromLocal = "-copyFromLocal <localsrc> ... <dst>: Identical to the -put command.\n";
        String moveFromLocal = "-moveFromLocal <localsrc> ... <dst>: Same as -put, except that the source is\n\t\tdeleted after it's copied.\n";
        String get = "-get [-ignoreCrc] [-crc] <src> <localdst>:  Copy files that match the file pattern <src> \n\t\tto the local name.  <src> is kept.  When copying mutiple, \n\t\tfiles, the destination must be a directory. \n";
        String getmerge = "-getmerge <src> <localdst>:  Get all the files in the directories that \n\t\tmatch the source file pattern and merge and sort them to only\n\t\tone file on local fs. <src> is kept.\n";
        String cat = "-cat <src>: \tFetch all files that match the file pattern <src> \n\t\tand display their content on stdout.\n";
        String text = "-text <src>: \tTakes a source file and outputs the file in text format.\n\t\tThe allowed formats are zip and TextRecordInputStream.\n";
        String copyToLocal = COPYTOLOCAL_SHORT_USAGE + ":  Identical to the -get command.\n";
        String moveToLocal = "-moveToLocal <src> <localdst>:  Not implemented yet \n";
        String mkdir = "-mkdir <path>: \tCreate a directory in specified location. \n";
        String setrep = "-setrep [-R] [-w] <rep> <path/file>:  Set the replication level of a file. \n\t\tThe -R flag requests a recursive change of replication level \n\t\tfor an entire tree.\n";
        String touchz = "-touchz <path>: Creates a file of zero length\n\t\t at <path> with current time as the timestamp of that <path>.\n\t\t An error is returned if the file exists with non-zero length\n";
        String test = "-test -[ezd] <path>: If file { exists, has zero length, is a directory\n\t\tthen return 0, else return 1.\n";
        String stat = "-stat [format] <path>: Print statistics about the file/directory at <path>\n\t\tin the specified format. Format accepts filesize in blocks (%b), filename (%n),\n\t\tblock size (%o), replication (%r), modification date (%y, %Y)\n";
        String tail = "-tail [-f] <file>:  Show the last 1KB of the file. \n\t\tThe -f option shows appended data as the file grows. \n";
        String chmod = FsShellPermissions.CHMOD_USAGE + "\n" + "\t\tChanges permissions of a file.\n" + "\t\tThis works similar to shell's chmod with a few exceptions.\n\n" + "\t-R\tmodifies the files recursively. This is the only option\n" + "\t\tcurrently supported.\n\n" + "\tMODE\tMode is same as mode used for chmod shell command.\n" + "\t\tOnly letters recognized are 'rwxXt'. E.g. +t,a+r,g-w,+rwx,o=r\n\n" + "\tOCTALMODE Mode specifed in 3 or 4 digits. If 4 digits, the first may\n" + "\tbe 1 or 0 to turn the sticky bit on or off, respectively.  Unlike " + "\tshell command, it is not possible to specify only part of the mode\n" + "\t\tE.g. 754 is same as u=rwx,g=rx,o=r\n\n" + "\t\tIf none of 'augo' is specified, 'a' is assumed and unlike\n" + "\t\tshell command, no umask is applied.\n";
        String chown = FsShellPermissions.CHOWN_USAGE + "\n" + "\t\tChanges owner and group of a file.\n" + "\t\tThis is similar to shell's chown with a few exceptions.\n\n" + "\t-R\tmodifies the files recursively. This is the only option\n" + "\t\tcurrently supported.\n\n" + "\t\tIf only owner or group is specified then only owner or\n" + "\t\tgroup is modified.\n\n" + "\t\tThe owner and group names may only cosists of digits, alphabet,\n" + "\t\tand any of '-_.@/' i.e. [-_.@/a-zA-Z0-9]. The names are case\n" + "\t\tsensitive.\n\n" + "\t\tWARNING: Avoid using '.' to separate user name and group though\n" + "\t\tLinux allows it. If user names have dots in them and you are\n" + "\t\tusing local file system, you might see surprising results since\n" + "\t\tshell command 'chown' is used for local files.\n";
        String chgrp = FsShellPermissions.CHGRP_USAGE + "\n" + "\t\tThis is equivalent to -chown ... :GROUP ...\n";
        String expunge = "-expunge: Empty the Trash.\n";
        String help = "-help [cmd]: \tDisplays help for given command or all commands if none\n\t\tis specified.\n";
        if ("fs".equals(cmd)) {
            System.out.println(fs);
        } else if ("conf".equals(cmd)) {
            System.out.println(conf);
        } else if ("D".equals(cmd)) {
            System.out.println(D);
        } else if ("ls".equals(cmd)) {
            System.out.println(ls);
        } else if ("lsr".equals(cmd)) {
            System.out.println(lsr);
        } else if ("df".equals(cmd)) {
            System.out.println(df);
        } else if ("du".equals(cmd)) {
            System.out.println(du);
        } else if ("dus".equals(cmd)) {
            System.out.println(dus);
        } else if ("rm".equals(cmd)) {
            System.out.println(rm);
        } else if ("expunge".equals(cmd)) {
            System.out.println(expunge);
        } else if ("rmr".equals(cmd)) {
            System.out.println(rmr);
        } else if ("mkdir".equals(cmd)) {
            System.out.println(mkdir);
        } else if ("mv".equals(cmd)) {
            System.out.println(mv);
        } else if ("cp".equals(cmd)) {
            System.out.println(cp);
        } else if ("put".equals(cmd)) {
            System.out.println(put);
        } else if ("copyFromLocal".equals(cmd)) {
            System.out.println(copyFromLocal);
        } else if ("moveFromLocal".equals(cmd)) {
            System.out.println(moveFromLocal);
        } else if ("get".equals(cmd)) {
            System.out.println(get);
        } else if ("getmerge".equals(cmd)) {
            System.out.println(getmerge);
        } else if ("copyToLocal".equals(cmd)) {
            System.out.println(copyToLocal);
        } else if ("moveToLocal".equals(cmd)) {
            System.out.println(moveToLocal);
        } else if ("cat".equals(cmd)) {
            System.out.println(cat);
        } else if ("get".equals(cmd)) {
            System.out.println(get);
        } else if ("setrep".equals(cmd)) {
            System.out.println(setrep);
        } else if ("touchz".equals(cmd)) {
            System.out.println(touchz);
        } else if ("test".equals(cmd)) {
            System.out.println(test);
        } else if ("text".equals(cmd)) {
            System.out.println(text);
        } else if ("stat".equals(cmd)) {
            System.out.println(stat);
        } else if ("tail".equals(cmd)) {
            System.out.println(tail);
        } else if ("chmod".equals(cmd)) {
            System.out.println(chmod);
        } else if ("chown".equals(cmd)) {
            System.out.println(chown);
        } else if ("chgrp".equals(cmd)) {
            System.out.println(chgrp);
        } else if ("count".equals(cmd)) {
            System.out.println(Count.DESCRIPTION);
        } else if ("help".equals(cmd)) {
            System.out.println(help);
        } else {
            System.out.println(summary);
            System.out.println(fs);
            System.out.println(ls);
            System.out.println(lsr);
            System.out.println(df);
            System.out.println(du);
            System.out.println(dus);
            System.out.println(mv);
            System.out.println(cp);
            System.out.println(rm);
            System.out.println(rmr);
            System.out.println(put);
            System.out.println(copyFromLocal);
            System.out.println(moveFromLocal);
            System.out.println(get);
            System.out.println(getmerge);
            System.out.println(cat);
            System.out.println(copyToLocal);
            System.out.println(moveToLocal);
            System.out.println(mkdir);
            System.out.println(setrep);
            System.out.println(tail);
            System.out.println(touchz);
            System.out.println(test);
            System.out.println(text);
            System.out.println(stat);
            System.out.println(chmod);
            System.out.println(chown);
            System.out.println(chgrp);
            System.out.println(Count.DESCRIPTION);
            System.out.println(help);
        }
    }

    private int doall(String cmd, String[] argv, int startindex) {
        int exitCode = 0;
        int i = startindex;
        boolean rmSkipTrash = false;
        if (("-rm".equals(cmd) || "-rmr".equals(cmd)) && "-skipTrash".equals(argv[i])) {
            rmSkipTrash = true;
            ++i;
        }
        while (i < argv.length) {
            Object content;
            try {
                if ("-cat".equals(cmd)) {
                    this.cat(argv[i], true);
                } else if ("-mkdir".equals(cmd)) {
                    this.mkdir(argv[i]);
                } else if ("-rm".equals(cmd)) {
                    this.delete(argv[i], false, rmSkipTrash);
                } else if ("-rmr".equals(cmd)) {
                    this.delete(argv[i], true, rmSkipTrash);
                } else if ("-df".equals(cmd)) {
                    this.df(argv[i]);
                } else if (Count.matches(cmd)) {
                    new Count(argv, i, this.getConf()).runAll();
                } else if ("-ls".equals(cmd)) {
                    exitCode = this.ls(argv[i], false);
                } else if ("-lsr".equals(cmd)) {
                    exitCode = this.ls(argv[i], true);
                } else if ("-touchz".equals(cmd)) {
                    this.touchz(argv[i]);
                } else if ("-text".equals(cmd)) {
                    this.text(argv[i]);
                }
            }
            catch (RemoteException e) {
                exitCode = -1;
                try {
                    content = e.getLocalizedMessage().split("\n");
                    System.err.println(cmd.substring(1) + ": " + content[0]);
                }
                catch (Exception ex) {
                    System.err.println(cmd.substring(1) + ": " + ex.getLocalizedMessage());
                }
            }
            catch (IOException e) {
                exitCode = -1;
                content = e.getLocalizedMessage();
                if (content != null) {
                    content = content.split("\n")[0];
                }
                System.err.println(cmd.substring(1) + ": " + (String)content);
            }
            ++i;
        }
        return exitCode;
    }

    private static void printUsage(String cmd) {
        String prefix = "Usage: java " + FsShell.class.getSimpleName();
        if ("-fs".equals(cmd)) {
            System.err.println("Usage: java FsShell [-fs <local | file system URI>]");
        } else if ("-conf".equals(cmd)) {
            System.err.println("Usage: java FsShell [-conf <configuration file>]");
        } else if ("-D".equals(cmd)) {
            System.err.println("Usage: java FsShell [-D <[property=value>]");
        } else if ("-ls".equals(cmd) || "-lsr".equals(cmd) || "-du".equals(cmd) || "-dus".equals(cmd) || "-touchz".equals(cmd) || "-mkdir".equals(cmd) || "-text".equals(cmd)) {
            System.err.println("Usage: java FsShell [" + cmd + " <path>]");
        } else if ("-df".equals(cmd)) {
            System.err.println("Usage: java FsShell [" + cmd + " [<path>]]");
        } else if (Count.matches(cmd)) {
            System.err.println(prefix + " [" + "-count[-q] <path>" + "]");
        } else if ("-rm".equals(cmd) || "-rmr".equals(cmd)) {
            System.err.println("Usage: java FsShell [" + cmd + " [-skipTrash] <src>]");
        } else if ("-mv".equals(cmd) || "-cp".equals(cmd)) {
            System.err.println("Usage: java FsShell [" + cmd + " <src> <dst>]");
        } else if ("-put".equals(cmd) || "-copyFromLocal".equals(cmd) || "-moveFromLocal".equals(cmd)) {
            System.err.println("Usage: java FsShell [" + cmd + " <localsrc> ... <dst>]");
        } else if ("-get".equals(cmd)) {
            System.err.println("Usage: java FsShell [-get [-ignoreCrc] [-crc] <src> <localdst>]");
        } else if ("-copyToLocal".equals(cmd)) {
            System.err.println("Usage: java FsShell [" + COPYTOLOCAL_SHORT_USAGE + "]");
        } else if ("-moveToLocal".equals(cmd)) {
            System.err.println("Usage: java FsShell [" + cmd + " [-crc] <src> <localdst>]");
        } else if ("-cat".equals(cmd)) {
            System.err.println("Usage: java FsShell [" + cmd + " <src>]");
        } else if ("-setrep".equals(cmd)) {
            System.err.println("Usage: java FsShell [-setrep [-R] [-w] <rep> <path/file>]");
        } else if ("-test".equals(cmd)) {
            System.err.println("Usage: java FsShell [-test -[ezd] <path>]");
        } else if ("-stat".equals(cmd)) {
            System.err.println("Usage: java FsShell [-stat [format] <path>]");
        } else if ("-tail".equals(cmd)) {
            System.err.println("Usage: java FsShell [-tail [-f] <file>]");
        } else {
            System.err.println("Usage: java FsShell");
            System.err.println("           [-ls <path>]");
            System.err.println("           [-lsr <path>]");
            System.err.println("           [-df [<path>]]");
            System.err.println("           [-du [-s] [-h] <path>]");
            System.err.println("           [-dus <path>]");
            System.err.println("           [-count[-q] <path>]");
            System.err.println("           [-mv <src> <dst>]");
            System.err.println("           [-cp <src> <dst>]");
            System.err.println("           [-rm [-skipTrash] <path>]");
            System.err.println("           [-rmr [-skipTrash] <path>]");
            System.err.println("           [-expunge]");
            System.err.println("           [-put <localsrc> ... <dst>]");
            System.err.println("           [-copyFromLocal <localsrc> ... <dst>]");
            System.err.println("           [-moveFromLocal <localsrc> ... <dst>]");
            System.err.println("           [-get [-ignoreCrc] [-crc] <src> <localdst>]");
            System.err.println("           [-getmerge <src> <localdst> [addnl]]");
            System.err.println("           [-cat <src>]");
            System.err.println("           [-text <src>]");
            System.err.println("           [" + COPYTOLOCAL_SHORT_USAGE + "]");
            System.err.println("           [-moveToLocal [-crc] <src> <localdst>]");
            System.err.println("           [-mkdir <path>]");
            System.err.println("           [-setrep [-R] [-w] <rep> <path/file>]");
            System.err.println("           [-touchz <path>]");
            System.err.println("           [-test -[ezd] <path>]");
            System.err.println("           [-stat [format] <path>]");
            System.err.println("           [-tail [-f] <file>]");
            System.err.println("           [" + FsShellPermissions.CHMOD_USAGE + "]");
            System.err.println("           [" + FsShellPermissions.CHOWN_USAGE + "]");
            System.err.println("           [" + FsShellPermissions.CHGRP_USAGE + "]");
            System.err.println("           [-help [cmd]]");
            System.err.println();
            ToolRunner.printGenericCommandUsage(System.err);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int run(String[] argv) throws Exception {
        int exitCode;
        block59: {
            String cmd;
            if (argv.length < 1) {
                FsShell.printUsage("");
                return -1;
            }
            exitCode = -1;
            int i = 0;
            if ("-put".equals(cmd = argv[i++]) || "-test".equals(cmd) || "-copyFromLocal".equals(cmd) || "-moveFromLocal".equals(cmd)) {
                if (argv.length < 3) {
                    FsShell.printUsage(cmd);
                    return exitCode;
                }
            } else if ("-get".equals(cmd) || "-copyToLocal".equals(cmd) || "-moveToLocal".equals(cmd)) {
                if (argv.length < 3) {
                    FsShell.printUsage(cmd);
                    return exitCode;
                }
            } else if ("-mv".equals(cmd) || "-cp".equals(cmd)) {
                if (argv.length < 3) {
                    FsShell.printUsage(cmd);
                    return exitCode;
                }
            } else if (("-rm".equals(cmd) || "-rmr".equals(cmd) || "-cat".equals(cmd) || "-mkdir".equals(cmd) || "-touchz".equals(cmd) || "-stat".equals(cmd) || "-text".equals(cmd)) && argv.length < 2) {
                FsShell.printUsage(cmd);
                return exitCode;
            }
            try {
                this.init();
            }
            catch (RPC.VersionMismatch v) {
                System.err.println("Version Mismatch between client and server... command aborted.");
                return exitCode;
            }
            catch (IOException e) {
                System.err.println("Bad connection to FS. Command aborted. Exception: " + e.getLocalizedMessage());
                return exitCode;
            }
            exitCode = 0;
            try {
                if ("-put".equals(cmd) || "-copyFromLocal".equals(cmd)) {
                    Path[] srcs = new Path[argv.length - 2];
                    int j = 0;
                    while (i < argv.length - 1) {
                        srcs[j++] = new Path(argv[i++]);
                    }
                    this.copyFromLocal(srcs, argv[i++]);
                    break block59;
                }
                if ("-moveFromLocal".equals(cmd)) {
                    Path[] srcs = new Path[argv.length - 2];
                    int j = 0;
                    while (i < argv.length - 1) {
                        srcs[j++] = new Path(argv[i++]);
                    }
                    this.moveFromLocal(srcs, argv[i++]);
                    break block59;
                }
                if ("-get".equals(cmd) || "-copyToLocal".equals(cmd)) {
                    this.copyToLocal(argv, i);
                    break block59;
                }
                if ("-getmerge".equals(cmd)) {
                    if (argv.length > i + 2) {
                        this.copyMergeToLocal(argv[i++], new Path(argv[i++]), Boolean.parseBoolean(argv[i++]));
                    } else {
                        this.copyMergeToLocal(argv[i++], new Path(argv[i++]));
                    }
                    break block59;
                }
                if ("-cat".equals(cmd)) {
                    exitCode = this.doall(cmd, argv, i);
                    break block59;
                }
                if ("-text".equals(cmd)) {
                    exitCode = this.doall(cmd, argv, i);
                    break block59;
                }
                if ("-moveToLocal".equals(cmd)) {
                    this.moveToLocal(argv[i++], new Path(argv[i++]));
                    break block59;
                }
                if ("-setrep".equals(cmd)) {
                    this.setReplication(argv, i);
                    break block59;
                }
                if ("-chmod".equals(cmd) || "-chown".equals(cmd) || "-chgrp".equals(cmd)) {
                    exitCode = FsShellPermissions.changePermissions(this.fs, cmd, argv, i, this);
                    break block59;
                }
                if ("-ls".equals(cmd)) {
                    exitCode = i < argv.length ? this.doall(cmd, argv, i) : this.ls(".", false);
                    break block59;
                }
                if ("-lsr".equals(cmd)) {
                    exitCode = i < argv.length ? this.doall(cmd, argv, i) : this.ls(".", true);
                    break block59;
                }
                if ("-mv".equals(cmd)) {
                    exitCode = this.rename(argv, this.getConf());
                    break block59;
                }
                if ("-cp".equals(cmd)) {
                    exitCode = this.copy(argv, this.getConf());
                    break block59;
                }
                if ("-rm".equals(cmd)) {
                    exitCode = this.doall(cmd, argv, i);
                    break block59;
                }
                if ("-rmr".equals(cmd)) {
                    exitCode = this.doall(cmd, argv, i);
                    break block59;
                }
                if ("-expunge".equals(cmd)) {
                    this.expunge();
                    break block59;
                }
                if ("-df".equals(cmd)) {
                    if (argv.length - 1 > 0) {
                        exitCode = this.doall(cmd, argv, i);
                    } else {
                        this.df(null);
                    }
                    break block59;
                }
                if ("-du".equals(cmd)) {
                    this.du(argv, i);
                    break block59;
                }
                if ("-dus".equals(cmd)) {
                    this.dus(argv, i);
                    break block59;
                }
                if (Count.matches(cmd)) {
                    exitCode = new Count(argv, i, this.getConf()).runAll();
                    break block59;
                }
                if ("-mkdir".equals(cmd)) {
                    exitCode = this.doall(cmd, argv, i);
                    break block59;
                }
                if ("-touchz".equals(cmd)) {
                    exitCode = this.doall(cmd, argv, i);
                    break block59;
                }
                if ("-test".equals(cmd)) {
                    exitCode = this.test(argv, i);
                    break block59;
                }
                if ("-stat".equals(cmd)) {
                    if (i + 1 < argv.length) {
                        this.stat(argv[i++].toCharArray(), argv[i++]);
                    } else {
                        this.stat("%y".toCharArray(), argv[i]);
                    }
                    break block59;
                }
                if ("-help".equals(cmd)) {
                    if (i < argv.length) {
                        this.printHelp(argv[i]);
                    } else {
                        this.printHelp("");
                    }
                    break block59;
                }
                if ("-tail".equals(cmd)) {
                    this.tail(argv, i);
                    break block59;
                }
                exitCode = -1;
                System.err.println(cmd.substring(1) + ": Unknown command");
                FsShell.printUsage("");
            }
            catch (IllegalArgumentException arge) {
                exitCode = -1;
                System.err.println(cmd.substring(1) + ": " + arge.getLocalizedMessage());
                FsShell.printUsage(cmd);
            }
            catch (RemoteException e) {
                exitCode = -1;
                try {
                    String[] content = e.getLocalizedMessage().split("\n");
                    System.err.println(cmd.substring(1) + ": " + content[0]);
                }
                catch (Exception ex) {
                    System.err.println(cmd.substring(1) + ": " + ex.getLocalizedMessage());
                }
            }
            catch (IOException e) {
                exitCode = -1;
                System.err.println(cmd.substring(1) + ": " + e.getLocalizedMessage());
            }
            catch (Exception re) {
                exitCode = -1;
                System.err.println(cmd.substring(1) + ": " + re.getLocalizedMessage());
            }
        }
        return exitCode;
    }

    public void close() throws IOException {
        if (this.fs != null) {
            this.fs.close();
            this.fs = null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void main(String[] argv) throws Exception {
        int res;
        FsShell shell = new FsShell();
        try {
            res = ToolRunner.run(shell, argv);
        }
        finally {
            shell.close();
        }
        System.exit(res);
    }

    static {
        modifFmt.setTimeZone(TimeZone.getTimeZone("UTC"));
        COPYTOLOCAL_SHORT_USAGE = GET_SHORT_USAGE.replace("-get", "-copyToLocal");
    }

    private static class UsagePair {
        public String path;
        public long bytes;

        public UsagePair(String path, long bytes) {
            this.path = path;
            this.bytes = bytes;
        }
    }

    private abstract class DelayedExceptionThrowing {
        private DelayedExceptionThrowing() {
        }

        abstract void process(Path var1, FileSystem var2) throws IOException;

        final void globAndProcess(Path srcPattern, FileSystem srcFs) throws IOException {
            ArrayList<IOException> exceptions = new ArrayList<IOException>();
            for (Path p : FileUtil.stat2Paths(srcFs.globStatus(srcPattern), srcPattern)) {
                try {
                    this.process(p, srcFs);
                }
                catch (IOException ioe) {
                    exceptions.add(ioe);
                }
            }
            if (!exceptions.isEmpty()) {
                if (exceptions.size() == 1) {
                    throw (IOException)exceptions.get(0);
                }
                throw new IOException("Multiple IOExceptions: " + exceptions);
            }
        }
    }

    static abstract class CmdHandler {
        protected int errorCode = 0;
        protected boolean okToContinue = true;
        protected String cmdName;

        int getErrorCode() {
            return this.errorCode;
        }

        boolean okToContinue() {
            return this.okToContinue;
        }

        String getName() {
            return this.cmdName;
        }

        protected CmdHandler(String cmdName, FileSystem fs) {
            this.cmdName = cmdName;
        }

        public abstract void run(FileStatus var1, FileSystem var2) throws IOException;
    }

    private class TextRecordInputStream
    extends InputStream {
        SequenceFile.Reader r;
        WritableComparable key;
        Writable val;
        DataInputBuffer inbuf;
        DataOutputBuffer outbuf;

        public TextRecordInputStream(FileStatus f) throws IOException {
            Path fpath = f.getPath();
            Configuration lconf = FsShell.this.getConf();
            this.r = new SequenceFile.Reader(lconf, SequenceFile.Reader.file(fpath));
            this.key = ReflectionUtils.newInstance(this.r.getKeyClass().asSubclass(WritableComparable.class), lconf);
            this.val = ReflectionUtils.newInstance(this.r.getValueClass().asSubclass(Writable.class), lconf);
            this.inbuf = new DataInputBuffer();
            this.outbuf = new DataOutputBuffer();
        }

        @Override
        public int read() throws IOException {
            int ret;
            if (null == this.inbuf || -1 == (ret = this.inbuf.read())) {
                if (!this.r.next(this.key, this.val)) {
                    return -1;
                }
                byte[] tmp = this.key.toString().getBytes();
                this.outbuf.write(tmp, 0, tmp.length);
                this.outbuf.write(9);
                tmp = this.val.toString().getBytes();
                this.outbuf.write(tmp, 0, tmp.length);
                this.outbuf.write(10);
                this.inbuf.reset(this.outbuf.getData(), this.outbuf.getLength());
                this.outbuf.reset();
                ret = this.inbuf.read();
            }
            return ret;
        }

        @Override
        public void close() throws IOException {
            this.r.close();
            super.close();
        }
    }
}

