/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.frame.file;

import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.EnumSet;
import javax.annotation.Nullable;
import org.apache.datasketches.memory.Memory;
import org.apache.druid.frame.Frame;
import org.apache.druid.frame.file.FrameFileWriter;
import org.apache.druid.java.util.common.FileUtils;
import org.apache.druid.java.util.common.IAE;
import org.apache.druid.java.util.common.IOE;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.MappedByteBufferHandler;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.java.util.common.io.Closer;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.segment.ReferenceCountingCloseableObject;
import org.apache.druid.utils.CloseableUtils;

public class FrameFile
implements Closeable {
    private static final Logger log = new Logger(FrameFile.class);
    private final File file;
    private final long fileLength;
    private final Memory footerMemory;
    private final int maxMmapSize;
    private final int numFrames;
    private final int numPartitions;
    private final ReferenceCountingCloseableObject<Closeable> referenceCounter;
    private final Closeable referenceReleaser;
    private Memory buffer;
    private long bufferOffset;
    private Runnable bufferCloser;

    private FrameFile(File file, long fileLength, Memory footerMemory, @Nullable Memory wholeFileMemory, int maxMmapSize, int numFrames, int numPartitions, ReferenceCountingCloseableObject<Closeable> referenceCounter, Closeable referenceReleaser) {
        this.file = file;
        this.fileLength = fileLength;
        this.footerMemory = footerMemory;
        this.maxMmapSize = maxMmapSize;
        this.numFrames = numFrames;
        this.numPartitions = numPartitions;
        this.referenceCounter = referenceCounter;
        this.referenceReleaser = referenceReleaser;
        if (wholeFileMemory != null) {
            assert (wholeFileMemory.getCapacity() == fileLength);
            this.buffer = wholeFileMemory;
        }
    }

    public static FrameFile open(File file, Flag ... flags) throws IOException {
        return FrameFile.open(file, Integer.MAX_VALUE, flags);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    static FrameFile open(File file, int maxMmapSize, Flag ... flags) throws IOException {
        EnumSet<Flag> flagSet;
        EnumSet<Flag> enumSet = flagSet = flags.length == 0 ? EnumSet.noneOf(Flag.class) : EnumSet.copyOf(Arrays.asList(flags));
        if (!file.exists()) {
            throw new FileNotFoundException(StringUtils.format((String)"File [%s] not found", (Object[])new Object[]{file}));
        }
        MappedByteBufferHandler sharedMapCloser = null;
        try (RandomAccessFile randomAccessFile = new RandomAccessFile(file, "r");){
            Memory footerMemory;
            Memory wholeFileMemory;
            long fileLength = randomAccessFile.length();
            if (fileLength < (long)(FrameFileWriter.MAGIC.length + 16 + 1)) {
                throw new IOE("File [%s] is too short (size = [%,d])", new Object[]{file, fileLength});
            }
            byte[] buf = new byte[16];
            Memory bufMemory = Memory.wrap((byte[])buf, (ByteOrder)ByteOrder.LITTLE_ENDIAN);
            randomAccessFile.readFully(buf, 0, FrameFileWriter.MAGIC.length);
            if (!bufMemory.equalTo(0L, (Object)Memory.wrap((byte[])FrameFileWriter.MAGIC), 0L, (long)FrameFileWriter.MAGIC.length)) {
                throw new IOE("File [%s] is not a frame file", new Object[]{file});
            }
            randomAccessFile.seek(fileLength - 16L);
            randomAccessFile.readFully(buf, 0, 16);
            int numFrames = bufMemory.getInt(0L);
            int numPartitions = bufMemory.getInt(4L);
            int footerLength = bufMemory.getInt(8L);
            int expectedFooterChecksum = bufMemory.getInt(12L);
            if (footerLength < 0) {
                throw new ISE("Negative-size footer. Corrupt or truncated file?", new Object[0]);
            }
            if ((long)footerLength > fileLength) {
                throw new ISE("Oversize footer. Corrupt or truncated file?", new Object[0]);
            }
            if (fileLength <= (long)maxMmapSize) {
                MappedByteBufferHandler mapHandle;
                sharedMapCloser = mapHandle = FileUtils.map((RandomAccessFile)randomAccessFile, (long)0L, (long)fileLength);
                wholeFileMemory = Memory.wrap((ByteBuffer)mapHandle.get(), (ByteOrder)ByteOrder.LITTLE_ENDIAN);
                if (wholeFileMemory.getCapacity() != fileLength) {
                    throw new ISE("Memory map size does not match file size", new Object[0]);
                }
                footerMemory = wholeFileMemory.region(fileLength - (long)footerLength, (long)footerLength, ByteOrder.LITTLE_ENDIAN);
            } else {
                MappedByteBufferHandler footerMapHandle;
                sharedMapCloser = footerMapHandle = FileUtils.map((RandomAccessFile)randomAccessFile, (long)(fileLength - (long)footerLength), (long)footerLength);
                wholeFileMemory = null;
                footerMemory = Memory.wrap((ByteBuffer)footerMapHandle.get(), (ByteOrder)ByteOrder.LITTLE_ENDIAN);
            }
            if (footerMemory.getByte(0L) != 2) {
                throw new IOE("File [%s] end marker not in expected location", new Object[]{file});
            }
            int actualChecksum = (int)footerMemory.xxHash64(0L, footerMemory.getCapacity() - 4L, 0L);
            if (expectedFooterChecksum != actualChecksum) {
                throw new ISE("Expected footer checksum did not match actual checksum. Corrupt or truncated file?", new Object[0]);
            }
            if (footerLength != FrameFileWriter.footerLength(numFrames, numPartitions)) {
                throw new ISE("Expected footer length did not match actual footer length. Corrupt or truncated file?", new Object[0]);
            }
            Closer fileCloser = Closer.create();
            fileCloser.register((Closeable)sharedMapCloser);
            if (flagSet.contains((Object)Flag.DELETE_ON_CLOSE)) {
                fileCloser.register(() -> {
                    if (!file.delete()) {
                        log.warn("Could not delete frame file [%s]", new Object[]{file});
                    }
                });
            }
            ReferenceCountingCloseableObject<Closeable> referenceCounter = new ReferenceCountingCloseableObject<Closeable>((Closeable)fileCloser){};
            FrameFile frameFile = new FrameFile(file, fileLength, footerMemory, wholeFileMemory, maxMmapSize, numFrames, numPartitions, referenceCounter, referenceCounter);
            return frameFile;
        }
        catch (Throwable e) {
            if (!(e instanceof IOException)) throw CloseableUtils.closeAndWrapInCatch((Throwable)e, sharedMapCloser);
            throw CloseableUtils.closeInCatch((Throwable)((IOException)e), sharedMapCloser);
        }
    }

    public int numFrames() {
        return this.numFrames;
    }

    public int numPartitions() {
        return this.numPartitions;
    }

    public int getPartitionStartFrame(int partition) {
        this.checkOpen();
        if (partition < 0) {
            throw new IAE("Partition [%,d] out of bounds", new Object[]{partition});
        }
        if (partition >= this.numPartitions) {
            return this.numFrames;
        }
        long partitionStartFrameLocation = this.footerMemory.getCapacity() - 16L - (long)this.numFrames * 8L - (long)(this.numPartitions - partition) * 4L;
        return this.footerMemory.getInt(partitionStartFrameLocation);
    }

    public Frame frame(int frameNumber) {
        this.checkOpen();
        if (frameNumber < 0 || frameNumber >= this.numFrames) {
            throw new IAE("Frame [%,d] out of bounds", new Object[]{frameNumber});
        }
        long frameEnd = this.getFrameEndPosition(frameNumber);
        long frameStart = frameNumber == 0 ? (long)(FrameFileWriter.MAGIC.length + 1) : this.getFrameEndPosition(frameNumber - 1) + 1L;
        if (this.buffer == null || frameStart < this.bufferOffset || frameEnd > this.bufferOffset + this.buffer.getCapacity()) {
            this.remapBuffer(frameStart);
        }
        if (frameStart < this.bufferOffset || frameEnd > this.bufferOffset + this.buffer.getCapacity()) {
            throw new ISE("Frame [%,d] too large (max size = %,d bytes)", new Object[]{frameNumber, this.maxMmapSize});
        }
        return Frame.decompress(this.buffer, frameStart - this.bufferOffset, frameEnd - frameStart);
    }

    public FrameFile newReference() {
        Closeable releaser = this.referenceCounter.incrementReferenceAndDecrementOnceCloseable().orElseThrow(() -> new ISE("Frame file is closed", new Object[0]));
        return new FrameFile(this.file, this.fileLength, this.footerMemory, this.bufferOffset == 0L && this.bufferCloser == null ? this.buffer : null, this.maxMmapSize, this.numFrames, this.numPartitions, this.referenceCounter, releaser);
    }

    public File file() {
        return this.file;
    }

    @Override
    public void close() throws IOException {
        CloseableUtils.closeAll(this::releaseBuffer, (Closeable[])new Closeable[]{this.referenceReleaser});
    }

    private void checkOpen() {
        if (this.referenceCounter.isClosed()) {
            throw new ISE("Frame file is closed", new Object[0]);
        }
    }

    private long getFrameEndPosition(int frameNumber) {
        assert (frameNumber >= 0 && frameNumber < this.numFrames);
        long frameEndPointerPosition = this.footerMemory.getCapacity() - 16L - (long)(this.numFrames - frameNumber) * 8L;
        long frameEndPosition = this.footerMemory.getLong(frameEndPointerPosition);
        if (frameEndPosition < 0L || frameEndPosition > this.fileLength - this.footerMemory.getCapacity()) {
            throw new ISE("Corrupt frame file: frame [%,d] location out of range", new Object[]{frameNumber});
        }
        return frameEndPosition;
    }

    private void remapBuffer(long offset) {
        MappedByteBufferHandler mapHandle;
        this.releaseBuffer();
        if (offset >= this.fileLength) {
            throw new IAE("Offset [%,d] out of range for file length [%,d]", new Object[]{offset, this.fileLength});
        }
        try {
            mapHandle = FileUtils.map((File)this.file, (long)offset, (long)Math.min(this.fileLength - offset, (long)this.maxMmapSize));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        this.buffer = Memory.wrap((ByteBuffer)mapHandle.get(), (ByteOrder)ByteOrder.LITTLE_ENDIAN);
        this.bufferCloser = () -> ((MappedByteBufferHandler)mapHandle).close();
        this.bufferOffset = offset;
    }

    private void releaseBuffer() {
        try {
            if (this.bufferCloser != null) {
                this.bufferCloser.run();
            }
        }
        finally {
            this.buffer = null;
            this.bufferCloser = null;
        }
    }

    public static enum Flag {
        DELETE_ON_CLOSE;

    }
}

