/*
 * Decompiled with CFR 0.152.
 */
package eu.maveniverse.maven.nisse.source.osdetector;

import eu.maveniverse.maven.nisse.core.NisseConfiguration;
import eu.maveniverse.maven.nisse.core.PropertySource;
import java.io.BufferedReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.inject.Named;
import javax.inject.Singleton;

@Singleton
@Named(value="os")
public class OsDetectorPropertySource
implements PropertySource {
    public static final String NAME = "os";
    private static final String DETECTED_NAME = "name";
    private static final String DETECTED_ARCH = "arch";
    private static final String DETECTED_BITNESS = "bitness";
    private static final String DETECTED_VERSION = "version";
    private static final String DETECTED_VERSION_MAJOR = "version.major";
    private static final String DETECTED_VERSION_MINOR = "version.minor";
    private static final String DETECTED_CLASSIFIER = "classifier";
    private static final String DETECTED_RELEASE = "release";
    private static final String DETECTED_RELEASE_VERSION = "release.version";
    private static final String DETECTED_RELEASE_LIKE_PREFIX = "release.like.";
    private static final String UNKNOWN = "unknown";
    private static final String LINUX_ID_PREFIX = "ID=";
    private static final String LINUX_ID_LIKE_PREFIX = "ID_LIKE=";
    private static final String LINUX_VERSION_ID_PREFIX = "VERSION_ID=";
    private static final String[] LINUX_OS_RELEASE_FILES = new String[]{"/etc/os-release", "/usr/lib/os-release"};
    private static final String REDHAT_RELEASE_FILE = "/etc/redhat-release";
    private static final String[] DEFAULT_REDHAT_VARIANTS = new String[]{"rhel", "fedora"};
    private static final Pattern VERSION_REGEX = Pattern.compile("((\\d+)\\.(\\d+)).*");
    private static final Pattern REDHAT_MAJOR_VERSION_REGEX = Pattern.compile("(\\d+)");

    public String getName() {
        return NAME;
    }

    public Map<String, String> getProperties(NisseConfiguration configuration) {
        return OsDetectorPropertySource.detectOs(configuration.getSystemProperties());
    }

    private static Map<String, String> detectOs(Map<String, String> systemProperties) {
        LinuxRelease linuxRelease;
        String failOnUnknownOS;
        HashMap<String, String> result = new HashMap<String, String>();
        String osName = systemProperties.get("os.name");
        String osArch = systemProperties.get("os.arch");
        String osVersion = systemProperties.get("os.version");
        String detectedName = OsDetectorPropertySource.normalizeOs(osName);
        String detectedArch = OsDetectorPropertySource.normalizeArch(osArch);
        int detectedBitness = OsDetectorPropertySource.determineBitness(systemProperties, detectedArch);
        result.put(DETECTED_NAME, detectedName);
        result.put(DETECTED_ARCH, detectedArch);
        result.put(DETECTED_BITNESS, "" + detectedBitness);
        Matcher versionMatcher = VERSION_REGEX.matcher(osVersion);
        if (versionMatcher.matches()) {
            result.put(DETECTED_VERSION, versionMatcher.group(1));
            result.put(DETECTED_VERSION_MAJOR, versionMatcher.group(2));
            result.put(DETECTED_VERSION_MINOR, versionMatcher.group(3));
        }
        if (!"false".equalsIgnoreCase(failOnUnknownOS = systemProperties.get("failOnUnknownOS"))) {
            if (UNKNOWN.equals(detectedName)) {
                throw new IllegalStateException("unknown os.name: " + osName);
            }
            if (UNKNOWN.equals(detectedArch)) {
                throw new IllegalStateException("unknown os.arch: " + osArch);
            }
        }
        String detectedClassifierBuilder = detectedName + '-' + detectedArch;
        LinuxRelease linuxRelease2 = linuxRelease = "linux".equals(detectedName) ? OsDetectorPropertySource.getLinuxRelease() : null;
        if (linuxRelease != null) {
            result.put(DETECTED_RELEASE, linuxRelease.id);
            if (linuxRelease.version != null) {
                result.put(DETECTED_RELEASE_VERSION, linuxRelease.version);
            }
            for (String like : linuxRelease.like) {
                String propKey = DETECTED_RELEASE_LIKE_PREFIX + like;
                result.put(propKey, "true");
            }
        }
        result.put(DETECTED_CLASSIFIER, detectedClassifierBuilder);
        return Collections.unmodifiableMap(result);
    }

    private static String normalizeOs(String value) {
        if ((value = OsDetectorPropertySource.normalize(value)).startsWith("aix")) {
            return "aix";
        }
        if (value.startsWith("hpux")) {
            return "hpux";
        }
        if (value.startsWith("os400") && (value.length() <= 5 || !Character.isDigit(value.charAt(5)))) {
            return "os400";
        }
        if (value.startsWith("linux")) {
            return "linux";
        }
        if (value.startsWith("mac") || value.startsWith("osx")) {
            return "osx";
        }
        if (value.startsWith("freebsd")) {
            return "freebsd";
        }
        if (value.startsWith("openbsd")) {
            return "openbsd";
        }
        if (value.startsWith("netbsd")) {
            return "netbsd";
        }
        if (value.startsWith("solaris") || value.startsWith("sunos")) {
            return "sunos";
        }
        if (value.startsWith("windows")) {
            return "windows";
        }
        if (value.startsWith("zos")) {
            return "zos";
        }
        return UNKNOWN;
    }

    private static String normalizeArch(String value) {
        if ((value = OsDetectorPropertySource.normalize(value)).matches("^(x8664|amd64|ia32e|em64t|x64)$")) {
            return "x86_64";
        }
        if (value.matches("^(x8632|x86|i[3-6]86|ia32|x32)$")) {
            return "x86_32";
        }
        if (value.matches("^(ia64w?|itanium64)$")) {
            return "itanium_64";
        }
        if ("ia64n".equals(value)) {
            return "itanium_32";
        }
        if (value.matches("^(sparc|sparc32)$")) {
            return "sparc_32";
        }
        if (value.matches("^(sparcv9|sparc64)$")) {
            return "sparc_64";
        }
        if (value.matches("^(arm|arm32)$")) {
            return "arm_32";
        }
        if ("aarch64".equals(value)) {
            return "aarch_64";
        }
        if (value.matches("^(mips|mips32)$")) {
            return "mips_32";
        }
        if (value.matches("^(mipsel|mips32el)$")) {
            return "mipsel_32";
        }
        if ("mips64".equals(value)) {
            return "mips_64";
        }
        if ("mips64el".equals(value)) {
            return "mipsel_64";
        }
        if (value.matches("^(ppc|ppc32)$")) {
            return "ppc_32";
        }
        if (value.matches("^(ppcle|ppc32le)$")) {
            return "ppcle_32";
        }
        if ("ppc64".equals(value)) {
            return "ppc_64";
        }
        if ("ppc64le".equals(value)) {
            return "ppcle_64";
        }
        if ("s390".equals(value)) {
            return "s390_32";
        }
        if ("s390x".equals(value)) {
            return "s390_64";
        }
        if (value.matches("^(riscv|riscv32)$")) {
            return "riscv";
        }
        if ("riscv64".equals(value)) {
            return "riscv64";
        }
        if ("e2k".equals(value)) {
            return "e2k";
        }
        if ("loongarch64".equals(value)) {
            return "loongarch_64";
        }
        return UNKNOWN;
    }

    private static String normalize(String value) {
        if (value == null) {
            return "";
        }
        return value.toLowerCase(Locale.US).replaceAll("[^a-z0-9]+", "");
    }

    private static LinuxRelease getLinuxRelease() {
        for (String osReleaseFileName : LINUX_OS_RELEASE_FILES) {
            LinuxRelease res = OsDetectorPropertySource.parseLinuxOsReleaseFile(osReleaseFileName);
            if (res == null) continue;
            return res;
        }
        return OsDetectorPropertySource.parseLinuxRedhatReleaseFile(REDHAT_RELEASE_FILE);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static LinuxRelease parseLinuxOsReleaseFile(String fileName) {
        try (BufferedReader reader = Files.newBufferedReader(Paths.get(fileName, new String[0]));){
            String line;
            String id = null;
            String version = null;
            LinkedHashSet<String> likeSet = new LinkedHashSet<String>();
            while ((line = reader.readLine()) != null) {
                if (line.startsWith(LINUX_ID_PREFIX)) {
                    id = OsDetectorPropertySource.normalizeOsReleaseValue(line.substring(LINUX_ID_PREFIX.length()));
                    likeSet.add(id);
                    continue;
                }
                if (line.startsWith(LINUX_VERSION_ID_PREFIX)) {
                    version = OsDetectorPropertySource.normalizeOsReleaseValue(line.substring(LINUX_VERSION_ID_PREFIX.length()));
                    continue;
                }
                if (!line.startsWith(LINUX_ID_LIKE_PREFIX)) continue;
                line = OsDetectorPropertySource.normalizeOsReleaseValue(line.substring(LINUX_ID_LIKE_PREFIX.length()));
                String[] parts = line.split("\\s+");
                Collections.addAll(likeSet, parts);
            }
            if (id == null) return null;
            LinuxRelease linuxRelease = new LinuxRelease(id, version, likeSet);
            return linuxRelease;
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static LinuxRelease parseLinuxRedhatReleaseFile(String fileName) {
        try (BufferedReader reader = Files.newBufferedReader(Paths.get(fileName, new String[0]));){
            String id;
            String line = reader.readLine();
            if (line == null) return null;
            line = line.toLowerCase(Locale.US);
            String version = null;
            if (line.contains("centos")) {
                id = "centos";
            } else if (line.contains("fedora")) {
                id = "fedora";
            } else {
                if (!line.contains("red hat enterprise linux")) {
                    LinuxRelease linuxRelease = null;
                    return linuxRelease;
                }
                id = "rhel";
            }
            Matcher versionMatcher = REDHAT_MAJOR_VERSION_REGEX.matcher(line);
            if (versionMatcher.find()) {
                version = versionMatcher.group(1);
            }
            LinkedHashSet<String> likeSet = new LinkedHashSet<String>(Arrays.asList(DEFAULT_REDHAT_VARIANTS));
            likeSet.add(id);
            LinuxRelease linuxRelease = new LinuxRelease(id, version, likeSet);
            return linuxRelease;
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return null;
    }

    private static String normalizeOsReleaseValue(String value) {
        return value.trim().replace("\"", "");
    }

    private static int determineBitness(Map<String, String> systemProperties, String architecture) {
        String bitness = systemProperties.getOrDefault("sun.arch.data.model", "");
        if (!bitness.isEmpty() && bitness.matches("[0-9]+")) {
            return Integer.parseInt(bitness, 10);
        }
        bitness = systemProperties.getOrDefault("com.ibm.vm.bitmode", "");
        if (!bitness.isEmpty() && bitness.matches("[0-9]+")) {
            return Integer.parseInt(bitness, 10);
        }
        return OsDetectorPropertySource.guessBitnessFromArchitecture(architecture);
    }

    public static int guessBitnessFromArchitecture(String arch) {
        if (arch.contains("64")) {
            return 64;
        }
        return 32;
    }

    private static class LinuxRelease {
        final String id;
        final String version;
        final Collection<String> like;

        LinuxRelease(String id, String version, Set<String> like) {
            this.id = id;
            this.version = version;
            this.like = Collections.unmodifiableCollection(like);
        }
    }
}

