/*
 * Decompiled with CFR 0.152.
 */
package azkaban.trigger;

import azkaban.db.DatabaseOperator;
import azkaban.db.EncodingType;
import azkaban.db.SQLTransaction;
import azkaban.trigger.Trigger;
import azkaban.trigger.TriggerLoader;
import azkaban.trigger.TriggerLoaderException;
import azkaban.utils.GZIPUtils;
import azkaban.utils.JSONUtils;
import java.io.IOException;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.apache.commons.dbutils.ResultSetHandler;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class JdbcTriggerImpl
implements TriggerLoader {
    private static final String TRIGGER_TABLE_NAME = "triggers";
    private static final String GET_UPDATED_TRIGGERS = "SELECT trigger_id, trigger_source, modify_time, enc_type, data FROM triggers WHERE modify_time>=?";
    private static final String GET_ALL_TRIGGERS = "SELECT trigger_id, trigger_source, modify_time, enc_type, data FROM triggers";
    private static final String GET_ALL_TRIGGERS_ID = "SELECT trigger_id FROM triggers";
    private static final String GET_TRIGGER = "SELECT trigger_id, trigger_source, modify_time, enc_type, data FROM triggers WHERE trigger_id=?";
    private static final String ADD_TRIGGER = "INSERT INTO triggers ( modify_time) values (?)";
    private static final String REMOVE_TRIGGER = "DELETE FROM triggers WHERE trigger_id=?";
    private static final String UPDATE_TRIGGER = "UPDATE triggers SET trigger_source=?, modify_time=?, enc_type=?, data=? WHERE trigger_id=?";
    private static final Logger logger = LoggerFactory.getLogger(JdbcTriggerImpl.class);
    private final DatabaseOperator dbOperator;
    private final EncodingType defaultEncodingType = EncodingType.GZIP;

    @Inject
    public JdbcTriggerImpl(DatabaseOperator databaseOperator) {
        this.dbOperator = databaseOperator;
    }

    @Override
    public List<Trigger> getUpdatedTriggers(long lastUpdateTime) throws TriggerLoaderException {
        logger.info("Loading triggers changed since " + new DateTime(lastUpdateTime).toString());
        TriggerResultHandler handler = new TriggerResultHandler();
        try {
            List triggers = (List)this.dbOperator.query(GET_UPDATED_TRIGGERS, (ResultSetHandler)handler, new Object[]{lastUpdateTime});
            logger.info("Loaded " + triggers.size() + " triggers.");
            return triggers;
        }
        catch (SQLException ex) {
            throw new TriggerLoaderException("Loading triggers from db failed.", ex);
        }
    }

    @Override
    public List<Trigger> loadTriggers() throws TriggerLoaderException {
        logger.info("Loading all triggers from db.");
        TriggerResultHandler handler = new TriggerResultHandler();
        try {
            List triggers = (List)this.dbOperator.query(GET_ALL_TRIGGERS, (ResultSetHandler)handler);
            logger.info("Loaded " + triggers.size() + " triggers.");
            return triggers;
        }
        catch (SQLException ex) {
            throw new TriggerLoaderException("Loading triggers from db failed.", ex);
        }
    }

    @Override
    public List<Integer> loadTriggersId() throws TriggerLoaderException {
        logger.info("Loading all triggers Id from db.");
        TriggerIdResultHandler handler = new TriggerIdResultHandler();
        try {
            List triggersId = (List)this.dbOperator.query(GET_ALL_TRIGGERS_ID, (ResultSetHandler)handler);
            logger.info("Loaded " + triggersId.size() + " triggersId.");
            return triggersId;
        }
        catch (SQLException ex) {
            throw new TriggerLoaderException("Loading triggersId from db failed.", ex);
        }
    }

    @Override
    public void removeTrigger(Trigger t) throws TriggerLoaderException {
        logger.info("Removing trigger " + t.toString() + " from db.");
        try {
            int removes = this.dbOperator.update(REMOVE_TRIGGER, new Object[]{t.getTriggerId()});
            if (removes == 0) {
                throw new TriggerLoaderException("No trigger has been removed.");
            }
        }
        catch (SQLException ex) {
            throw new TriggerLoaderException("Remove trigger " + t.getTriggerId() + " from db failed. ", ex);
        }
    }

    @Override
    public synchronized void addTrigger(Trigger t) throws TriggerLoaderException {
        logger.info("Inserting trigger " + t.toString() + " into db.");
        SQLTransaction insertAndGetLastID = transOperator -> {
            transOperator.update(ADD_TRIGGER, new Object[]{DateTime.now().getMillis()});
            transOperator.getConnection().commit();
            return transOperator.getLastInsertId();
        };
        try {
            long id = (Long)this.dbOperator.transaction(insertAndGetLastID);
            t.setTriggerId((int)id);
            this.updateTrigger(t);
            logger.info("uploaded trigger " + t.getDescription());
        }
        catch (SQLException ex) {
            logger.error("Adding Trigger " + t.getTriggerId() + " failed.");
            throw new TriggerLoaderException("trigger id is not properly created.", ex);
        }
    }

    @Override
    public void updateTrigger(Trigger t) throws TriggerLoaderException {
        logger.info("Updating trigger " + t.getTriggerId() + " into db.");
        t.setLastModifyTime(System.currentTimeMillis());
        this.updateTrigger(t, this.defaultEncodingType);
    }

    private void updateTrigger(Trigger t, EncodingType encType) throws TriggerLoaderException {
        String json = JSONUtils.toJSON(t.toJson());
        byte[] data = null;
        try {
            byte[] stringData;
            data = stringData = json.getBytes("UTF-8");
            if (encType == EncodingType.GZIP) {
                data = GZIPUtils.gzipBytes(stringData);
            }
            logger.debug("NumChars: " + json.length() + " UTF-8:" + stringData.length + " Gzip:" + data.length);
        }
        catch (IOException e) {
            logger.error("Trigger encoding fails", (Throwable)e);
            throw new TriggerLoaderException("Error encoding the trigger " + t.toString(), e);
        }
        try {
            int updates = this.dbOperator.update(UPDATE_TRIGGER, new Object[]{t.getSource(), t.getLastModifyTime(), encType.getNumVal(), data, t.getTriggerId()});
            if (updates == 0) {
                throw new TriggerLoaderException("No trigger has been updated.");
            }
        }
        catch (SQLException ex) {
            logger.error("Updating Trigger " + t.getTriggerId() + " failed.");
            throw new TriggerLoaderException("DB Trigger update failed. ", ex);
        }
    }

    @Override
    public Trigger loadTrigger(int triggerId) throws TriggerLoaderException {
        logger.info("Loading trigger " + triggerId + " from db.");
        TriggerResultHandler handler = new TriggerResultHandler();
        try {
            List triggers = (List)this.dbOperator.query(GET_TRIGGER, (ResultSetHandler)handler, new Object[]{triggerId});
            if (triggers.size() == 0) {
                logger.error("Loaded 0 triggers. Failed to load trigger " + triggerId);
                throw new TriggerLoaderException("Loaded 0 triggers. Failed to load trigger " + triggerId);
            }
            return (Trigger)triggers.get(0);
        }
        catch (SQLException ex) {
            logger.error("Failed to load trigger " + triggerId);
            throw new TriggerLoaderException("Load a specific trigger failed.", ex);
        }
    }

    public static class TriggerIdResultHandler
    implements ResultSetHandler<List<Integer>> {
        public List<Integer> handle(ResultSet rs) throws SQLException {
            if (!rs.next()) {
                return Collections.emptyList();
            }
            ArrayList<Integer> triggersId = new ArrayList<Integer>();
            do {
                int triggerId = rs.getInt(1);
                triggersId.add(triggerId);
            } while (rs.next());
            return triggersId;
        }
    }

    public static class TriggerResultHandler
    implements ResultSetHandler<List<Trigger>> {
        public List<Trigger> handle(ResultSet rs) throws SQLException {
            if (!rs.next()) {
                return Collections.emptyList();
            }
            ArrayList<Trigger> triggers = new ArrayList<Trigger>();
            do {
                int triggerId = rs.getInt(1);
                int encodingType = rs.getInt(4);
                byte[] data = rs.getBytes(5);
                Object jsonObj = null;
                if (data != null) {
                    EncodingType encType = EncodingType.fromInteger((int)encodingType);
                    try {
                        jsonObj = JSONUtils.parseJSONFromString((String)(encType == EncodingType.GZIP ? GZIPUtils.unGzipString(data, "UTF-8") : new String(data, "UTF-8")));
                    }
                    catch (IOException e) {
                        throw new SQLException("Error reconstructing trigger data ");
                    }
                }
                Trigger t = null;
                try {
                    t = Trigger.fromJson(jsonObj);
                    triggers.add(t);
                }
                catch (Exception e) {
                    logger.error("Failed to load trigger " + triggerId, (Throwable)e);
                }
            } while (rs.next());
            return triggers;
        }
    }
}

