/*
 * Decompiled with CFR 0.152.
 */
package azkaban.storage;

import azkaban.project.Project;
import azkaban.project.ProjectFileHandler;
import azkaban.project.ProjectLoader;
import azkaban.spi.Storage;
import azkaban.spi.StorageException;
import azkaban.spi.StorageMetadata;
import azkaban.storage.DatabaseStorage;
import azkaban.storage.StorageCleaner;
import azkaban.user.User;
import azkaban.utils.Md5Hasher;
import azkaban.utils.Props;
import com.google.common.base.Preconditions;
import com.webank.wedatasphere.schedulis.common.function.CheckedSupplier;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Objects;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class StorageManager {
    private static final Logger logger = LoggerFactory.getLogger(StorageManager.class);
    private final StorageCleaner storageCleaner;
    private final Storage storage;
    private final ProjectLoader projectLoader;
    private final File tempDir;

    @Inject
    public StorageManager(Props props, Storage storage, ProjectLoader projectLoader, StorageCleaner storageCleaner) {
        this.tempDir = new File(props.getString("project.temp.dir", "temp"));
        this.storage = Objects.requireNonNull(storage, "storage is null");
        this.projectLoader = Objects.requireNonNull(projectLoader, "projectLoader is null");
        this.storageCleaner = Objects.requireNonNull(storageCleaner, "storageCleanUp is null");
        this.prepareTempDir();
    }

    private void prepareTempDir() {
        if (!this.tempDir.exists()) {
            this.tempDir.mkdirs();
        }
        Preconditions.checkArgument((boolean)this.tempDir.isDirectory());
    }

    public void uploadProject(Project project, int version, File localFile, User uploader) {
        byte[] md5 = null;
        if (!(this.storage instanceof DatabaseStorage)) {
            md5 = this.computeHash(localFile);
        }
        StorageMetadata metadata = new StorageMetadata(project.getId(), version, uploader.getUserId(), md5);
        logger.info(String.format("Adding archive to storage. Meta:%s File: %s[%d bytes]", metadata, localFile.getName(), localFile.length()));
        String resourceId = null;
        resourceId = this.storage.put(metadata, localFile);
        if (!(this.storage instanceof DatabaseStorage)) {
            this.projectLoader.addProjectVersion(project.getId(), version, localFile, uploader.getUserId(), Objects.requireNonNull(md5), Objects.requireNonNull(resourceId));
            logger.info(String.format("Added project metadata to DB. Meta:%s File: %s[%d bytes] URI: %s", metadata, localFile.getName(), localFile.length(), resourceId));
        }
    }

    public void cleanupProjectArtifacts(int projectId) {
        try {
            this.storageCleaner.cleanupProjectArtifacts(projectId);
        }
        catch (Exception e) {
            logger.error("Error occured during cleanup. Ignoring and continuing...", (Throwable)e);
        }
    }

    private byte[] computeHash(File localFile) {
        byte[] md5;
        try {
            md5 = Md5Hasher.md5Hash(localFile);
        }
        catch (IOException e) {
            throw new StorageException((Throwable)e);
        }
        return md5;
    }

    public ProjectFileHandler getProjectFile(int projectId, int version) {
        logger.info(String.format("Fetching project file. project ID: %d version: %d", projectId, version));
        if (this.storage instanceof DatabaseStorage) {
            return ((DatabaseStorage)this.storage).get(projectId, version);
        }
        ProjectFileHandler pfh = this.projectLoader.fetchProjectMetaData(projectId, version);
        String resourceId = Objects.requireNonNull(pfh.getResourceId(), String.format("URI is null. project ID: %d version: %d", pfh.getProjectId(), pfh.getVersion()));
        return this.getProjectFileFromFileSystem(pfh, () -> this.storage.get(resourceId));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private ProjectFileHandler getProjectFileFromFileSystem(ProjectFileHandler pfh, CheckedSupplier<InputStream, IOException> supplier) {
        try (InputStream is = supplier.get();){
            File file = this.createTempOutputFile(pfh);
            try (FileOutputStream fos = new FileOutputStream(file);){
                IOUtils.copy((InputStream)is, (OutputStream)fos);
            }
            this.validateChecksum(file, pfh);
            pfh.setLocalFile(file);
            ProjectFileHandler projectFileHandler = pfh;
            return projectFileHandler;
        }
        catch (IOException e) {
            throw new StorageException((Throwable)e);
        }
    }

    private void validateChecksum(File file, ProjectFileHandler pfh) throws IOException {
        byte[] hash = Md5Hasher.md5Hash(file);
        Preconditions.checkState((boolean)Arrays.equals(pfh.getMd5Hash(), hash), (Object)String.format("MD5 HASH Failed. project ID: %d version: %d Expected: %s Actual: %s", pfh.getProjectId(), pfh.getVersion(), new String(pfh.getMd5Hash(), StandardCharsets.UTF_8), new String(hash, StandardCharsets.UTF_8)));
    }

    private File createTempOutputFile(ProjectFileHandler projectFileHandler) throws IOException {
        return File.createTempFile(projectFileHandler.getFileName(), String.valueOf(projectFileHandler.getVersion()), this.tempDir);
    }
}

