/*
 * Decompiled with CFR 0.152.
 */
package azkaban.project;

import azkaban.db.DatabaseOperator;
import azkaban.db.DatabaseTransOperator;
import azkaban.db.EncodingType;
import azkaban.db.SQLTransaction;
import azkaban.flow.Flow;
import azkaban.project.JdbcProjectHandlerSet;
import azkaban.project.Project;
import azkaban.project.ProjectFileHandler;
import azkaban.project.ProjectLoader;
import azkaban.project.ProjectLogEvent;
import azkaban.project.ProjectManagerException;
import azkaban.user.Permission;
import azkaban.user.User;
import azkaban.utils.GZIPUtils;
import azkaban.utils.JSONUtils;
import azkaban.utils.Md5Hasher;
import azkaban.utils.Pair;
import azkaban.utils.Props;
import azkaban.utils.PropsUtils;
import azkaban.utils.Triple;
import com.google.common.io.Files;
import com.webank.wedatasphere.schedulis.common.project.entity.ProjectPermission;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.apache.commons.dbutils.ResultSetHandler;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class JdbcProjectImpl
implements ProjectLoader {
    private static final Logger logger = LoggerFactory.getLogger(JdbcProjectImpl.class);
    private static final int CHUCK_SIZE = 0xA00000;
    private static final int MAX_FLOW_FILE_SIZE_IN_BYTES = 0xA00000;
    private final DatabaseOperator dbOperator;
    private final File tempDir;
    private final EncodingType defaultEncodingType = EncodingType.GZIP;

    @Inject
    public JdbcProjectImpl(Props props, DatabaseOperator databaseOperator) {
        this.dbOperator = databaseOperator;
        this.tempDir = new File(props.getString("project.temp.dir", "temp"));
        if (!this.tempDir.exists()) {
            if (this.tempDir.mkdirs()) {
                logger.info("project temporary folder is being constructed.");
            } else {
                logger.info("project temporary folder already existed.");
            }
        }
    }

    @Override
    public List<Project> fetchAllActiveProjects() throws ProjectManagerException {
        JdbcProjectHandlerSet.ProjectResultHandler handler = new JdbcProjectHandlerSet.ProjectResultHandler();
        List projects = null;
        try {
            String serchSQL = JdbcProjectHandlerSet.ProjectResultHandler.SELECT_ALL_ACTIVE_PROJECTS;
            serchSQL = serchSQL + " ORDER BY name";
            projects = (List)this.dbOperator.query(serchSQL, (ResultSetHandler)handler);
            projects.forEach(project -> {
                for (ProjectPermission projectPermission : this.fetchAllPermissionsForProject((Project)project)) {
                    this.setProjectAllPermission((Project)project, projectPermission);
                }
            });
        }
        catch (SQLException ex) {
            logger.error(JdbcProjectHandlerSet.ProjectResultHandler.SELECT_PROJECT_BY_ID + " failed.", (Throwable)ex);
            throw new ProjectManagerException("Error retrieving all projects", ex);
        }
        return projects;
    }

    private void setProjectPermission(Project project, Triple<String, Boolean, Permission> perm) {
        if (perm.getSecond().booleanValue()) {
            project.setGroupPermission(perm.getFirst(), perm.getThird());
        } else {
            project.setUserPermission(perm.getFirst(), perm.getThird());
        }
    }

    @Override
    public Project fetchProjectById(int id) throws ProjectManagerException {
        Project project = null;
        JdbcProjectHandlerSet.ProjectResultHandler handler = new JdbcProjectHandlerSet.ProjectResultHandler();
        try {
            List projects = (List)this.dbOperator.query(JdbcProjectHandlerSet.ProjectResultHandler.SELECT_PROJECT_BY_ID, (ResultSetHandler)handler, new Object[]{id});
            if (projects.isEmpty()) {
                throw new ProjectManagerException("No project with id " + id + " exists in db.");
            }
            project = (Project)projects.get(0);
            for (Triple<String, Boolean, Permission> perm : this.fetchPermissionsForProject(project)) {
                if (perm.getThird().toFlags() == 0) continue;
                this.setProjectPermission(project, perm);
            }
        }
        catch (SQLException ex) {
            logger.error(JdbcProjectHandlerSet.ProjectResultHandler.SELECT_PROJECT_BY_ID + " failed.", (Throwable)ex);
            throw new ProjectManagerException("Query for existing project failed. Project " + id, ex);
        }
        return project;
    }

    @Override
    public Project fetchProjectByName(String name) throws ProjectManagerException {
        Project project = null;
        JdbcProjectHandlerSet.ProjectResultHandler handler = new JdbcProjectHandlerSet.ProjectResultHandler();
        try {
            List projects = (List)this.dbOperator.query(JdbcProjectHandlerSet.ProjectResultHandler.SELECT_ACTIVE_PROJECT_BY_NAME, (ResultSetHandler)handler, new Object[]{name});
            if (projects.isEmpty() && (projects = (List)this.dbOperator.query(JdbcProjectHandlerSet.ProjectResultHandler.SELECT_PROJECT_BY_NAME, (ResultSetHandler)handler, new Object[]{name})).isEmpty()) {
                throw new ProjectManagerException("No project with name " + name + " exists in db.");
            }
            project = (Project)projects.get(0);
            for (Triple<String, Boolean, Permission> perm : this.fetchPermissionsForProject(project)) {
                if (perm.getThird().toFlags() == 0) continue;
                this.setProjectPermission(project, perm);
            }
        }
        catch (SQLException ex) {
            logger.error(JdbcProjectHandlerSet.ProjectResultHandler.SELECT_ACTIVE_PROJECT_BY_NAME + " failed.", (Throwable)ex);
            throw new ProjectManagerException(JdbcProjectHandlerSet.ProjectResultHandler.SELECT_ACTIVE_PROJECT_BY_NAME + " failed.", ex);
        }
        return project;
    }

    private List<Triple<String, Boolean, Permission>> fetchPermissionsForProject(Project project) throws ProjectManagerException {
        JdbcProjectHandlerSet.ProjectPermissionsResultHandler permHander = new JdbcProjectHandlerSet.ProjectPermissionsResultHandler();
        List permissions = null;
        try {
            permissions = (List)this.dbOperator.query(JdbcProjectHandlerSet.ProjectPermissionsResultHandler.SELECT_PROJECT_PERMISSION, (ResultSetHandler)permHander, new Object[]{project.getId()});
        }
        catch (SQLException ex) {
            logger.error(JdbcProjectHandlerSet.ProjectPermissionsResultHandler.SELECT_PROJECT_PERMISSION + " failed.", (Throwable)ex);
            throw new ProjectManagerException("Query for permissions for " + project.getName() + " failed.", ex);
        }
        return permissions;
    }

    @Override
    public List<Integer> fetchPermissionsProjectId(String user) throws ProjectManagerException {
        ResultSetHandler handler = rs -> {
            if (!rs.next()) {
                return Collections.emptyList();
            }
            ArrayList<Integer> projectIds = new ArrayList<Integer>();
            do {
                int projectId = rs.getInt(1);
                projectIds.add(projectId);
            } while (rs.next());
            return projectIds;
        };
        String sql = "SELECT project_id FROM project_permissions WHERE `name` = ? ;";
        ArrayList<Integer> projectIds = new ArrayList();
        try {
            projectIds = (List)this.dbOperator.query(sql, handler, new Object[]{user});
        }
        catch (SQLException ex) {
            logger.error("exec sql:{} failed.", (Object)sql, (Object)ex);
            throw new ProjectManagerException("Query for permissions by " + user + " failed.", ex);
        }
        return projectIds;
    }

    @Override
    public synchronized Project createNewProject(String name, String description, User creator) throws ProjectManagerException {
        JdbcProjectHandlerSet.ProjectResultHandler handler = new JdbcProjectHandlerSet.ProjectResultHandler();
        try {
            List projects = (List)this.dbOperator.query(JdbcProjectHandlerSet.ProjectResultHandler.SELECT_ACTIVE_PROJECT_BY_NAME, (ResultSetHandler)handler, new Object[]{name});
            if (!projects.isEmpty()) {
                throw new ProjectManagerException("Active project with name " + name + " already exists in db.");
            }
        }
        catch (SQLException ex) {
            logger.error("", (Throwable)ex);
            throw new ProjectManagerException("Checking for existing project failed. " + name, ex);
        }
        String INSERT_PROJECT = "INSERT INTO projects ( name, active, modified_time, create_time, version, last_modified_by, description, create_user, enc_type, settings_blob) values (?,?,?,?,?,?,?,?,?,?)";
        SQLTransaction insertProject = transOperator -> {
            long time = System.currentTimeMillis();
            return transOperator.update("INSERT INTO projects ( name, active, modified_time, create_time, version, last_modified_by, description, create_user, enc_type, settings_blob) values (?,?,?,?,?,?,?,?,?,?)", new Object[]{name, true, time, time, null, creator.getUserId(), description, creator.getUserId(), this.defaultEncodingType.getNumVal(), null});
        };
        try {
            int numRowsInserted = (Integer)this.dbOperator.transaction(insertProject);
            if (numRowsInserted == 0) {
                throw new ProjectManagerException("No projects have been inserted.");
            }
        }
        catch (SQLException ex) {
            logger.error("INSERT INTO projects ( name, active, modified_time, create_time, version, last_modified_by, description, create_user, enc_type, settings_blob) values (?,?,?,?,?,?,?,?,?,?) failed.", (Throwable)ex);
            throw new ProjectManagerException("Insert project" + name + " for existing project failed. ", ex);
        }
        return this.fetchProjectByName(name);
    }

    @Override
    public void uploadProjectFile(int projectId, int version, File localFile, String uploader) throws ProjectManagerException {
        SQLTransaction uploadProjectFileTransaction = transOperator -> {
            this.addProjectToProjectVersions(transOperator, projectId, version, localFile, uploader, this.computeHash(localFile), null);
            transOperator.getConnection().commit();
            int chunks = this.uploadFileInChunks(transOperator, projectId, version, localFile);
            this.updateChunksInProjectVersions(transOperator, projectId, version, chunks);
            return 1;
        };
        this.uploadProjectFile(projectId, localFile, (SQLTransaction<Integer>)uploadProjectFileTransaction);
    }

    @Override
    public void uploadProjectFile(int projectId, int version, File localFile, String uploader, String resourceID) throws ProjectManagerException {
        SQLTransaction uploadProjectFileTransaction = transOperator -> {
            this.addProjectToProjectVersions(transOperator, projectId, version, localFile, uploader, this.computeHash(localFile), resourceID);
            transOperator.getConnection().commit();
            int chunks = this.uploadFileInChunks(transOperator, projectId, version, localFile);
            this.updateChunksInProjectVersions(transOperator, projectId, version, chunks);
            return 1;
        };
        this.uploadProjectFile(projectId, localFile, (SQLTransaction<Integer>)uploadProjectFileTransaction);
    }

    private void uploadProjectFile(int projectId, File localFile, SQLTransaction<Integer> uploadProjectFileTransaction) throws ProjectManagerException {
        long startMs = System.currentTimeMillis();
        logger.info(String.format("Uploading Project ID: %d file: %s [%d bytes]", projectId, localFile.getName(), localFile.length()));
        try {
            this.dbOperator.transaction(uploadProjectFileTransaction);
        }
        catch (SQLException e) {
            logger.error("upload project files failed.", (Throwable)e);
            throw new ProjectManagerException("upload project files failed.", e);
        }
        long duration = (System.currentTimeMillis() - startMs) / 1000L;
        logger.info(String.format("Uploaded Project ID: %d file: %s [%d bytes] in %d sec", projectId, localFile.getName(), localFile.length(), duration));
    }

    private byte[] computeHash(File localFile) {
        byte[] md5;
        logger.info("Creating message digest for upload " + localFile.getName());
        try {
            md5 = Md5Hasher.md5Hash(localFile);
        }
        catch (IOException e) {
            throw new ProjectManagerException("Error getting md5 hash.", e);
        }
        logger.info("Md5 hash created");
        return md5;
    }

    @Override
    public void addProjectVersion(int projectId, int version, File localFile, String uploader, byte[] md5, String resourceId) throws ProjectManagerException {
        SQLTransaction transaction = transOperator -> {
            this.addProjectToProjectVersions(transOperator, projectId, version, localFile, uploader, md5, resourceId);
            return 1;
        };
        try {
            this.dbOperator.transaction(transaction);
        }
        catch (SQLException e) {
            logger.error("addProjectVersion failed.", (Throwable)e);
            throw new ProjectManagerException("addProjectVersion failed.", e);
        }
    }

    private void addProjectToProjectVersions(DatabaseTransOperator transOperator, int projectId, int version, File localFile, String uploader, byte[] md5, String resourceId) throws ProjectManagerException {
        long updateTime = System.currentTimeMillis();
        String INSERT_PROJECT_VERSION = "INSERT INTO project_versions (project_id, version, upload_time, uploader, file_type, file_name, md5, num_chunks, resource_id) values (?,?,?,?,?,?,?,?,?)";
        try {
            transOperator.update("INSERT INTO project_versions (project_id, version, upload_time, uploader, file_type, file_name, md5, num_chunks, resource_id) values (?,?,?,?,?,?,?,?,?)", new Object[]{projectId, version, updateTime, uploader, Files.getFileExtension((String)localFile.getName()), localFile.getName(), md5, 0, resourceId});
        }
        catch (SQLException e) {
            String msg = String.format("Error initializing project id: %d version: %d ", projectId, version);
            logger.error(msg, (Throwable)e);
            throw new ProjectManagerException(msg, e);
        }
    }

    private int uploadFileInChunks(DatabaseTransOperator transOperator, int projectId, int version, File localFile) throws ProjectManagerException {
        byte[] buffer = new byte[0xA00000];
        String INSERT_PROJECT_FILES = "INSERT INTO project_files (project_id, version, chunk, size, file) values (?,?,?,?,?)";
        BufferedInputStream bufferedStream = null;
        int chunk = 0;
        try {
            bufferedStream = new BufferedInputStream(new FileInputStream(localFile));
            int size = bufferedStream.read(buffer);
            while (size >= 0) {
                logger.info("Read bytes for " + localFile.getName() + " size:" + size);
                byte[] buf = buffer;
                if (size < buffer.length) {
                    buf = Arrays.copyOfRange(buffer, 0, size);
                }
                try {
                    logger.info("Running update for " + localFile.getName() + " chunk " + chunk);
                    transOperator.update("INSERT INTO project_files (project_id, version, chunk, size, file) values (?,?,?,?,?)", new Object[]{projectId, version, chunk, size, buf});
                    transOperator.getConnection().commit();
                    logger.info("Finished update for " + localFile.getName() + " chunk " + chunk);
                }
                catch (SQLException e) {
                    throw new ProjectManagerException("Error Chunking during uploading files to db...");
                }
                ++chunk;
                size = bufferedStream.read(buffer);
            }
        }
        catch (IOException e) {
            try {
                throw new ProjectManagerException(String.format("Error chunking file. projectId: %d, version: %d, file:%s[%d bytes], chunk: %d", projectId, version, localFile.getName(), localFile.length(), chunk));
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(bufferedStream);
                throw throwable;
            }
        }
        IOUtils.closeQuietly((InputStream)bufferedStream);
        return chunk;
    }

    private void updateChunksInProjectVersions(DatabaseTransOperator transOperator, int projectId, int version, int chunk) throws ProjectManagerException {
        String UPDATE_PROJECT_NUM_CHUNKS = "UPDATE project_versions SET num_chunks=? WHERE project_id=? AND version=?";
        try {
            transOperator.update("UPDATE project_versions SET num_chunks=? WHERE project_id=? AND version=?", new Object[]{chunk, projectId, version});
            transOperator.getConnection().commit();
        }
        catch (SQLException e) {
            logger.error("Error updating project " + projectId + " : chunk_num " + chunk, (Throwable)e);
            throw new ProjectManagerException("Error updating project " + projectId + " : chunk_num " + chunk, e);
        }
    }

    @Override
    public ProjectFileHandler fetchProjectMetaData(int projectId, int version) {
        JdbcProjectHandlerSet.ProjectVersionResultHandler pfHandler = new JdbcProjectHandlerSet.ProjectVersionResultHandler();
        try {
            List projectFiles = (List)this.dbOperator.query(JdbcProjectHandlerSet.ProjectVersionResultHandler.SELECT_PROJECT_VERSION, (ResultSetHandler)pfHandler, new Object[]{projectId, version});
            if (projectFiles == null || projectFiles.isEmpty()) {
                return null;
            }
            return (ProjectFileHandler)projectFiles.get(0);
        }
        catch (SQLException ex) {
            logger.error("Query for uploaded file for project id " + projectId + " failed.", (Throwable)ex);
            throw new ProjectManagerException("Query for uploaded file for project id " + projectId + " failed.", ex);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ProjectFileHandler getUploadedFile(int projectId, int version) throws ProjectManagerException {
        byte[] md5;
        File file;
        ProjectFileHandler projHandler = this.fetchProjectMetaData(projectId, version);
        if (projHandler == null) {
            return null;
        }
        int numChunks = projHandler.getNumChunks();
        if (numChunks <= 0) {
            throw new ProjectManagerException(String.format("Got numChunks=%s for version %s of project %s - seems like this version has been cleaned up already, because enough newer versions have been uploaded. To increase the retention of project versions, set %s", numChunks, version, projectId, "project.version.retention"));
        }
        BufferedOutputStream bStream = null;
        try {
            try {
                file = File.createTempFile(projHandler.getFileName(), String.valueOf(version), this.tempDir);
                bStream = new BufferedOutputStream(new FileOutputStream(file));
            }
            catch (IOException e) {
                throw new ProjectManagerException("Error creating temp file for stream.");
            }
            int collect = 5;
            int fromChunk = 0;
            int toChunk = 5;
            do {
                JdbcProjectHandlerSet.ProjectFileChunkResultHandler chunkHandler = new JdbcProjectHandlerSet.ProjectFileChunkResultHandler();
                List data = null;
                try {
                    data = (List)this.dbOperator.query(JdbcProjectHandlerSet.ProjectFileChunkResultHandler.SELECT_PROJECT_CHUNKS_FILE, (ResultSetHandler)chunkHandler, new Object[]{projectId, version, fromChunk, toChunk});
                }
                catch (SQLException e) {
                    logger.error("", (Throwable)e);
                    throw new ProjectManagerException("Query for uploaded file for " + projectId + " failed.", e);
                }
                try {
                    for (byte[] d : data) {
                        bStream.write(d);
                    }
                }
                catch (IOException e) {
                    throw new ProjectManagerException("Error writing file", e);
                }
                toChunk += 5;
            } while ((fromChunk += 5) <= numChunks);
        }
        catch (Throwable throwable) {
            IOUtils.closeQuietly(bStream);
            throw throwable;
        }
        IOUtils.closeQuietly((OutputStream)bStream);
        try {
            md5 = Md5Hasher.md5Hash(file);
        }
        catch (IOException e) {
            throw new ProjectManagerException("Error getting md5 hash.", e);
        }
        if (!Arrays.equals(projHandler.getMd5Hash(), md5)) {
            throw new ProjectManagerException(String.format("Md5 Hash failed on project %s version %s retrieval of file %s. Expected hash: %s , got hash: %s", projHandler.getProjectId(), projHandler.getVersion(), file.getAbsolutePath(), Arrays.toString(projHandler.getMd5Hash()), Arrays.toString(md5)));
        }
        logger.info("Md5 Hash is valid");
        projHandler.setLocalFile(file);
        return projHandler;
    }

    @Override
    public void changeProjectVersion(Project project, int version, String user) throws ProjectManagerException {
        long timestamp = System.currentTimeMillis();
        try {
            String UPDATE_PROJECT_VERSION = "UPDATE projects SET version=?,modified_time=?,last_modified_by=? WHERE id=?";
            this.dbOperator.update("UPDATE projects SET version=?,modified_time=?,last_modified_by=? WHERE id=?", new Object[]{version, timestamp, user, project.getId()});
            project.setVersion(version);
            project.setLastModifiedTimestamp(timestamp);
            project.setLastModifiedUser(user);
        }
        catch (SQLException e) {
            logger.error("Error updating switching project version " + project.getName(), (Throwable)e);
            throw new ProjectManagerException("Error updating switching project version " + project.getName(), e);
        }
    }

    @Override
    public void updatePermission(Project project, String name, Permission perm, boolean isGroup) throws ProjectManagerException {
        long updateTime = System.currentTimeMillis();
        try {
            if (this.dbOperator.getDataSource().allowsOnDuplicateKey()) {
                String INSERT_PROJECT_PERMISSION = "INSERT INTO project_permissions (project_id, modified_time, name, permissions, isGroup) values (?,?,?,?,?)ON DUPLICATE KEY UPDATE modified_time = VALUES(modified_time), permissions = VALUES(permissions)";
                this.dbOperator.update("INSERT INTO project_permissions (project_id, modified_time, name, permissions, isGroup) values (?,?,?,?,?)ON DUPLICATE KEY UPDATE modified_time = VALUES(modified_time), permissions = VALUES(permissions)", new Object[]{project.getId(), updateTime, name, perm.toFlags(), isGroup});
            } else {
                String MERGE_PROJECT_PERMISSION = "MERGE INTO project_permissions (project_id, modified_time, name, permissions, isGroup) KEY (project_id, name) values (?,?,?,?,?)";
                this.dbOperator.update("MERGE INTO project_permissions (project_id, modified_time, name, permissions, isGroup) KEY (project_id, name) values (?,?,?,?,?)", new Object[]{project.getId(), updateTime, name, perm.toFlags(), isGroup});
            }
        }
        catch (SQLException ex) {
            logger.error("Error updating project permission", (Throwable)ex);
            throw new ProjectManagerException("Error updating project " + project.getName() + " permissions for " + name, ex);
        }
        if (isGroup) {
            project.setGroupPermission(name, perm);
            project.setUserPermission(name, perm);
        } else {
            project.setUserPermission(name, perm);
        }
    }

    @Override
    public void updateProjectSettings(Project project) throws ProjectManagerException {
        this.updateProjectSettings(project, this.defaultEncodingType);
    }

    private byte[] convertJsonToBytes(EncodingType type, String json) throws IOException {
        byte[] data = json.getBytes("UTF-8");
        if (type == EncodingType.GZIP) {
            data = GZIPUtils.gzipBytes(data);
        }
        return data;
    }

    private void updateProjectSettings(Project project, EncodingType encType) throws ProjectManagerException {
        String UPDATE_PROJECT_SETTINGS = "UPDATE projects SET enc_type=?, settings_blob=? WHERE id=?";
        String json = JSONUtils.toJSON((Object)project.toObject());
        byte[] data = null;
        try {
            data = this.convertJsonToBytes(encType, json);
            logger.debug("NumChars: " + json.length() + " Gzip:" + data.length);
        }
        catch (IOException e) {
            throw new ProjectManagerException("Failed to encode. ", e);
        }
        try {
            this.dbOperator.update("UPDATE projects SET enc_type=?, settings_blob=? WHERE id=?", new Object[]{encType.getNumVal(), data, project.getId()});
        }
        catch (SQLException e) {
            logger.error("update Project Settings failed.", (Throwable)e);
            throw new ProjectManagerException("Error updating project " + project.getName() + " version " + project.getVersion(), e);
        }
    }

    @Override
    public void removePermission(Project project, String name, boolean isGroup) throws ProjectManagerException {
        String DELETE_PROJECT_PERMISSION = "DELETE FROM project_permissions WHERE project_id=? AND name=? AND isGroup=?";
        try {
            this.dbOperator.update("DELETE FROM project_permissions WHERE project_id=? AND name=? AND isGroup=?", new Object[]{project.getId(), name, isGroup});
        }
        catch (SQLException e) {
            logger.error("remove Permission failed.", (Throwable)e);
            throw new ProjectManagerException("Error deleting project " + project.getName() + " permissions for " + name, e);
        }
        if (isGroup) {
            project.removeGroupPermission(name);
        } else {
            project.removeUserPermission(name);
        }
    }

    @Override
    public List<Triple<String, Boolean, Permission>> getProjectPermissions(Project project) throws ProjectManagerException {
        return this.fetchPermissionsForProject(project);
    }

    @Override
    public void removeProject(Project project, String user) throws ProjectManagerException {
        long updateTime = System.currentTimeMillis();
        String UPDATE_INACTIVE_PROJECT = "UPDATE projects SET active=false,modified_time=?,last_modified_by=? WHERE id=?";
        try {
            this.dbOperator.update("UPDATE projects SET active=false,modified_time=?,last_modified_by=? WHERE id=?", new Object[]{updateTime, user, project.getId()});
        }
        catch (SQLException e) {
            logger.error("error remove project " + project.getName(), (Throwable)e);
            throw new ProjectManagerException("Error remove project " + project.getName(), e);
        }
    }

    @Override
    public List<Flow> getRunningFlow(Project project) throws ProjectManagerException {
        List runningFlowList = null;
        JdbcProjectHandlerSet.ProjectRunningFlowHandler permHander = new JdbcProjectHandlerSet.ProjectRunningFlowHandler();
        try {
            runningFlowList = (List)this.dbOperator.query(JdbcProjectHandlerSet.ProjectRunningFlowHandler.QUERY_RUNNING_FLOWS, (ResultSetHandler)permHander, new Object[]{project.getId()});
        }
        catch (SQLException e) {
            logger.error("get running flow failed, by project name: " + project.getName(), (Throwable)e);
            throw new ProjectManagerException("get running flow failed, by project name: " + project.getName(), e);
        }
        return runningFlowList;
    }

    @Override
    public boolean postEvent(Project project, ProjectLogEvent.EventType type, String user, String message) {
        String INSERT_PROJECT_EVENTS = "INSERT INTO project_events (project_id, event_type, event_time, username, message) values (?,?,?,?,?)";
        long updateTime = System.currentTimeMillis();
        try {
            this.dbOperator.update("INSERT INTO project_events (project_id, event_type, event_time, username, message) values (?,?,?,?,?)", new Object[]{project.getId(), type.getNumVal(), updateTime, user, message});
        }
        catch (SQLException e) {
            logger.error("post event failed,", (Throwable)e);
            return false;
        }
        return true;
    }

    @Override
    public List<ProjectLogEvent> getProjectEvents(Project project, int num, int skip) throws ProjectManagerException {
        JdbcProjectHandlerSet.ProjectLogsResultHandler logHandler = new JdbcProjectHandlerSet.ProjectLogsResultHandler();
        List events = null;
        try {
            events = (List)this.dbOperator.query(JdbcProjectHandlerSet.ProjectLogsResultHandler.SELECT_PROJECT_EVENTS_ORDER, (ResultSetHandler)logHandler, new Object[]{project.getId(), num, skip});
        }
        catch (SQLException e) {
            logger.error("Error getProjectEvents, project " + project.getName(), (Throwable)e);
            throw new ProjectManagerException("Error getProjectEvents, project " + project.getName(), e);
        }
        return events;
    }

    @Override
    public void updateDescription(Project project, String description, String user) throws ProjectManagerException {
        String UPDATE_PROJECT_DESCRIPTION = "UPDATE projects SET description=?,modified_time=?,last_modified_by=? WHERE id=?";
        long updateTime = System.currentTimeMillis();
        try {
            this.dbOperator.update("UPDATE projects SET description=?,modified_time=?,last_modified_by=? WHERE id=?", new Object[]{description, updateTime, user, project.getId()});
            project.setDescription(description);
            project.setLastModifiedTimestamp(updateTime);
            project.setLastModifiedUser(user);
        }
        catch (SQLException e) {
            logger.error("", (Throwable)e);
            throw new ProjectManagerException("Error update Description, project " + project.getName(), e);
        }
    }

    @Override
    public int getLatestProjectVersion(Project project) throws ProjectManagerException {
        JdbcProjectHandlerSet.IntHandler handler = new JdbcProjectHandlerSet.IntHandler();
        try {
            return (Integer)this.dbOperator.query(JdbcProjectHandlerSet.IntHandler.SELECT_LATEST_VERSION, (ResultSetHandler)handler, new Object[]{project.getId()});
        }
        catch (SQLException e) {
            logger.error("", (Throwable)e);
            throw new ProjectManagerException("Error marking project " + project.getName() + " as inactive", e);
        }
    }

    @Override
    public void uploadFlows(Project project, int version, Collection<Flow> flows) throws ProjectManagerException {
        logger.info("Uploading flows");
        try {
            for (Flow flow : flows) {
                this.uploadFlow(project, version, flow, this.defaultEncodingType);
            }
        }
        catch (IOException e) {
            throw new ProjectManagerException("Flow Upload failed.", e);
        }
    }

    @Override
    public void uploadFlow(Project project, int version, Flow flow) throws ProjectManagerException {
        logger.info("Uploading flow " + flow.getId());
        try {
            this.uploadFlow(project, version, flow, this.defaultEncodingType);
        }
        catch (IOException e) {
            throw new ProjectManagerException("Flow Upload failed.", e);
        }
    }

    @Override
    public void updateFlow(Project project, int version, Flow flow) throws ProjectManagerException {
        logger.info("Uploading flow " + flow.getId());
        try {
            String json = JSONUtils.toJSON(flow.toObject());
            byte[] data = this.convertJsonToBytes(this.defaultEncodingType, json);
            logger.info("Flow upload " + flow.getId() + " is byte size " + data.length);
            String UPDATE_FLOW = "UPDATE project_flows SET encoding_type=?,json=? WHERE project_id=? AND version=? AND flow_id=?";
            try {
                this.dbOperator.update("UPDATE project_flows SET encoding_type=?,json=? WHERE project_id=? AND version=? AND flow_id=?", new Object[]{this.defaultEncodingType.getNumVal(), data, project.getId(), version, flow.getId()});
            }
            catch (SQLException e) {
                logger.error("Error inserting flow", (Throwable)e);
                throw new ProjectManagerException("Error inserting flow " + flow.getId(), e);
            }
        }
        catch (IOException e) {
            throw new ProjectManagerException("Flow Upload failed.", e);
        }
    }

    private void uploadFlow(Project project, int version, Flow flow, EncodingType encType) throws ProjectManagerException, IOException {
        String json = JSONUtils.toJSON(flow.toObject());
        byte[] data = this.convertJsonToBytes(encType, json);
        logger.info("Flow upload " + flow.getId() + " is byte size " + data.length);
        String INSERT_FLOW = "INSERT INTO project_flows (project_id, version, flow_id, modified_time, encoding_type, json) values (?,?,?,?,?,?)";
        try {
            this.dbOperator.update("INSERT INTO project_flows (project_id, version, flow_id, modified_time, encoding_type, json) values (?,?,?,?,?,?)", new Object[]{project.getId(), version, flow.getId(), System.currentTimeMillis(), encType.getNumVal(), data});
        }
        catch (SQLException e) {
            logger.error("Error inserting flow", (Throwable)e);
            throw new ProjectManagerException("Error inserting flow " + flow.getId(), e);
        }
    }

    @Override
    public Flow fetchFlow(Project project, String flowId) throws ProjectManagerException {
        throw new UnsupportedOperationException("this method has not been instantiated.");
    }

    @Override
    public List<Flow> fetchAllProjectFlows(Project project) throws ProjectManagerException {
        JdbcProjectHandlerSet.ProjectFlowsResultHandler handler = new JdbcProjectHandlerSet.ProjectFlowsResultHandler();
        List flows = null;
        try {
            flows = (List)this.dbOperator.query(JdbcProjectHandlerSet.ProjectFlowsResultHandler.SELECT_ALL_PROJECT_FLOWS, (ResultSetHandler)handler, new Object[]{project.getId(), project.getVersion()});
        }
        catch (SQLException e) {
            throw new ProjectManagerException("Error fetching flows from project " + project.getName() + " version " + project.getVersion(), e);
        }
        return flows;
    }

    @Override
    public void uploadProjectProperties(Project project, List<Props> properties) throws ProjectManagerException {
        for (Props props : properties) {
            try {
                this.uploadProjectProperty(project, props.getSource(), props);
            }
            catch (IOException e) {
                throw new ProjectManagerException("Error uploading project property file", e);
            }
        }
    }

    @Override
    public void uploadProjectProperty(Project project, Props props) throws ProjectManagerException {
        try {
            this.uploadProjectProperty(project, props.getSource(), props);
        }
        catch (IOException e) {
            throw new ProjectManagerException("Error uploading project property file", e);
        }
    }

    @Override
    public void updateProjectProperty(Project project, Props props) throws ProjectManagerException {
        try {
            this.updateProjectProperty(project, props.getSource(), props);
        }
        catch (IOException e) {
            throw new ProjectManagerException("Error uploading project property file", e);
        }
    }

    private void updateProjectProperty(Project project, String name, Props props) throws ProjectManagerException, IOException {
        String UPDATE_PROPERTIES = "UPDATE project_properties SET property=? WHERE project_id=? AND version=? AND name=?";
        byte[] propsData = this.getBytes(props);
        try {
            this.dbOperator.update("UPDATE project_properties SET property=? WHERE project_id=? AND version=? AND name=?", new Object[]{propsData, project.getId(), project.getVersion(), name});
        }
        catch (SQLException e) {
            throw new ProjectManagerException("Error updating property " + project.getName() + " version " + project.getVersion(), e);
        }
    }

    private void uploadProjectProperty(Project project, String name, Props props) throws ProjectManagerException, IOException {
        String INSERT_PROPERTIES = "INSERT INTO project_properties (project_id, version, name, modified_time, encoding_type, property) values (?,?,?,?,?,?)";
        byte[] propsData = this.getBytes(props);
        try {
            this.dbOperator.update("INSERT INTO project_properties (project_id, version, name, modified_time, encoding_type, property) values (?,?,?,?,?,?)", new Object[]{project.getId(), project.getVersion(), name, System.currentTimeMillis(), this.defaultEncodingType.getNumVal(), propsData});
        }
        catch (SQLException e) {
            throw new ProjectManagerException("Error uploading project properties " + name + " into " + project.getName() + " version " + project.getVersion(), e);
        }
    }

    private byte[] getBytes(Props props) throws IOException {
        String propertyJSON = PropsUtils.toJSONString((Props)props, (boolean)true);
        byte[] data = propertyJSON.getBytes("UTF-8");
        if (this.defaultEncodingType == EncodingType.GZIP) {
            data = GZIPUtils.gzipBytes(data);
        }
        return data;
    }

    @Override
    public Props fetchProjectProperty(int projectId, int projectVer, String propsName) throws ProjectManagerException {
        JdbcProjectHandlerSet.ProjectPropertiesResultsHandler handler = new JdbcProjectHandlerSet.ProjectPropertiesResultsHandler();
        try {
            List properties = (List)this.dbOperator.query(JdbcProjectHandlerSet.ProjectPropertiesResultsHandler.SELECT_PROJECT_PROPERTY, (ResultSetHandler)handler, new Object[]{projectId, projectVer, propsName});
            if (properties == null || properties.isEmpty()) {
                logger.debug("Project " + projectId + " version " + projectVer + " property " + propsName + " is empty.");
                return null;
            }
            return (Props)((Pair)properties.get(0)).getSecond();
        }
        catch (SQLException e) {
            logger.error("Error fetching property " + propsName + " Project " + projectId + " version " + projectVer, (Throwable)e);
            throw new ProjectManagerException("Error fetching property " + propsName, e);
        }
    }

    @Override
    public Props fetchProjectProperty(Project project, String propsName) throws ProjectManagerException {
        return this.fetchProjectProperty(project.getId(), project.getVersion(), propsName);
    }

    @Override
    public Map<String, Props> fetchProjectProperties(int projectId, int version) throws ProjectManagerException {
        try {
            List properties = (List)this.dbOperator.query(JdbcProjectHandlerSet.ProjectPropertiesResultsHandler.SELECT_PROJECT_PROPERTIES, (ResultSetHandler)new JdbcProjectHandlerSet.ProjectPropertiesResultsHandler(), new Object[]{projectId, version});
            if (properties == null || properties.isEmpty()) {
                return null;
            }
            HashMap<String, Props> props = new HashMap<String, Props>();
            for (Pair pair : properties) {
                props.put((String)pair.getFirst(), (Props)pair.getSecond());
            }
            return props;
        }
        catch (SQLException e) {
            logger.error("Error fetching properties, project id" + projectId + " version " + version, (Throwable)e);
            throw new ProjectManagerException("Error fetching properties", e);
        }
    }

    @Override
    public void cleanOlderProjectVersion(int projectId, int version, List<Integer> excludedVersions) throws ProjectManagerException {
        String EXCLUDED_VERSIONS_FILTER = excludedVersions.stream().map(excluded -> " AND version != " + excluded).collect(Collectors.joining());
        String VERSION_FILTER = " AND version < ?" + EXCLUDED_VERSIONS_FILTER;
        String DELETE_FLOW = "DELETE FROM project_flows WHERE project_id=?" + VERSION_FILTER;
        String DELETE_PROPERTIES = "DELETE FROM project_properties WHERE project_id=?" + VERSION_FILTER;
        String DELETE_PROJECT_FILES = "DELETE FROM project_files WHERE project_id=?" + VERSION_FILTER;
        String UPDATE_PROJECT_VERSIONS = "UPDATE project_versions SET num_chunks=0 WHERE project_id=?" + VERSION_FILTER;
        SQLTransaction cleanOlderProjectTransaction = transOperator -> {
            transOperator.update(DELETE_FLOW, new Object[]{projectId, version});
            transOperator.update(DELETE_PROPERTIES, new Object[]{projectId, version});
            transOperator.update(DELETE_PROJECT_FILES, new Object[]{projectId, version});
            return transOperator.update(UPDATE_PROJECT_VERSIONS, new Object[]{projectId, version});
        };
        try {
            int res = (Integer)this.dbOperator.transaction(cleanOlderProjectTransaction);
            if (res == 0) {
                logger.info("clean older project given project id " + projectId + " doesn't take effect.");
            }
        }
        catch (SQLException e) {
            logger.error("clean older project transaction failed", (Throwable)e);
            throw new ProjectManagerException("clean older project transaction failed", e);
        }
    }

    @Override
    public void uploadFlowFile(int projectId, int projectVersion, File flowFile, int flowVersion) throws ProjectManagerException {
        logger.info(String.format("Uploading flow file %s, version %d for project %d, version %d, file length is [%d bytes]", flowFile.getName(), flowVersion, projectId, projectVersion, flowFile.length()));
        if (flowFile.length() > 0xA00000L) {
            throw new ProjectManagerException("Flow file length exceeds 10 MB limit.");
        }
        byte[] buffer = new byte[0xA00000];
        String INSERT_FLOW_FILES = "INSERT INTO project_flow_files (project_id, project_version, flow_name, flow_version, modified_time, flow_file) values (?,?,?,?,?,?)";
        try (FileInputStream input = new FileInputStream(flowFile);
             BufferedInputStream bufferedStream = new BufferedInputStream(input);){
            int size = bufferedStream.read(buffer);
            logger.info("Read bytes for " + flowFile.getName() + ", size:" + size);
            byte[] buf = Arrays.copyOfRange(buffer, 0, size);
            try {
                this.dbOperator.update("INSERT INTO project_flow_files (project_id, project_version, flow_name, flow_version, modified_time, flow_file) values (?,?,?,?,?,?)", new Object[]{projectId, projectVersion, flowFile.getName(), flowVersion, System.currentTimeMillis(), buf});
            }
            catch (SQLException e) {
                throw new ProjectManagerException("Error uploading flow file " + flowFile.getName() + ", version " + flowVersion + ".", e);
            }
        }
        catch (IOException e) {
            throw new ProjectManagerException(String.format("Error reading flow file %s, version: %d, length: [%d bytes].", flowFile.getName(), flowVersion, flowFile.length()));
        }
    }

    @Override
    public File getUploadedFlowFile(int projectId, int projectVersion, String flowFileName, int flowVersion, File tempDir) throws ProjectManagerException, IOException {
        JdbcProjectHandlerSet.FlowFileResultHandler handler = new JdbcProjectHandlerSet.FlowFileResultHandler();
        File file = new File(tempDir, flowFileName);
        try (FileOutputStream output = new FileOutputStream(file);
             BufferedOutputStream bufferedStream = new BufferedOutputStream(output);){
            List data;
            try {
                data = (List)this.dbOperator.query(JdbcProjectHandlerSet.FlowFileResultHandler.SELECT_FLOW_FILE, (ResultSetHandler)handler, new Object[]{projectId, projectVersion, flowFileName, flowVersion});
            }
            catch (SQLException e) {
                throw new ProjectManagerException("Failed to query uploaded flow file for project " + projectId + " version " + projectVersion + ", flow file " + flowFileName + " version " + flowVersion, e);
            }
            if (data == null || data.isEmpty()) {
                throw new ProjectManagerException("No flow file could be found in DB table for project " + projectId + " version " + projectVersion + ", flow file " + flowFileName + " version " + flowVersion);
            }
            bufferedStream.write((byte[])data.get(0));
        }
        catch (IOException e) {
            throw new ProjectManagerException("Error writing to output stream for project " + projectId + " version " + projectVersion + ", flow file " + flowFileName + " version " + flowVersion, e);
        }
        return file;
    }

    @Override
    public int getLatestFlowVersion(int projectId, int projectVersion, String flowName) throws ProjectManagerException {
        JdbcProjectHandlerSet.IntHandler handler = new JdbcProjectHandlerSet.IntHandler();
        try {
            return (Integer)this.dbOperator.query(JdbcProjectHandlerSet.IntHandler.SELECT_LATEST_FLOW_VERSION, (ResultSetHandler)handler, new Object[]{projectId, projectVersion, flowName});
        }
        catch (SQLException e) {
            logger.error("", (Throwable)e);
            throw new ProjectManagerException("Error selecting latest flow version from project " + projectId + ", version " + projectVersion + ", flow " + flowName + ".", e);
        }
    }

    @Override
    public boolean isFlowFileUploaded(int projectId, int projectVersion) throws ProjectManagerException {
        List data;
        JdbcProjectHandlerSet.FlowFileResultHandler handler = new JdbcProjectHandlerSet.FlowFileResultHandler();
        try {
            data = (List)this.dbOperator.query(JdbcProjectHandlerSet.FlowFileResultHandler.SELECT_ALL_FLOW_FILES, (ResultSetHandler)handler, new Object[]{projectId, projectVersion});
        }
        catch (SQLException e) {
            logger.error("", (Throwable)e);
            throw new ProjectManagerException("Failed to query uploaded flow files ", e);
        }
        return !data.isEmpty();
    }

    @Override
    public void updatePermission(Project project, String name, Permission perm, boolean isGroup, String group) throws ProjectManagerException {
        long updateTime = System.currentTimeMillis();
        try {
            if (this.dbOperator.getDataSource().allowsOnDuplicateKey()) {
                String INSERT_PROJECT_PERMISSION = "INSERT INTO project_permissions (project_id, modified_time, name, permissions, isGroup, project_group) values (?,?,?,?,?,?)ON DUPLICATE KEY UPDATE modified_time = VALUES(modified_time), permissions = VALUES(permissions)";
                this.dbOperator.update("INSERT INTO project_permissions (project_id, modified_time, name, permissions, isGroup, project_group) values (?,?,?,?,?,?)ON DUPLICATE KEY UPDATE modified_time = VALUES(modified_time), permissions = VALUES(permissions)", new Object[]{project.getId(), updateTime, name, perm.toFlags(), isGroup, group});
            } else {
                String MERGE_PROJECT_PERMISSION = "MERGE INTO project_permissions (project_id, modified_time, name, permissions, isGroup, project_group) KEY (project_id, name) values (?,?,?,?,?,?)";
                this.dbOperator.update("MERGE INTO project_permissions (project_id, modified_time, name, permissions, isGroup, project_group) KEY (project_id, name) values (?,?,?,?,?,?)", new Object[]{project.getId(), updateTime, name, perm.toFlags(), isGroup, group});
            }
        }
        catch (SQLException ex) {
            logger.error("Error updating project permission", (Throwable)ex);
            throw new ProjectManagerException("Error updating project " + project.getName() + " permissions for " + name, ex);
        }
        if (isGroup) {
            project.setGroupPermission(group, perm);
            project.setUserPermission(name, perm);
        } else {
            project.setUserPermission(name, perm);
        }
    }

    private List<ProjectPermission> fetchAllPermissionsForProject(Project project) throws ProjectManagerException {
        JdbcProjectHandlerSet.ProjectAllPermissionsResultHandler permHander = new JdbcProjectHandlerSet.ProjectAllPermissionsResultHandler();
        List projectPermissionList = null;
        try {
            projectPermissionList = (List)this.dbOperator.query(JdbcProjectHandlerSet.ProjectPermissionsResultHandler.SELECT_PROJECT_PERMISSION, (ResultSetHandler)permHander, new Object[]{project.getId()});
        }
        catch (SQLException ex) {
            logger.error(JdbcProjectHandlerSet.ProjectPermissionsResultHandler.SELECT_PROJECT_PERMISSION + " failed.", (Throwable)ex);
            throw new ProjectManagerException("Query for permissions for " + project.getName() + " failed.", ex);
        }
        return projectPermissionList;
    }

    private void setProjectAllPermission(Project project, ProjectPermission projectPermission) {
        if (projectPermission.getIsGroup()) {
            project.setGroupPermission(projectPermission.getProjectGroup(), projectPermission.getPermission());
            project.setUserPermission(projectPermission.getUsername(), projectPermission.getPermission());
        } else {
            project.setUserPermission(projectPermission.getUsername(), projectPermission.getPermission());
        }
    }

    @Override
    public void removeProjectPermission(Project project, String userId) throws ProjectManagerException {
        String DELETE_PROJECT_PERMISSION = "DELETE FROM project_permissions WHERE project_id=? AND name=? ";
        try {
            this.dbOperator.update("DELETE FROM project_permissions WHERE project_id=? AND name=? ", new Object[]{project.getId(), userId});
        }
        catch (SQLException e) {
            logger.error("remove Permission failed.", (Throwable)e);
            throw new ProjectManagerException("Error deleting project " + project.getName() + " permissions for " + userId, e);
        }
        project.removeUserPermission(userId);
    }

    @Override
    public List<Project> getTodayCreateProjects(String username) throws ProjectManagerException {
        JdbcProjectHandlerSet.ProjectResultHandler handler = new JdbcProjectHandlerSet.ProjectResultHandler();
        List projects = null;
        ArrayList<Object> params = new ArrayList<Object>();
        try {
            String serchSQL = "";
            if (null != username) {
                serchSQL = "SELECT p.id, p.name, p.active, p.modified_time, p.create_time, p.version, p.last_modified_by, p.description, p.create_user, p.enc_type, p.settings_blob FROM projects p, project_permissions pp WHERE active=true AND p.id = pp.project_id AND pp.name=? ";
                params.add(username);
            } else {
                serchSQL = "SELECT id, name, active, modified_time, create_time, version, last_modified_by, description, create_user, enc_type, settings_blob FROM projects p WHERE active=true ";
            }
            Calendar calendar = Calendar.getInstance();
            calendar.set(11, 0);
            calendar.set(12, 0);
            calendar.set(13, 0);
            calendar.set(14, 1);
            serchSQL = serchSQL + " AND create_time >= ?";
            params.add(calendar.getTimeInMillis());
            calendar.set(11, 23);
            calendar.set(12, 59);
            calendar.set(13, 59);
            serchSQL = serchSQL + " AND create_time <= ?";
            params.add(calendar.getTimeInMillis());
            projects = (List)this.dbOperator.query(serchSQL, (ResultSetHandler)handler, params.toArray());
            projects.forEach(project -> {
                for (ProjectPermission projectPermission : this.fetchAllPermissionsForProject((Project)project)) {
                    this.setProjectAllPermission((Project)project, projectPermission);
                }
            });
        }
        catch (SQLException ex) {
            logger.error("", (Throwable)ex);
            throw new ProjectManagerException("\u67e5\u627e\u5f53\u65e5\u65b0\u5efa\u9879\u76ee\u5217\u8868SQL\u6267\u884c\u5f02\u5e38\uff01", ex);
        }
        return projects;
    }

    @Override
    public int getTodayRunFlow(int projectId, String flowName) throws ProjectManagerException {
        JdbcProjectHandlerSet.IntHandler handler = new JdbcProjectHandlerSet.IntHandler();
        ArrayList<Object> params = new ArrayList<Object>();
        String serchSQL = "SELECT count(*) FROM execution_flows WHERE project_id =? AND flow_id = ?";
        params.add(projectId);
        params.add(flowName);
        Calendar calendar = Calendar.getInstance();
        calendar.set(11, 0);
        calendar.set(12, 0);
        calendar.set(13, 0);
        calendar.set(14, 1);
        serchSQL = serchSQL + " AND submit_time >= ?";
        params.add(calendar.getTimeInMillis());
        calendar.set(11, 23);
        calendar.set(12, 59);
        calendar.set(13, 59);
        serchSQL = serchSQL + " AND submit_time <= ?";
        params.add(calendar.getTimeInMillis());
        try {
            return (Integer)this.dbOperator.query(serchSQL, (ResultSetHandler)handler, params.toArray());
        }
        catch (SQLException e) {
            logger.error("", (Throwable)e);
            throw new ProjectManagerException("Statistics Program " + projectId + " Flow " + flowName + " Exception number of execute SQL in a day ", e);
        }
    }
}

