package com.envisioniot.enos.event_service.vo.condition;

import java.io.Serializable;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * @author hongjie.shan
 * @date 2018/12/14
 */
public abstract class AbstractCondition implements Serializable {

    private static final long serialVersionUID = -1552975627546257175L;

    public abstract ConditionType getConditionType();

    /**
     * get relative fields from condition recursively
     * @return null if is or has no relative condition
     */
    public Set<String> getRelativeFields(){

        if(this instanceof SimpleCondition){
            return null;
        }else if(this instanceof RelativeCondition){
            String relativeField = ((RelativeCondition) this).getAnotherName();
            Set<String> relativeSet = new HashSet<>();
            if(relativeField!=null){
                relativeSet.add(relativeField);
            }
            return relativeSet;
        }else if (this instanceof RangeCondition) {
            return null;
        }else if( this instanceof NotCondition){
            AbstractCondition oppositeCondition = ((NotCondition) this).getCondition();
            return oppositeCondition.getRelativeFields();
        }else if(this instanceof OrCondition || this instanceof AndCondition){
            List<AbstractCondition> conditionList;
            if(this instanceof OrCondition){
                conditionList = ((OrCondition) this).getConditionList();
            }else{
                conditionList = ((AndCondition) this).getConditionList();
            }

            Set<String> relativeSet = new HashSet<>();
            for(AbstractCondition condition: conditionList){
                Set<String> tmpSet = condition.getRelativeFields();
                if(tmpSet!=null && tmpSet.size()>0){
                    relativeSet.addAll(tmpSet);
                }
            }
            return relativeSet;
        }

        return null;
    }

    /**
     * detect point id from condition
     * all the conditions are defined against only one point
     * @return point id. if the condition is invalid, return null.
     */
    public String detectPointId(){
        String pointId = detectRawPointId();
        if(pointId==null){
            return null;
        }else {
            //remove items for struct
            return pointId.replaceAll("/.*","");
        }
    }

    private String detectRawPointId(){
        if(this instanceof SimpleCondition){
            return ((SimpleCondition) this).getName();
        }else if(this instanceof RelativeCondition){
            return ((RelativeCondition) this).getName();
        }else if (this instanceof RangeCondition) {
            return ((RangeCondition) this).getName();
        }else if( this instanceof NotCondition){
            AbstractCondition oppositeCondition = ((NotCondition) this).getCondition();
            if (oppositeCondition != null) {
                return oppositeCondition.detectRawPointId();
            }else {
                return null;
            }
        }else if(this instanceof OrCondition){
            List<AbstractCondition> conditionList = ((OrCondition) this).getConditionList();
            return detectPointIdFromConditionList(conditionList);
        }else if(this instanceof AndCondition){
            List<AbstractCondition> conditionList = ((AndCondition) this).getConditionList();
            return detectPointIdFromConditionList(conditionList);
        }

        return null;
    }

    private String detectPointIdFromConditionList(List<AbstractCondition> conditionList){
        if(conditionList==null || conditionList.size()==0){
            return null;
        }

        for(AbstractCondition condition: conditionList){
            String pointId =  condition.detectRawPointId();
            if(pointId!=null){
                return pointId;
            }
        }

        return null;
    }

}
