/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.groundstation;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.groundstation.model.CancelContactRequest;
import software.amazon.awssdk.services.groundstation.model.CancelContactResponse;
import software.amazon.awssdk.services.groundstation.model.CreateConfigRequest;
import software.amazon.awssdk.services.groundstation.model.CreateConfigResponse;
import software.amazon.awssdk.services.groundstation.model.CreateDataflowEndpointGroupRequest;
import software.amazon.awssdk.services.groundstation.model.CreateDataflowEndpointGroupResponse;
import software.amazon.awssdk.services.groundstation.model.CreateMissionProfileRequest;
import software.amazon.awssdk.services.groundstation.model.CreateMissionProfileResponse;
import software.amazon.awssdk.services.groundstation.model.DeleteConfigRequest;
import software.amazon.awssdk.services.groundstation.model.DeleteConfigResponse;
import software.amazon.awssdk.services.groundstation.model.DeleteDataflowEndpointGroupRequest;
import software.amazon.awssdk.services.groundstation.model.DeleteDataflowEndpointGroupResponse;
import software.amazon.awssdk.services.groundstation.model.DeleteMissionProfileRequest;
import software.amazon.awssdk.services.groundstation.model.DeleteMissionProfileResponse;
import software.amazon.awssdk.services.groundstation.model.DependencyException;
import software.amazon.awssdk.services.groundstation.model.DescribeContactRequest;
import software.amazon.awssdk.services.groundstation.model.DescribeContactResponse;
import software.amazon.awssdk.services.groundstation.model.GetConfigRequest;
import software.amazon.awssdk.services.groundstation.model.GetConfigResponse;
import software.amazon.awssdk.services.groundstation.model.GetDataflowEndpointGroupRequest;
import software.amazon.awssdk.services.groundstation.model.GetDataflowEndpointGroupResponse;
import software.amazon.awssdk.services.groundstation.model.GetMinuteUsageRequest;
import software.amazon.awssdk.services.groundstation.model.GetMinuteUsageResponse;
import software.amazon.awssdk.services.groundstation.model.GetMissionProfileRequest;
import software.amazon.awssdk.services.groundstation.model.GetMissionProfileResponse;
import software.amazon.awssdk.services.groundstation.model.GetSatelliteRequest;
import software.amazon.awssdk.services.groundstation.model.GetSatelliteResponse;
import software.amazon.awssdk.services.groundstation.model.GroundStationException;
import software.amazon.awssdk.services.groundstation.model.GroundStationRequest;
import software.amazon.awssdk.services.groundstation.model.InvalidParameterException;
import software.amazon.awssdk.services.groundstation.model.ListConfigsRequest;
import software.amazon.awssdk.services.groundstation.model.ListConfigsResponse;
import software.amazon.awssdk.services.groundstation.model.ListContactsRequest;
import software.amazon.awssdk.services.groundstation.model.ListContactsResponse;
import software.amazon.awssdk.services.groundstation.model.ListDataflowEndpointGroupsRequest;
import software.amazon.awssdk.services.groundstation.model.ListDataflowEndpointGroupsResponse;
import software.amazon.awssdk.services.groundstation.model.ListGroundStationsRequest;
import software.amazon.awssdk.services.groundstation.model.ListGroundStationsResponse;
import software.amazon.awssdk.services.groundstation.model.ListMissionProfilesRequest;
import software.amazon.awssdk.services.groundstation.model.ListMissionProfilesResponse;
import software.amazon.awssdk.services.groundstation.model.ListSatellitesRequest;
import software.amazon.awssdk.services.groundstation.model.ListSatellitesResponse;
import software.amazon.awssdk.services.groundstation.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.groundstation.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.groundstation.model.ReserveContactRequest;
import software.amazon.awssdk.services.groundstation.model.ReserveContactResponse;
import software.amazon.awssdk.services.groundstation.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.groundstation.model.ResourceNotFoundException;
import software.amazon.awssdk.services.groundstation.model.TagResourceRequest;
import software.amazon.awssdk.services.groundstation.model.TagResourceResponse;
import software.amazon.awssdk.services.groundstation.model.UntagResourceRequest;
import software.amazon.awssdk.services.groundstation.model.UntagResourceResponse;
import software.amazon.awssdk.services.groundstation.model.UpdateConfigRequest;
import software.amazon.awssdk.services.groundstation.model.UpdateConfigResponse;
import software.amazon.awssdk.services.groundstation.model.UpdateMissionProfileRequest;
import software.amazon.awssdk.services.groundstation.model.UpdateMissionProfileResponse;
import software.amazon.awssdk.services.groundstation.paginators.ListConfigsIterable;
import software.amazon.awssdk.services.groundstation.paginators.ListContactsIterable;
import software.amazon.awssdk.services.groundstation.paginators.ListDataflowEndpointGroupsIterable;
import software.amazon.awssdk.services.groundstation.paginators.ListGroundStationsIterable;
import software.amazon.awssdk.services.groundstation.paginators.ListMissionProfilesIterable;
import software.amazon.awssdk.services.groundstation.paginators.ListSatellitesIterable;
import software.amazon.awssdk.services.groundstation.transform.CancelContactRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.CreateConfigRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.CreateDataflowEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.CreateMissionProfileRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DeleteConfigRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DeleteDataflowEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DeleteMissionProfileRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DescribeContactRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetConfigRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetDataflowEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetMinuteUsageRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetMissionProfileRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetSatelliteRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListConfigsRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListContactsRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListDataflowEndpointGroupsRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListGroundStationsRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListMissionProfilesRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListSatellitesRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ReserveContactRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.UpdateConfigRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.UpdateMissionProfileRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link GroundStationClient}.
 *
 * @see GroundStationClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGroundStationClient implements GroundStationClient {
    private static final Logger log = Logger.loggerFor(DefaultGroundStationClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultGroundStationClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels a contact with a specified contact ID.
     * </p>
     *
     * @param cancelContactRequest
     * @return Result of the CancelContact operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.CancelContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CancelContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CancelContactResponse cancelContact(CancelContactRequest cancelContactRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelContact");

            return clientHandler.execute(new ClientExecutionParams<CancelContactRequest, CancelContactResponse>()
                    .withOperationName("CancelContact").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(cancelContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a <code>Config</code> with the specified <code>configData</code> parameters.
     * </p>
     * <p>
     * Only one type of <code>configData</code> can be specified.
     * </p>
     *
     * @param createConfigRequest
     * @return Result of the CreateConfig operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceLimitExceededException
     *         Account limits for this resource have been exceeded.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.CreateConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CreateConfig" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateConfigResponse createConfig(CreateConfigRequest createConfigRequest) throws InvalidParameterException,
            DependencyException, ResourceLimitExceededException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateConfigResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfig");

            return clientHandler.execute(new ClientExecutionParams<CreateConfigRequest, CreateConfigResponse>()
                    .withOperationName("CreateConfig").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createConfigRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a <code>DataflowEndpoint</code> group containing the specified list of <code>DataflowEndpoint</code>
     * objects.
     * </p>
     * <p>
     * The <code>name</code> field in each endpoint is used in your mission profile <code>DataflowEndpointConfig</code>
     * to specify which endpoints to use during a contact.
     * </p>
     * <p>
     * When a contact uses multiple <code>DataflowEndpointConfig</code> objects, each <code>Config</code> must match a
     * <code>DataflowEndpoint</code> in the same group.
     * </p>
     *
     * @param createDataflowEndpointGroupRequest
     * @return Result of the CreateDataflowEndpointGroup operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.CreateDataflowEndpointGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CreateDataflowEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDataflowEndpointGroupResponse createDataflowEndpointGroup(
            CreateDataflowEndpointGroupRequest createDataflowEndpointGroupRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataflowEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDataflowEndpointGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataflowEndpointGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataflowEndpointGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDataflowEndpointGroupRequest, CreateDataflowEndpointGroupResponse>()
                            .withOperationName("CreateDataflowEndpointGroup").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createDataflowEndpointGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDataflowEndpointGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a mission profile.
     * </p>
     * <p>
     * <code>dataflowEdges</code> is a list of lists of strings. Each lower level list of strings has two elements: a
     * <i>from</i> ARN and a <i>to</i> ARN.
     * </p>
     *
     * @param createMissionProfileRequest
     * @return Result of the CreateMissionProfile operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.CreateMissionProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CreateMissionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateMissionProfileResponse createMissionProfile(CreateMissionProfileRequest createMissionProfileRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMissionProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateMissionProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMissionProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMissionProfile");

            return clientHandler.execute(new ClientExecutionParams<CreateMissionProfileRequest, CreateMissionProfileResponse>()
                    .withOperationName("CreateMissionProfile").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createMissionProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateMissionProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a <code>Config</code>.
     * </p>
     *
     * @param deleteConfigRequest
     * @return Result of the DeleteConfig operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.DeleteConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DeleteConfig" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteConfigResponse deleteConfig(DeleteConfigRequest deleteConfigRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteConfigResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfig");

            return clientHandler.execute(new ClientExecutionParams<DeleteConfigRequest, DeleteConfigResponse>()
                    .withOperationName("DeleteConfig").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteConfigRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a dataflow endpoint group.
     * </p>
     *
     * @param deleteDataflowEndpointGroupRequest
     * @return Result of the DeleteDataflowEndpointGroup operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.DeleteDataflowEndpointGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DeleteDataflowEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDataflowEndpointGroupResponse deleteDataflowEndpointGroup(
            DeleteDataflowEndpointGroupRequest deleteDataflowEndpointGroupRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDataflowEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDataflowEndpointGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataflowEndpointGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataflowEndpointGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteDataflowEndpointGroupRequest, DeleteDataflowEndpointGroupResponse>()
                            .withOperationName("DeleteDataflowEndpointGroup").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteDataflowEndpointGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteDataflowEndpointGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a mission profile.
     * </p>
     *
     * @param deleteMissionProfileRequest
     * @return Result of the DeleteMissionProfile operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.DeleteMissionProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DeleteMissionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteMissionProfileResponse deleteMissionProfile(DeleteMissionProfileRequest deleteMissionProfileRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMissionProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteMissionProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMissionProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMissionProfile");

            return clientHandler.execute(new ClientExecutionParams<DeleteMissionProfileRequest, DeleteMissionProfileResponse>()
                    .withOperationName("DeleteMissionProfile").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteMissionProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteMissionProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an existing contact.
     * </p>
     *
     * @param describeContactRequest
     * @return Result of the DescribeContact operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.DescribeContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DescribeContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeContactResponse describeContact(DescribeContactRequest describeContactRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeContact");

            return clientHandler.execute(new ClientExecutionParams<DescribeContactRequest, DescribeContactResponse>()
                    .withOperationName("DescribeContact").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns <code>Config</code> information.
     * </p>
     * <p>
     * Only one <code>Config</code> response can be returned.
     * </p>
     *
     * @param getConfigRequest
     * @return Result of the GetConfig operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.GetConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetConfigResponse getConfig(GetConfigRequest getConfigRequest) throws InvalidParameterException, DependencyException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetConfigResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfig");

            return clientHandler.execute(new ClientExecutionParams<GetConfigRequest, GetConfigResponse>()
                    .withOperationName("GetConfig").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getConfigRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the dataflow endpoint group.
     * </p>
     *
     * @param getDataflowEndpointGroupRequest
     * @return Result of the GetDataflowEndpointGroup operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.GetDataflowEndpointGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetDataflowEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDataflowEndpointGroupResponse getDataflowEndpointGroup(
            GetDataflowEndpointGroupRequest getDataflowEndpointGroupRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDataflowEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDataflowEndpointGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataflowEndpointGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataflowEndpointGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<GetDataflowEndpointGroupRequest, GetDataflowEndpointGroupResponse>()
                            .withOperationName("GetDataflowEndpointGroup").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getDataflowEndpointGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetDataflowEndpointGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the number of minutes used by account.
     * </p>
     *
     * @param getMinuteUsageRequest
     * @return Result of the GetMinuteUsage operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.GetMinuteUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetMinuteUsage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetMinuteUsageResponse getMinuteUsage(GetMinuteUsageRequest getMinuteUsageRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMinuteUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMinuteUsageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMinuteUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMinuteUsage");

            return clientHandler.execute(new ClientExecutionParams<GetMinuteUsageRequest, GetMinuteUsageResponse>()
                    .withOperationName("GetMinuteUsage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getMinuteUsageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMinuteUsageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a mission profile.
     * </p>
     *
     * @param getMissionProfileRequest
     * @return Result of the GetMissionProfile operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.GetMissionProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetMissionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMissionProfileResponse getMissionProfile(GetMissionProfileRequest getMissionProfileRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMissionProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMissionProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMissionProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMissionProfile");

            return clientHandler.execute(new ClientExecutionParams<GetMissionProfileRequest, GetMissionProfileResponse>()
                    .withOperationName("GetMissionProfile").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getMissionProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMissionProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a satellite.
     * </p>
     *
     * @param getSatelliteRequest
     * @return Result of the GetSatellite operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.GetSatellite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetSatellite" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetSatelliteResponse getSatellite(GetSatelliteRequest getSatelliteRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSatelliteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSatelliteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSatelliteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSatellite");

            return clientHandler.execute(new ClientExecutionParams<GetSatelliteRequest, GetSatelliteResponse>()
                    .withOperationName("GetSatellite").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSatelliteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSatelliteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>Config</code> objects.
     * </p>
     *
     * @param listConfigsRequest
     * @return Result of the ListConfigs operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListConfigs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListConfigs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListConfigsResponse listConfigs(ListConfigsRequest listConfigsRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConfigsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListConfigsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfigsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigs");

            return clientHandler.execute(new ClientExecutionParams<ListConfigsRequest, ListConfigsResponse>()
                    .withOperationName("ListConfigs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listConfigsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListConfigsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>Config</code> objects.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listConfigs(software.amazon.awssdk.services.groundstation.model.ListConfigsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.groundstation.paginators.ListConfigsIterable responses = client.listConfigsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.groundstation.paginators.ListConfigsIterable responses = client.listConfigsPaginator(request);
     *     for (software.amazon.awssdk.services.groundstation.model.ListConfigsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.groundstation.paginators.ListConfigsIterable responses = client.listConfigsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listConfigs(software.amazon.awssdk.services.groundstation.model.ListConfigsRequest)} operation.</b>
     * </p>
     *
     * @param listConfigsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListConfigs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListConfigs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListConfigsIterable listConfigsPaginator(ListConfigsRequest listConfigsRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        return new ListConfigsIterable(this, applyPaginatorUserAgent(listConfigsRequest));
    }

    /**
     * <p>
     * Returns a list of contacts.
     * </p>
     * <p>
     * If <code>statusList</code> contains AVAILABLE, the request must include <code>groundStation</code>,
     * <code>missionprofileArn</code>, and <code>satelliteArn</code>.
     * </p>
     *
     * @param listContactsRequest
     * @return Result of the ListContacts operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListContacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListContacts" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListContactsResponse listContacts(ListContactsRequest listContactsRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListContactsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListContactsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContactsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContacts");

            return clientHandler.execute(new ClientExecutionParams<ListContactsRequest, ListContactsResponse>()
                    .withOperationName("ListContacts").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listContactsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListContactsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of contacts.
     * </p>
     * <p>
     * If <code>statusList</code> contains AVAILABLE, the request must include <code>groundStation</code>,
     * <code>missionprofileArn</code>, and <code>satelliteArn</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listContacts(software.amazon.awssdk.services.groundstation.model.ListContactsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.groundstation.paginators.ListContactsIterable responses = client.listContactsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.groundstation.paginators.ListContactsIterable responses = client
     *             .listContactsPaginator(request);
     *     for (software.amazon.awssdk.services.groundstation.model.ListContactsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.groundstation.paginators.ListContactsIterable responses = client.listContactsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listContacts(software.amazon.awssdk.services.groundstation.model.ListContactsRequest)} operation.</b>
     * </p>
     *
     * @param listContactsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListContacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListContacts" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListContactsIterable listContactsPaginator(ListContactsRequest listContactsRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        return new ListContactsIterable(this, applyPaginatorUserAgent(listContactsRequest));
    }

    /**
     * <p>
     * Returns a list of <code>DataflowEndpoint</code> groups.
     * </p>
     *
     * @param listDataflowEndpointGroupsRequest
     * @return Result of the ListDataflowEndpointGroups operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListDataflowEndpointGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListDataflowEndpointGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDataflowEndpointGroupsResponse listDataflowEndpointGroups(
            ListDataflowEndpointGroupsRequest listDataflowEndpointGroupsRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDataflowEndpointGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDataflowEndpointGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataflowEndpointGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataflowEndpointGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<ListDataflowEndpointGroupsRequest, ListDataflowEndpointGroupsResponse>()
                            .withOperationName("ListDataflowEndpointGroups").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listDataflowEndpointGroupsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListDataflowEndpointGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>DataflowEndpoint</code> groups.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDataflowEndpointGroups(software.amazon.awssdk.services.groundstation.model.ListDataflowEndpointGroupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.groundstation.paginators.ListDataflowEndpointGroupsIterable responses = client.listDataflowEndpointGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.groundstation.paginators.ListDataflowEndpointGroupsIterable responses = client
     *             .listDataflowEndpointGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.groundstation.model.ListDataflowEndpointGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.groundstation.paginators.ListDataflowEndpointGroupsIterable responses = client.listDataflowEndpointGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDataflowEndpointGroups(software.amazon.awssdk.services.groundstation.model.ListDataflowEndpointGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDataflowEndpointGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListDataflowEndpointGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListDataflowEndpointGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDataflowEndpointGroupsIterable listDataflowEndpointGroupsPaginator(
            ListDataflowEndpointGroupsRequest listDataflowEndpointGroupsRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        return new ListDataflowEndpointGroupsIterable(this, applyPaginatorUserAgent(listDataflowEndpointGroupsRequest));
    }

    /**
     * <p>
     * Returns a list of ground stations.
     * </p>
     *
     * @param listGroundStationsRequest
     * @return Result of the ListGroundStations operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListGroundStations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListGroundStations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListGroundStationsResponse listGroundStations(ListGroundStationsRequest listGroundStationsRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGroundStationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListGroundStationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroundStationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroundStations");

            return clientHandler.execute(new ClientExecutionParams<ListGroundStationsRequest, ListGroundStationsResponse>()
                    .withOperationName("ListGroundStations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listGroundStationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListGroundStationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of ground stations.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listGroundStations(software.amazon.awssdk.services.groundstation.model.ListGroundStationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.groundstation.paginators.ListGroundStationsIterable responses = client.listGroundStationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.groundstation.paginators.ListGroundStationsIterable responses = client
     *             .listGroundStationsPaginator(request);
     *     for (software.amazon.awssdk.services.groundstation.model.ListGroundStationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.groundstation.paginators.ListGroundStationsIterable responses = client.listGroundStationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroundStations(software.amazon.awssdk.services.groundstation.model.ListGroundStationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listGroundStationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListGroundStations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListGroundStations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListGroundStationsIterable listGroundStationsPaginator(ListGroundStationsRequest listGroundStationsRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        return new ListGroundStationsIterable(this, applyPaginatorUserAgent(listGroundStationsRequest));
    }

    /**
     * <p>
     * Returns a list of mission profiles.
     * </p>
     *
     * @param listMissionProfilesRequest
     * @return Result of the ListMissionProfiles operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListMissionProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListMissionProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMissionProfilesResponse listMissionProfiles(ListMissionProfilesRequest listMissionProfilesRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMissionProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListMissionProfilesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMissionProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMissionProfiles");

            return clientHandler.execute(new ClientExecutionParams<ListMissionProfilesRequest, ListMissionProfilesResponse>()
                    .withOperationName("ListMissionProfiles").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listMissionProfilesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMissionProfilesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of mission profiles.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMissionProfiles(software.amazon.awssdk.services.groundstation.model.ListMissionProfilesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.groundstation.paginators.ListMissionProfilesIterable responses = client.listMissionProfilesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.groundstation.paginators.ListMissionProfilesIterable responses = client
     *             .listMissionProfilesPaginator(request);
     *     for (software.amazon.awssdk.services.groundstation.model.ListMissionProfilesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.groundstation.paginators.ListMissionProfilesIterable responses = client.listMissionProfilesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMissionProfiles(software.amazon.awssdk.services.groundstation.model.ListMissionProfilesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listMissionProfilesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListMissionProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListMissionProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMissionProfilesIterable listMissionProfilesPaginator(ListMissionProfilesRequest listMissionProfilesRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        return new ListMissionProfilesIterable(this, applyPaginatorUserAgent(listMissionProfilesRequest));
    }

    /**
     * <p>
     * Returns a list of satellites.
     * </p>
     *
     * @param listSatellitesRequest
     * @return Result of the ListSatellites operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListSatellites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListSatellites" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSatellitesResponse listSatellites(ListSatellitesRequest listSatellitesRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSatellitesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSatellitesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSatellitesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSatellites");

            return clientHandler.execute(new ClientExecutionParams<ListSatellitesRequest, ListSatellitesResponse>()
                    .withOperationName("ListSatellites").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSatellitesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSatellitesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of satellites.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSatellites(software.amazon.awssdk.services.groundstation.model.ListSatellitesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.groundstation.paginators.ListSatellitesIterable responses = client.listSatellitesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.groundstation.paginators.ListSatellitesIterable responses = client
     *             .listSatellitesPaginator(request);
     *     for (software.amazon.awssdk.services.groundstation.model.ListSatellitesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.groundstation.paginators.ListSatellitesIterable responses = client.listSatellitesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSatellites(software.amazon.awssdk.services.groundstation.model.ListSatellitesRequest)} operation.</b>
     * </p>
     *
     * @param listSatellitesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListSatellites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListSatellites" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSatellitesIterable listSatellitesPaginator(ListSatellitesRequest listSatellitesRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        return new ListSatellitesIterable(this, applyPaginatorUserAgent(listSatellitesRequest));
    }

    /**
     * <p>
     * Returns a list of tags for a specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Reserves a contact using specified parameters.
     * </p>
     *
     * @param reserveContactRequest
     * @return Result of the ReserveContact operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ReserveContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ReserveContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ReserveContactResponse reserveContact(ReserveContactRequest reserveContactRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ReserveContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ReserveContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, reserveContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReserveContact");

            return clientHandler.execute(new ClientExecutionParams<ReserveContactRequest, ReserveContactResponse>()
                    .withOperationName("ReserveContact").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(reserveContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ReserveContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns a tag to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deassigns a resource tag.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the <code>Config</code> used when scheduling contacts.
     * </p>
     * <p>
     * Updating a <code>Config</code> will not update the execution parameters for existing future contacts scheduled
     * with this <code>Config</code>.
     * </p>
     *
     * @param updateConfigRequest
     * @return Result of the UpdateConfig operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.UpdateConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/UpdateConfig" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateConfigResponse updateConfig(UpdateConfigRequest updateConfigRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateConfigResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfig");

            return clientHandler.execute(new ClientExecutionParams<UpdateConfigRequest, UpdateConfigResponse>()
                    .withOperationName("UpdateConfig").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateConfigRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a mission profile.
     * </p>
     * <p>
     * Updating a mission profile will not update the execution parameters for existing future contacts.
     * </p>
     *
     * @param updateMissionProfileRequest
     * @return Result of the UpdateMissionProfile operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.UpdateMissionProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/UpdateMissionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateMissionProfileResponse updateMissionProfile(UpdateMissionProfileRequest updateMissionProfileRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateMissionProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateMissionProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMissionProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMissionProfile");

            return clientHandler.execute(new ClientExecutionParams<UpdateMissionProfileRequest, UpdateMissionProfileResponse>()
                    .withOperationName("UpdateMissionProfile").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateMissionProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateMissionProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GroundStationException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(431).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(434).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyException")
                                .exceptionBuilderSupplier(DependencyException::builder).httpStatusCode(531).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).httpStatusCode(429).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends GroundStationRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
