/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains summary information about a batch build group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BuildSummary implements SdkPojo, Serializable, ToCopyableBuilder<BuildSummary.Builder, BuildSummary> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(BuildSummary::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Instant> REQUESTED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("requestedOn").getter(getter(BuildSummary::requestedOn)).setter(setter(Builder::requestedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("requestedOn").build()).build();

    private static final SdkField<String> BUILD_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("buildStatus").getter(getter(BuildSummary::buildStatusAsString)).setter(setter(Builder::buildStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildStatus").build()).build();

    private static final SdkField<ResolvedArtifact> PRIMARY_ARTIFACT_FIELD = SdkField
            .<ResolvedArtifact> builder(MarshallingType.SDK_POJO).memberName("primaryArtifact")
            .getter(getter(BuildSummary::primaryArtifact)).setter(setter(Builder::primaryArtifact))
            .constructor(ResolvedArtifact::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("primaryArtifact").build()).build();

    private static final SdkField<List<ResolvedArtifact>> SECONDARY_ARTIFACTS_FIELD = SdkField
            .<List<ResolvedArtifact>> builder(MarshallingType.LIST)
            .memberName("secondaryArtifacts")
            .getter(getter(BuildSummary::secondaryArtifacts))
            .setter(setter(Builder::secondaryArtifacts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secondaryArtifacts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResolvedArtifact> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResolvedArtifact::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, REQUESTED_ON_FIELD,
            BUILD_STATUS_FIELD, PRIMARY_ARTIFACT_FIELD, SECONDARY_ARTIFACTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final Instant requestedOn;

    private final String buildStatus;

    private final ResolvedArtifact primaryArtifact;

    private final List<ResolvedArtifact> secondaryArtifacts;

    private BuildSummary(BuilderImpl builder) {
        this.arn = builder.arn;
        this.requestedOn = builder.requestedOn;
        this.buildStatus = builder.buildStatus;
        this.primaryArtifact = builder.primaryArtifact;
        this.secondaryArtifacts = builder.secondaryArtifacts;
    }

    /**
     * <p>
     * The batch build ARN.
     * </p>
     * 
     * @return The batch build ARN.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * When the build was started, expressed in Unix time format.
     * </p>
     * 
     * @return When the build was started, expressed in Unix time format.
     */
    public final Instant requestedOn() {
        return requestedOn;
    }

    /**
     * <p>
     * The status of the build group.
     * </p>
     * <dl>
     * <dt>FAILED</dt>
     * <dd>
     * <p>
     * The build group failed.
     * </p>
     * </dd>
     * <dt>FAULT</dt>
     * <dd>
     * <p>
     * The build group faulted.
     * </p>
     * </dd>
     * <dt>IN_PROGRESS</dt>
     * <dd>
     * <p>
     * The build group is still in progress.
     * </p>
     * </dd>
     * <dt>STOPPED</dt>
     * <dd>
     * <p>
     * The build group stopped.
     * </p>
     * </dd>
     * <dt>SUCCEEDED</dt>
     * <dd>
     * <p>
     * The build group succeeded.
     * </p>
     * </dd>
     * <dt>TIMED_OUT</dt>
     * <dd>
     * <p>
     * The build group timed out.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #buildStatus} will
     * return {@link StatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #buildStatusAsString}.
     * </p>
     * 
     * @return The status of the build group.</p>
     *         <dl>
     *         <dt>FAILED</dt>
     *         <dd>
     *         <p>
     *         The build group failed.
     *         </p>
     *         </dd>
     *         <dt>FAULT</dt>
     *         <dd>
     *         <p>
     *         The build group faulted.
     *         </p>
     *         </dd>
     *         <dt>IN_PROGRESS</dt>
     *         <dd>
     *         <p>
     *         The build group is still in progress.
     *         </p>
     *         </dd>
     *         <dt>STOPPED</dt>
     *         <dd>
     *         <p>
     *         The build group stopped.
     *         </p>
     *         </dd>
     *         <dt>SUCCEEDED</dt>
     *         <dd>
     *         <p>
     *         The build group succeeded.
     *         </p>
     *         </dd>
     *         <dt>TIMED_OUT</dt>
     *         <dd>
     *         <p>
     *         The build group timed out.
     *         </p>
     *         </dd>
     * @see StatusType
     */
    public final StatusType buildStatus() {
        return StatusType.fromValue(buildStatus);
    }

    /**
     * <p>
     * The status of the build group.
     * </p>
     * <dl>
     * <dt>FAILED</dt>
     * <dd>
     * <p>
     * The build group failed.
     * </p>
     * </dd>
     * <dt>FAULT</dt>
     * <dd>
     * <p>
     * The build group faulted.
     * </p>
     * </dd>
     * <dt>IN_PROGRESS</dt>
     * <dd>
     * <p>
     * The build group is still in progress.
     * </p>
     * </dd>
     * <dt>STOPPED</dt>
     * <dd>
     * <p>
     * The build group stopped.
     * </p>
     * </dd>
     * <dt>SUCCEEDED</dt>
     * <dd>
     * <p>
     * The build group succeeded.
     * </p>
     * </dd>
     * <dt>TIMED_OUT</dt>
     * <dd>
     * <p>
     * The build group timed out.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #buildStatus} will
     * return {@link StatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #buildStatusAsString}.
     * </p>
     * 
     * @return The status of the build group.</p>
     *         <dl>
     *         <dt>FAILED</dt>
     *         <dd>
     *         <p>
     *         The build group failed.
     *         </p>
     *         </dd>
     *         <dt>FAULT</dt>
     *         <dd>
     *         <p>
     *         The build group faulted.
     *         </p>
     *         </dd>
     *         <dt>IN_PROGRESS</dt>
     *         <dd>
     *         <p>
     *         The build group is still in progress.
     *         </p>
     *         </dd>
     *         <dt>STOPPED</dt>
     *         <dd>
     *         <p>
     *         The build group stopped.
     *         </p>
     *         </dd>
     *         <dt>SUCCEEDED</dt>
     *         <dd>
     *         <p>
     *         The build group succeeded.
     *         </p>
     *         </dd>
     *         <dt>TIMED_OUT</dt>
     *         <dd>
     *         <p>
     *         The build group timed out.
     *         </p>
     *         </dd>
     * @see StatusType
     */
    public final String buildStatusAsString() {
        return buildStatus;
    }

    /**
     * <p>
     * A <code>ResolvedArtifact</code> object that represents the primary build artifacts for the build group.
     * </p>
     * 
     * @return A <code>ResolvedArtifact</code> object that represents the primary build artifacts for the build group.
     */
    public final ResolvedArtifact primaryArtifact() {
        return primaryArtifact;
    }

    /**
     * Returns true if the SecondaryArtifacts property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasSecondaryArtifacts() {
        return secondaryArtifacts != null && !(secondaryArtifacts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>ResolvedArtifact</code> objects that represents the secondary build artifacts for the build
     * group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSecondaryArtifacts()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of <code>ResolvedArtifact</code> objects that represents the secondary build artifacts for the
     *         build group.
     */
    public final List<ResolvedArtifact> secondaryArtifacts() {
        return secondaryArtifacts;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(requestedOn());
        hashCode = 31 * hashCode + Objects.hashCode(buildStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(primaryArtifact());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecondaryArtifacts() ? secondaryArtifacts() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BuildSummary)) {
            return false;
        }
        BuildSummary other = (BuildSummary) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(requestedOn(), other.requestedOn())
                && Objects.equals(buildStatusAsString(), other.buildStatusAsString())
                && Objects.equals(primaryArtifact(), other.primaryArtifact())
                && hasSecondaryArtifacts() == other.hasSecondaryArtifacts()
                && Objects.equals(secondaryArtifacts(), other.secondaryArtifacts());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BuildSummary").add("Arn", arn()).add("RequestedOn", requestedOn())
                .add("BuildStatus", buildStatusAsString()).add("PrimaryArtifact", primaryArtifact())
                .add("SecondaryArtifacts", hasSecondaryArtifacts() ? secondaryArtifacts() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "requestedOn":
            return Optional.ofNullable(clazz.cast(requestedOn()));
        case "buildStatus":
            return Optional.ofNullable(clazz.cast(buildStatusAsString()));
        case "primaryArtifact":
            return Optional.ofNullable(clazz.cast(primaryArtifact()));
        case "secondaryArtifacts":
            return Optional.ofNullable(clazz.cast(secondaryArtifacts()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BuildSummary, T> g) {
        return obj -> g.apply((BuildSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BuildSummary> {
        /**
         * <p>
         * The batch build ARN.
         * </p>
         * 
         * @param arn
         *        The batch build ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * When the build was started, expressed in Unix time format.
         * </p>
         * 
         * @param requestedOn
         *        When the build was started, expressed in Unix time format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedOn(Instant requestedOn);

        /**
         * <p>
         * The status of the build group.
         * </p>
         * <dl>
         * <dt>FAILED</dt>
         * <dd>
         * <p>
         * The build group failed.
         * </p>
         * </dd>
         * <dt>FAULT</dt>
         * <dd>
         * <p>
         * The build group faulted.
         * </p>
         * </dd>
         * <dt>IN_PROGRESS</dt>
         * <dd>
         * <p>
         * The build group is still in progress.
         * </p>
         * </dd>
         * <dt>STOPPED</dt>
         * <dd>
         * <p>
         * The build group stopped.
         * </p>
         * </dd>
         * <dt>SUCCEEDED</dt>
         * <dd>
         * <p>
         * The build group succeeded.
         * </p>
         * </dd>
         * <dt>TIMED_OUT</dt>
         * <dd>
         * <p>
         * The build group timed out.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param buildStatus
         *        The status of the build group.</p>
         *        <dl>
         *        <dt>FAILED</dt>
         *        <dd>
         *        <p>
         *        The build group failed.
         *        </p>
         *        </dd>
         *        <dt>FAULT</dt>
         *        <dd>
         *        <p>
         *        The build group faulted.
         *        </p>
         *        </dd>
         *        <dt>IN_PROGRESS</dt>
         *        <dd>
         *        <p>
         *        The build group is still in progress.
         *        </p>
         *        </dd>
         *        <dt>STOPPED</dt>
         *        <dd>
         *        <p>
         *        The build group stopped.
         *        </p>
         *        </dd>
         *        <dt>SUCCEEDED</dt>
         *        <dd>
         *        <p>
         *        The build group succeeded.
         *        </p>
         *        </dd>
         *        <dt>TIMED_OUT</dt>
         *        <dd>
         *        <p>
         *        The build group timed out.
         *        </p>
         *        </dd>
         * @see StatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatusType
         */
        Builder buildStatus(String buildStatus);

        /**
         * <p>
         * The status of the build group.
         * </p>
         * <dl>
         * <dt>FAILED</dt>
         * <dd>
         * <p>
         * The build group failed.
         * </p>
         * </dd>
         * <dt>FAULT</dt>
         * <dd>
         * <p>
         * The build group faulted.
         * </p>
         * </dd>
         * <dt>IN_PROGRESS</dt>
         * <dd>
         * <p>
         * The build group is still in progress.
         * </p>
         * </dd>
         * <dt>STOPPED</dt>
         * <dd>
         * <p>
         * The build group stopped.
         * </p>
         * </dd>
         * <dt>SUCCEEDED</dt>
         * <dd>
         * <p>
         * The build group succeeded.
         * </p>
         * </dd>
         * <dt>TIMED_OUT</dt>
         * <dd>
         * <p>
         * The build group timed out.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param buildStatus
         *        The status of the build group.</p>
         *        <dl>
         *        <dt>FAILED</dt>
         *        <dd>
         *        <p>
         *        The build group failed.
         *        </p>
         *        </dd>
         *        <dt>FAULT</dt>
         *        <dd>
         *        <p>
         *        The build group faulted.
         *        </p>
         *        </dd>
         *        <dt>IN_PROGRESS</dt>
         *        <dd>
         *        <p>
         *        The build group is still in progress.
         *        </p>
         *        </dd>
         *        <dt>STOPPED</dt>
         *        <dd>
         *        <p>
         *        The build group stopped.
         *        </p>
         *        </dd>
         *        <dt>SUCCEEDED</dt>
         *        <dd>
         *        <p>
         *        The build group succeeded.
         *        </p>
         *        </dd>
         *        <dt>TIMED_OUT</dt>
         *        <dd>
         *        <p>
         *        The build group timed out.
         *        </p>
         *        </dd>
         * @see StatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatusType
         */
        Builder buildStatus(StatusType buildStatus);

        /**
         * <p>
         * A <code>ResolvedArtifact</code> object that represents the primary build artifacts for the build group.
         * </p>
         * 
         * @param primaryArtifact
         *        A <code>ResolvedArtifact</code> object that represents the primary build artifacts for the build
         *        group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder primaryArtifact(ResolvedArtifact primaryArtifact);

        /**
         * <p>
         * A <code>ResolvedArtifact</code> object that represents the primary build artifacts for the build group.
         * </p>
         * This is a convenience that creates an instance of the {@link ResolvedArtifact.Builder} avoiding the need to
         * create one manually via {@link ResolvedArtifact#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResolvedArtifact.Builder#build()} is called immediately and its
         * result is passed to {@link #primaryArtifact(ResolvedArtifact)}.
         * 
         * @param primaryArtifact
         *        a consumer that will call methods on {@link ResolvedArtifact.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #primaryArtifact(ResolvedArtifact)
         */
        default Builder primaryArtifact(Consumer<ResolvedArtifact.Builder> primaryArtifact) {
            return primaryArtifact(ResolvedArtifact.builder().applyMutation(primaryArtifact).build());
        }

        /**
         * <p>
         * An array of <code>ResolvedArtifact</code> objects that represents the secondary build artifacts for the build
         * group.
         * </p>
         * 
         * @param secondaryArtifacts
         *        An array of <code>ResolvedArtifact</code> objects that represents the secondary build artifacts for
         *        the build group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondaryArtifacts(Collection<ResolvedArtifact> secondaryArtifacts);

        /**
         * <p>
         * An array of <code>ResolvedArtifact</code> objects that represents the secondary build artifacts for the build
         * group.
         * </p>
         * 
         * @param secondaryArtifacts
         *        An array of <code>ResolvedArtifact</code> objects that represents the secondary build artifacts for
         *        the build group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondaryArtifacts(ResolvedArtifact... secondaryArtifacts);

        /**
         * <p>
         * An array of <code>ResolvedArtifact</code> objects that represents the secondary build artifacts for the build
         * group.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ResolvedArtifact>.Builder} avoiding the
         * need to create one manually via {@link List<ResolvedArtifact>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ResolvedArtifact>.Builder#build()} is called immediately and
         * its result is passed to {@link #secondaryArtifacts(List<ResolvedArtifact>)}.
         * 
         * @param secondaryArtifacts
         *        a consumer that will call methods on {@link List<ResolvedArtifact>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secondaryArtifacts(List<ResolvedArtifact>)
         */
        Builder secondaryArtifacts(Consumer<ResolvedArtifact.Builder>... secondaryArtifacts);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private Instant requestedOn;

        private String buildStatus;

        private ResolvedArtifact primaryArtifact;

        private List<ResolvedArtifact> secondaryArtifacts = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BuildSummary model) {
            arn(model.arn);
            requestedOn(model.requestedOn);
            buildStatus(model.buildStatus);
            primaryArtifact(model.primaryArtifact);
            secondaryArtifacts(model.secondaryArtifacts);
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final Instant getRequestedOn() {
            return requestedOn;
        }

        @Override
        public final Builder requestedOn(Instant requestedOn) {
            this.requestedOn = requestedOn;
            return this;
        }

        public final void setRequestedOn(Instant requestedOn) {
            this.requestedOn = requestedOn;
        }

        public final String getBuildStatus() {
            return buildStatus;
        }

        @Override
        public final Builder buildStatus(String buildStatus) {
            this.buildStatus = buildStatus;
            return this;
        }

        @Override
        public final Builder buildStatus(StatusType buildStatus) {
            this.buildStatus(buildStatus == null ? null : buildStatus.toString());
            return this;
        }

        public final void setBuildStatus(String buildStatus) {
            this.buildStatus = buildStatus;
        }

        public final ResolvedArtifact.Builder getPrimaryArtifact() {
            return primaryArtifact != null ? primaryArtifact.toBuilder() : null;
        }

        @Override
        public final Builder primaryArtifact(ResolvedArtifact primaryArtifact) {
            this.primaryArtifact = primaryArtifact;
            return this;
        }

        public final void setPrimaryArtifact(ResolvedArtifact.BuilderImpl primaryArtifact) {
            this.primaryArtifact = primaryArtifact != null ? primaryArtifact.build() : null;
        }

        public final List<ResolvedArtifact.Builder> getSecondaryArtifacts() {
            List<ResolvedArtifact.Builder> result = ResolvedSecondaryArtifactsCopier.copyToBuilder(this.secondaryArtifacts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder secondaryArtifacts(Collection<ResolvedArtifact> secondaryArtifacts) {
            this.secondaryArtifacts = ResolvedSecondaryArtifactsCopier.copy(secondaryArtifacts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secondaryArtifacts(ResolvedArtifact... secondaryArtifacts) {
            secondaryArtifacts(Arrays.asList(secondaryArtifacts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secondaryArtifacts(Consumer<ResolvedArtifact.Builder>... secondaryArtifacts) {
            secondaryArtifacts(Stream.of(secondaryArtifacts).map(c -> ResolvedArtifact.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSecondaryArtifacts(Collection<ResolvedArtifact.BuilderImpl> secondaryArtifacts) {
            this.secondaryArtifacts = ResolvedSecondaryArtifactsCopier.copyFromBuilder(secondaryArtifacts);
        }

        @Override
        public BuildSummary build() {
            return new BuildSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
