/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.redis.connection;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.core.task.AsyncTaskExecutor;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.ClusterRedirectException;
import org.springframework.data.redis.ClusterStateFailureException;
import org.springframework.data.redis.ExceptionTranslationStrategy;
import org.springframework.data.redis.TooManyClusterRedirectionsException;
import org.springframework.data.redis.connection.ClusterCommandExecutionFailureException;
import org.springframework.data.redis.connection.ClusterNodeResourceProvider;
import org.springframework.data.redis.connection.ClusterTopology;
import org.springframework.data.redis.connection.ClusterTopologyProvider;
import org.springframework.data.redis.connection.RedisClusterNode;
import org.springframework.data.redis.connection.util.ByteArraySet;
import org.springframework.data.redis.connection.util.ByteArrayWrapper;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

public class ClusterCommandExecutor
implements DisposableBean {
    private AsyncTaskExecutor executor;
    private final ClusterTopologyProvider topologyProvider;
    private final ClusterNodeResourceProvider resourceProvider;
    private final ExceptionTranslationStrategy exceptionTranslationStrategy;
    private int maxRedirects = 5;

    public ClusterCommandExecutor(ClusterTopologyProvider topologyProvider, ClusterNodeResourceProvider resourceProvider, ExceptionTranslationStrategy exceptionTranslation) {
        if (this.executor == null) {
            ThreadPoolTaskExecutor threadPoolTaskExecutor = new ThreadPoolTaskExecutor();
            threadPoolTaskExecutor.initialize();
            this.executor = threadPoolTaskExecutor;
        }
        Assert.notNull((Object)topologyProvider, (String)"ClusterTopologyProvider must not be null!");
        Assert.notNull((Object)resourceProvider, (String)"ClusterNodeResourceProvider must not be null!");
        Assert.notNull((Object)exceptionTranslation, (String)"ExceptionTranslationStrategy must not be null!");
        this.topologyProvider = topologyProvider;
        this.resourceProvider = resourceProvider;
        this.exceptionTranslationStrategy = exceptionTranslation;
    }

    public ClusterCommandExecutor(ClusterTopologyProvider topologyProvider, ClusterNodeResourceProvider resourceProvider, ExceptionTranslationStrategy exceptionTranslation, AsyncTaskExecutor executor) {
        this(topologyProvider, resourceProvider, exceptionTranslation);
        this.executor = executor;
    }

    public <T> NodeResult<T> executeCommandOnArbitraryNode(ClusterCommandCallback<?, T> cmd) {
        Assert.notNull(cmd, (String)"ClusterCommandCallback must not be null!");
        ArrayList<RedisClusterNode> nodes = new ArrayList<RedisClusterNode>(this.getClusterTopology().getActiveNodes());
        return this.executeCommandOnSingleNode(cmd, (RedisClusterNode)nodes.get(new Random().nextInt(nodes.size())));
    }

    public <S, T> NodeResult<T> executeCommandOnSingleNode(ClusterCommandCallback<S, T> cmd, RedisClusterNode node) {
        return this.executeCommandOnSingleNode(cmd, node, 0);
    }

    private <S, T> NodeResult<T> executeCommandOnSingleNode(ClusterCommandCallback<S, T> cmd, RedisClusterNode node, int redirectCount) {
        Assert.notNull(cmd, (String)"ClusterCommandCallback must not be null!");
        Assert.notNull((Object)node, (String)"RedisClusterNode must not be null!");
        if (redirectCount > this.maxRedirects) {
            throw new TooManyClusterRedirectionsException(String.format("Cannot follow Cluster Redirects over more than %s legs. Please consider increasing the number of redirects to follow. Current value is: %s.", redirectCount, this.maxRedirects));
        }
        RedisClusterNode nodeToUse = this.lookupNode(node);
        Object client = this.resourceProvider.getResourceForSpecificNode(nodeToUse);
        Assert.notNull(client, (String)"Could not acquire resource for node. Is your cluster info up to date?");
        try {
            NodeResult<T> nodeResult = new NodeResult<T>(node, cmd.doInCluster(client));
            return nodeResult;
        }
        catch (RuntimeException ex) {
            DataAccessException translatedException = this.convertToDataAccessExeption(ex);
            if (translatedException instanceof ClusterRedirectException) {
                ClusterRedirectException cre = (ClusterRedirectException)((Object)translatedException);
                NodeResult<T> nodeResult = this.executeCommandOnSingleNode(cmd, this.topologyProvider.getTopology().lookup(cre.getTargetHost(), cre.getTargetPort()), redirectCount + 1);
                return nodeResult;
            }
            throw translatedException != null ? translatedException : ex;
        }
        finally {
            this.resourceProvider.returnResourceForSpecificNode(nodeToUse, client);
        }
    }

    private RedisClusterNode lookupNode(RedisClusterNode node) {
        try {
            return this.topologyProvider.getTopology().lookup(node);
        }
        catch (ClusterStateFailureException e) {
            throw new IllegalArgumentException(String.format("Node %s is unknown to cluster", node), (Throwable)((Object)e));
        }
    }

    public <S, T> MulitNodeResult<T> executeCommandOnAllNodes(ClusterCommandCallback<S, T> cmd) {
        return this.executeCommandAsyncOnNodes(cmd, this.getClusterTopology().getActiveMasterNodes());
    }

    public <S, T> MulitNodeResult<T> executeCommandAsyncOnNodes(final ClusterCommandCallback<S, T> callback, Iterable<RedisClusterNode> nodes) {
        Assert.notNull(callback, (String)"Callback must not be null!");
        Assert.notNull(nodes, (String)"Nodes must not be null!");
        ArrayList<RedisClusterNode> resolvedRedisClusterNodes = new ArrayList<RedisClusterNode>();
        ClusterTopology topology = this.topologyProvider.getTopology();
        for (RedisClusterNode node : nodes) {
            try {
                resolvedRedisClusterNodes.add(topology.lookup(node));
            }
            catch (ClusterStateFailureException e) {
                throw new IllegalArgumentException(String.format("Node %s is unknown to cluster", node), (Throwable)((Object)e));
            }
        }
        LinkedHashMap<NodeExecution, Future<NodeResult<T>>> futures = new LinkedHashMap<NodeExecution, Future<NodeResult<T>>>();
        for (final RedisClusterNode node : resolvedRedisClusterNodes) {
            futures.put(new NodeExecution(node, new Object[0]), this.executor.submit(new Callable<NodeResult<T>>(){

                @Override
                public NodeResult<T> call() throws Exception {
                    return ClusterCommandExecutor.this.executeCommandOnSingleNode(callback, node);
                }
            }));
        }
        return this.collectResults(futures);
    }

    private <T> MulitNodeResult<T> collectResults(Map<NodeExecution, Future<NodeResult<T>>> futures) {
        boolean done = false;
        MulitNodeResult result = new MulitNodeResult();
        HashMap<RedisClusterNode, Throwable> exceptions = new HashMap<RedisClusterNode, Throwable>();
        HashSet<String> saveGuard = new HashSet<String>();
        while (!done) {
            done = true;
            for (Map.Entry<NodeExecution, Future<NodeResult<T>>> entry : futures.entrySet()) {
                DataAccessException ex;
                if (!entry.getValue().isDone() && !entry.getValue().isCancelled()) {
                    done = false;
                    continue;
                }
                try {
                    String futureId = ObjectUtils.getIdentityHexString(entry.getValue());
                    if (saveGuard.contains(futureId)) continue;
                    result.add(entry.getValue().get());
                    saveGuard.add(futureId);
                }
                catch (ExecutionException e) {
                    ex = this.convertToDataAccessExeption((Exception)e.getCause());
                    exceptions.put(entry.getKey().getNode(), ex != null ? ex : e.getCause());
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    ex = this.convertToDataAccessExeption((Exception)e.getCause());
                    exceptions.put(entry.getKey().getNode(), ex != null ? ex : e.getCause());
                    break;
                }
            }
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException e) {
                done = true;
                Thread.currentThread().interrupt();
            }
        }
        if (!exceptions.isEmpty()) {
            throw new ClusterCommandExecutionFailureException(new ArrayList(exceptions.values()));
        }
        return result;
    }

    public <S, T> MulitNodeResult<T> executeMuliKeyCommand(final MultiKeyClusterCommandCallback<S, T> cmd, Iterable<byte[]> keys) {
        HashMap nodeKeyMap = new HashMap();
        for (byte[] key : keys) {
            for (RedisClusterNode node : this.getClusterTopology().getKeyServingNodes(key)) {
                if (nodeKeyMap.containsKey(node)) {
                    ((Set)nodeKeyMap.get(node)).add(key);
                    continue;
                }
                LinkedHashSet<byte[]> keySet = new LinkedHashSet<byte[]>();
                keySet.add(key);
                nodeKeyMap.put(node, keySet);
            }
        }
        LinkedHashMap<NodeExecution, Future<NodeResult<T>>> futures = new LinkedHashMap<NodeExecution, Future<NodeResult<T>>>();
        for (final Map.Entry entry : nodeKeyMap.entrySet()) {
            if (!((RedisClusterNode)entry.getKey()).isMaster()) continue;
            for (final byte[] key : (Set)entry.getValue()) {
                futures.put(new NodeExecution((RedisClusterNode)entry.getKey(), new Object[]{key}), this.executor.submit(new Callable<NodeResult<T>>(){

                    @Override
                    public NodeResult<T> call() throws Exception {
                        return ClusterCommandExecutor.this.executeMultiKeyCommandOnSingleNode(cmd, (RedisClusterNode)entry.getKey(), key);
                    }
                }));
            }
        }
        return this.collectResults(futures);
    }

    private <S, T> NodeResult<T> executeMultiKeyCommandOnSingleNode(MultiKeyClusterCommandCallback<S, T> cmd, RedisClusterNode node, byte[] key) {
        Assert.notNull(cmd, (String)"MultiKeyCommandCallback must not be null!");
        Assert.notNull((Object)node, (String)"RedisClusterNode must not be null!");
        Assert.notNull((Object)key, (String)"Keys for execution must not be null!");
        Object client = this.resourceProvider.getResourceForSpecificNode(node);
        Assert.notNull(client, (String)"Could not acquire resource for node. Is your cluster info up to date?");
        try {
            NodeResult<T> nodeResult = new NodeResult<T>(node, cmd.doInCluster(client, key), key);
            return nodeResult;
        }
        catch (RuntimeException ex) {
            DataAccessException translatedException = this.convertToDataAccessExeption(ex);
            throw translatedException != null ? translatedException : ex;
        }
        finally {
            this.resourceProvider.returnResourceForSpecificNode(node, client);
        }
    }

    private ClusterTopology getClusterTopology() {
        return this.topologyProvider.getTopology();
    }

    private DataAccessException convertToDataAccessExeption(Exception e) {
        return this.exceptionTranslationStrategy.translate(e);
    }

    public void setMaxRedirects(int maxRedirects) {
        this.maxRedirects = maxRedirects;
    }

    public void destroy() throws Exception {
        if (this.executor instanceof DisposableBean) {
            ((DisposableBean)this.executor).destroy();
        }
        if (this.resourceProvider instanceof DisposableBean) {
            ((DisposableBean)this.resourceProvider).destroy();
        }
    }

    public static class MulitNodeResult<T> {
        List<NodeResult<T>> nodeResults = new ArrayList<NodeResult<T>>();

        private void add(NodeResult<T> result) {
            this.nodeResults.add(result);
        }

        public List<NodeResult<T>> getResults() {
            return Collections.unmodifiableList(this.nodeResults);
        }

        public List<T> resultsAsList() {
            return this.toList(this.nodeResults);
        }

        public List<T> resultsAsListSortBy(byte[] ... keys) {
            ArrayList<NodeResult<T>> clone = new ArrayList<NodeResult<T>>(this.nodeResults);
            Collections.sort(clone, new ResultByReferenceKeyPositionComparator(keys));
            return this.toList(clone);
        }

        public T getFirstNonNullNotEmptyOrDefault(T returnValue) {
            for (NodeResult<T> nodeResult : this.nodeResults) {
                if (nodeResult.getValue() == null) continue;
                if (nodeResult.getValue() instanceof Map) {
                    if (!CollectionUtils.isEmpty((Map)((Map)nodeResult.getValue()))) continue;
                    return nodeResult.getValue();
                }
                if (CollectionUtils.isEmpty((Collection)((Collection)nodeResult.getValue()))) {
                    return nodeResult.getValue();
                }
                return nodeResult.getValue();
            }
            return returnValue;
        }

        private List<T> toList(Collection<NodeResult<T>> source) {
            ArrayList<T> result = new ArrayList<T>();
            for (NodeResult<T> nodeResult : source) {
                result.add(nodeResult.getValue());
            }
            return result;
        }

        private static class ResultByReferenceKeyPositionComparator
        implements Comparator<NodeResult<?>> {
            List<ByteArrayWrapper> reference;

            public ResultByReferenceKeyPositionComparator(byte[] ... keys) {
                this.reference = new ArrayList<ByteArrayWrapper>(new ByteArraySet(Arrays.asList(keys)));
            }

            @Override
            public int compare(NodeResult<?> o1, NodeResult<?> o2) {
                return Integer.valueOf(this.reference.indexOf(((NodeResult)o1).key)).compareTo(this.reference.indexOf(((NodeResult)o2).key));
            }
        }
    }

    public static class NodeResult<T> {
        private RedisClusterNode node;
        private T value;
        private ByteArrayWrapper key;

        public NodeResult(RedisClusterNode node, T value) {
            this(node, value, new byte[0]);
        }

        public NodeResult(RedisClusterNode node, T value, byte[] key) {
            this.node = node;
            this.value = value;
            this.key = new ByteArrayWrapper(key);
        }

        public T getValue() {
            return this.value;
        }

        public RedisClusterNode getNode() {
            return this.node;
        }

        public byte[] getKey() {
            return this.key.getArray();
        }
    }

    private static class NodeExecution {
        private RedisClusterNode node;
        private Object[] args;

        public NodeExecution(RedisClusterNode node, Object ... args) {
            this.node = node;
            this.args = args;
        }

        public RedisClusterNode getNode() {
            return this.node;
        }

        public int hashCode() {
            int result = ObjectUtils.nullSafeHashCode((Object)this.node);
            return result + ObjectUtils.nullSafeHashCode((Object[])this.args);
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (!(obj instanceof NodeExecution)) {
                return false;
            }
            NodeExecution that = (NodeExecution)obj;
            if (!ObjectUtils.nullSafeEquals((Object)this.node, (Object)that.node)) {
                return false;
            }
            return ObjectUtils.nullSafeEquals((Object)this.args, (Object)that.args);
        }
    }

    public static interface MultiKeyClusterCommandCallback<T, S> {
        public S doInCluster(T var1, byte[] var2);
    }

    public static interface ClusterCommandCallback<T, S> {
        public S doInCluster(T var1);
    }
}

