/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.r2dbc.connectionfactory;

import io.r2dbc.spi.Connection;
import io.r2dbc.spi.ConnectionFactory;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.reactivestreams.Publisher;
import org.springframework.core.Ordered;
import org.springframework.dao.DataAccessResourceFailureException;
import org.springframework.data.r2dbc.connectionfactory.ConnectionHolder;
import org.springframework.data.r2dbc.connectionfactory.ConnectionProxy;
import org.springframework.data.r2dbc.connectionfactory.DelegatingConnectionFactory;
import org.springframework.data.r2dbc.connectionfactory.SmartConnectionFactory;
import org.springframework.lang.Nullable;
import org.springframework.transaction.NoTransactionException;
import org.springframework.transaction.reactive.TransactionSynchronization;
import org.springframework.transaction.reactive.TransactionSynchronizationManager;
import org.springframework.util.Assert;
import reactor.core.publisher.Mono;

public abstract class ConnectionFactoryUtils {
    public static final int CONNECTION_SYNCHRONIZATION_ORDER = 1000;
    private static final Log logger = LogFactory.getLog(ConnectionFactoryUtils.class);

    private ConnectionFactoryUtils() {
    }

    public static Mono<Connection> getConnection(ConnectionFactory connectionFactory) {
        return ConnectionFactoryUtils.doGetConnection(connectionFactory).onErrorMap(e -> new DataAccessResourceFailureException("Failed to obtain R2DBC Connection", e));
    }

    public static Mono<Connection> doGetConnection(ConnectionFactory connectionFactory) {
        Assert.notNull((Object)connectionFactory, (String)"ConnectionFactory must not be null!");
        return TransactionSynchronizationManager.forCurrentTransaction().flatMap(synchronizationManager -> {
            ConnectionHolder conHolder = (ConnectionHolder)((Object)((Object)synchronizationManager.getResource((Object)connectionFactory)));
            if (conHolder != null && (conHolder.hasConnection() || conHolder.isSynchronizedWithTransaction())) {
                conHolder.requested();
                if (!conHolder.hasConnection()) {
                    if (logger.isDebugEnabled()) {
                        logger.debug((Object)"Fetching resumed R2DBC Connection from ConnectionFactory");
                    }
                    return ConnectionFactoryUtils.fetchConnection(connectionFactory).doOnNext(conHolder::setConnection);
                }
                return Mono.just((Object)conHolder.getConnection());
            }
            if (logger.isDebugEnabled()) {
                logger.debug((Object)"Fetching R2DBC Connection from ConnectionFactory");
            }
            Mono<Connection> con = ConnectionFactoryUtils.fetchConnection(connectionFactory);
            if (synchronizationManager.isSynchronizationActive()) {
                return con.flatMap(it -> Mono.just((Object)it).doOnNext(conn -> {
                    ConnectionHolder holderToUse = conHolder;
                    if (holderToUse == null) {
                        holderToUse = new ConnectionHolder((Connection)conn);
                    } else {
                        holderToUse.setConnection((Connection)conn);
                    }
                    holderToUse.requested();
                    synchronizationManager.registerSynchronization((TransactionSynchronization)new ConnectionSynchronization(holderToUse, connectionFactory));
                    holderToUse.setSynchronizedWithTransaction(true);
                    if (holderToUse != conHolder) {
                        synchronizationManager.bindResource((Object)connectionFactory, (Object)holderToUse);
                    }
                }).onErrorResume(e -> ConnectionFactoryUtils.releaseConnection(it, connectionFactory).then(Mono.error((Throwable)e))));
            }
            return con;
        }).onErrorResume(NoTransactionException.class, e -> Mono.from((Publisher)connectionFactory.create()));
    }

    private static Mono<Connection> fetchConnection(ConnectionFactory connectionFactory) {
        return Mono.from((Publisher)connectionFactory.create());
    }

    public static Mono<Void> releaseConnection(Connection con, ConnectionFactory connectionFactory) {
        return ConnectionFactoryUtils.doReleaseConnection(con, connectionFactory).onErrorMap(e -> new DataAccessResourceFailureException("Failed to close R2DBC Connection", e));
    }

    public static Mono<Void> doReleaseConnection(Connection connection, ConnectionFactory connectionFactory) {
        return TransactionSynchronizationManager.forCurrentTransaction().flatMap(it -> {
            ConnectionHolder conHolder = (ConnectionHolder)((Object)((Object)it.getResource((Object)connectionFactory)));
            if (conHolder != null && ConnectionFactoryUtils.connectionEquals(conHolder, connection)) {
                conHolder.released();
            }
            return Mono.from((Publisher)connection.close());
        }).onErrorResume(NoTransactionException.class, e -> ConnectionFactoryUtils.doCloseConnection(connection, connectionFactory));
    }

    public static Mono<Void> closeConnection(Connection connection, ConnectionFactory connectionFactory) {
        Assert.notNull((Object)connection, (String)"Connection must not be null!");
        Assert.notNull((Object)connectionFactory, (String)"ConnectionFactory must not be null!");
        return ConnectionFactoryUtils.doCloseConnection(connection, connectionFactory).onErrorMap(e -> new DataAccessResourceFailureException("Failed to obtain R2DBC Connection", e));
    }

    public static Mono<Void> doCloseConnection(Connection connection, @Nullable ConnectionFactory connectionFactory) {
        if (!(connectionFactory instanceof SmartConnectionFactory) || ((SmartConnectionFactory)connectionFactory).shouldClose(connection)) {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)"Closing R2DBC Connection");
            }
            return Mono.from((Publisher)connection.close());
        }
        return Mono.empty();
    }

    public static Mono<ConnectionFactory> currentConnectionFactory(ConnectionFactory connectionFactory) {
        return TransactionSynchronizationManager.forCurrentTransaction().filter(TransactionSynchronizationManager::isSynchronizationActive).filter(it -> {
            ConnectionHolder conHolder = (ConnectionHolder)((Object)((Object)it.getResource((Object)connectionFactory)));
            return conHolder != null && (conHolder.hasConnection() || conHolder.isSynchronizedWithTransaction());
        }).map(it -> connectionFactory);
    }

    private static boolean connectionEquals(ConnectionHolder conHolder, Connection passedInCon) {
        if (!conHolder.hasConnection()) {
            return false;
        }
        Connection heldCon = conHolder.getConnection();
        return heldCon == passedInCon || heldCon.equals(passedInCon) || ConnectionFactoryUtils.getTargetConnection(heldCon).equals(passedInCon);
    }

    public static Connection getTargetConnection(Connection con) {
        Connection conToUse = con;
        while (conToUse instanceof ConnectionProxy) {
            conToUse = ((ConnectionProxy)conToUse).getTargetConnection();
        }
        return conToUse;
    }

    private static int getConnectionSynchronizationOrder(ConnectionFactory connectionFactory) {
        int order = 1000;
        ConnectionFactory current = connectionFactory;
        while (current instanceof DelegatingConnectionFactory) {
            --order;
            current = ((DelegatingConnectionFactory)current).getTargetConnectionFactory();
        }
        return order;
    }

    private static class ConnectionSynchronization
    implements TransactionSynchronization,
    Ordered {
        private final ConnectionHolder connectionHolder;
        private final ConnectionFactory connectionFactory;
        private int order;
        private boolean holderActive = true;

        ConnectionSynchronization(ConnectionHolder connectionHolder, ConnectionFactory connectionFactory) {
            this.connectionHolder = connectionHolder;
            this.connectionFactory = connectionFactory;
            this.order = ConnectionFactoryUtils.getConnectionSynchronizationOrder(connectionFactory);
        }

        public int getOrder() {
            return this.order;
        }

        public Mono<Void> suspend() {
            if (this.holderActive) {
                return TransactionSynchronizationManager.forCurrentTransaction().flatMap(it -> {
                    it.unbindResource((Object)this.connectionFactory);
                    if (this.connectionHolder.hasConnection() && !this.connectionHolder.isOpen()) {
                        return ConnectionFactoryUtils.releaseConnection(this.connectionHolder.getConnection(), this.connectionFactory).doOnTerminate(() -> this.connectionHolder.setConnection(null));
                    }
                    return Mono.empty();
                });
            }
            return Mono.empty();
        }

        public Mono<Void> resume() {
            if (this.holderActive) {
                return TransactionSynchronizationManager.forCurrentTransaction().doOnNext(it -> it.bindResource((Object)this.connectionFactory, (Object)this.connectionHolder)).then();
            }
            return Mono.empty();
        }

        public Mono<Void> beforeCompletion() {
            if (!this.connectionHolder.isOpen()) {
                return TransactionSynchronizationManager.forCurrentTransaction().flatMap(it -> {
                    it.unbindResource((Object)this.connectionFactory);
                    this.holderActive = false;
                    if (this.connectionHolder.hasConnection()) {
                        return ConnectionFactoryUtils.releaseConnection(this.connectionHolder.getConnection(), this.connectionFactory);
                    }
                    return Mono.empty();
                });
            }
            return Mono.empty();
        }

        public Mono<Void> afterCompletion(int status) {
            if (this.holderActive) {
                return TransactionSynchronizationManager.forCurrentTransaction().flatMap(it -> {
                    it.unbindResourceIfPossible((Object)this.connectionFactory);
                    this.holderActive = false;
                    if (this.connectionHolder.hasConnection()) {
                        return ConnectionFactoryUtils.releaseConnection(this.connectionHolder.getConnection(), this.connectionFactory).doOnTerminate(() -> this.connectionHolder.setConnection(null));
                    }
                    return Mono.empty();
                });
            }
            this.connectionHolder.reset();
            return Mono.empty();
        }
    }
}

