/*
 * Decompiled with CFR 0.152.
 */
package hudson.plugins.ec2;

import com.amazonaws.AmazonClientException;
import hudson.init.InitMilestone;
import hudson.model.Descriptor;
import hudson.model.Executor;
import hudson.model.ExecutorListener;
import hudson.model.Label;
import hudson.model.Queue;
import hudson.model.labels.LabelAtom;
import hudson.plugins.ec2.EC2AbstractSlave;
import hudson.plugins.ec2.EC2Computer;
import hudson.plugins.ec2.InstanceState;
import hudson.plugins.ec2.SlaveTemplate;
import hudson.plugins.ec2.util.MinimumInstanceChecker;
import hudson.slaves.RetentionStrategy;
import java.time.Clock;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import jenkins.model.Jenkins;
import org.kohsuke.stapler.DataBoundConstructor;

public class EC2RetentionStrategy
extends RetentionStrategy<EC2Computer>
implements ExecutorListener {
    private static final Logger LOGGER = Logger.getLogger(EC2RetentionStrategy.class.getName());
    public static final boolean DISABLED = Boolean.getBoolean(EC2RetentionStrategy.class.getName() + ".disabled");
    private long nextCheckAfter = -1L;
    private transient Clock clock;
    public final int idleTerminationMinutes;
    private transient ReentrantLock checkLock;
    private static final int STARTUP_TIME_DEFAULT_VALUE = 30;
    private static final Integer CHECK_INTERVAL_MINUTES = Integer.getInteger("jenkins.ec2.checkIntervalMinutes", 1);

    @DataBoundConstructor
    public EC2RetentionStrategy(String idleTerminationMinutes) {
        this.readResolve();
        if (idleTerminationMinutes == null || idleTerminationMinutes.trim().isEmpty()) {
            this.idleTerminationMinutes = 0;
        } else {
            int value = 30;
            try {
                value = Integer.parseInt(idleTerminationMinutes);
            }
            catch (NumberFormatException nfe) {
                LOGGER.info("Malformed default idleTermination value: " + idleTerminationMinutes);
            }
            this.idleTerminationMinutes = value;
        }
    }

    EC2RetentionStrategy(String idleTerminationMinutes, Clock clock, long nextCheckAfter) {
        this(idleTerminationMinutes);
        this.clock = clock;
        this.nextCheckAfter = nextCheckAfter;
    }

    long getNextCheckAfter() {
        return this.nextCheckAfter;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long check(EC2Computer c) {
        if (!this.checkLock.tryLock()) {
            return CHECK_INTERVAL_MINUTES.intValue();
        }
        try {
            long currentTime = this.clock.millis();
            if (currentTime > this.nextCheckAfter) {
                long intervalMins = this.internalCheck(c);
                this.nextCheckAfter = currentTime + TimeUnit.MINUTES.toMillis(intervalMins);
                long l = intervalMins;
                return l;
            }
            long l = CHECK_INTERVAL_MINUTES.intValue();
            return l;
        }
        finally {
            this.checkLock.unlock();
        }
    }

    private long internalCheck(EC2Computer computer) {
        long numberOfCurrentInstancesForTemplate;
        if (this.idleTerminationMinutes == 0 || computer.getNode() == null) {
            return CHECK_INTERVAL_MINUTES.intValue();
        }
        SlaveTemplate slaveTemplate = computer.getSlaveTemplate();
        if (slaveTemplate != null && (numberOfCurrentInstancesForTemplate = (long)MinimumInstanceChecker.countCurrentNumberOfAgents(slaveTemplate)) > 0L && numberOfCurrentInstancesForTemplate <= (long)slaveTemplate.getMinimumNumberOfInstances() && MinimumInstanceChecker.minimumInstancesActive(slaveTemplate.getMinimumNumberOfInstancesTimeRangeConfig())) {
            return CHECK_INTERVAL_MINUTES.intValue();
        }
        if (computer.isIdle() && !DISABLED) {
            long launchedAtMs;
            long uptime;
            InstanceState state;
            try {
                state = computer.getState();
                uptime = computer.getUptime();
                launchedAtMs = computer.getLaunchTime();
            }
            catch (AmazonClientException | InterruptedException e) {
                LOGGER.fine("Exception while checking host uptime for " + computer.getName() + ", will retry next check. Exception: " + (Exception)e);
                return CHECK_INTERVAL_MINUTES.intValue();
            }
            if (InstanceState.TERMINATED.equals((Object)state) || slaveTemplate != null && slaveTemplate.stopOnTerminate && (InstanceState.STOPPED.equals((Object)state) || InstanceState.STOPPING.equals((Object)state))) {
                if (computer.isOnline()) {
                    LOGGER.info("External Stop of " + computer.getName() + " detected - disconnecting. instance status" + state.toString());
                    computer.disconnect(null);
                }
                return CHECK_INTERVAL_MINUTES.intValue();
            }
            if (computer.isOffline()) {
                if (computer.isConnecting()) {
                    LOGGER.log(Level.FINE, "Computer {0} connecting and still offline, will check if the launch timeout has expired", computer.getInstanceId());
                    EC2AbstractSlave node = computer.getNode();
                    if (Objects.isNull((Object)node)) {
                        return CHECK_INTERVAL_MINUTES.intValue();
                    }
                    long launchTimeout = node.getLaunchTimeoutInMillis();
                    if (launchTimeout > 0L && uptime > launchTimeout) {
                        LOGGER.info("Startup timeout of " + computer.getName() + " after " + uptime + " milliseconds (timeout: " + launchTimeout + " milliseconds), instance status: " + state.toString());
                        node.launchTimeout();
                    }
                    return CHECK_INTERVAL_MINUTES.intValue();
                }
                LOGGER.log(Level.FINE, "Computer {0} offline but not connecting, will check if it should be terminated because of the idle time configured", computer.getInstanceId());
            }
            long idleMilliseconds = this.clock.millis() - Math.max(computer.getIdleStartMilliseconds(), launchedAtMs);
            if (this.idleTerminationMinutes > 0) {
                if (idleMilliseconds > TimeUnit.MINUTES.toMillis(this.idleTerminationMinutes) && !this.itemsInQueueForThisSlave(computer)) {
                    LOGGER.info("Idle timeout of " + computer.getName() + " after " + TimeUnit.MILLISECONDS.toMinutes(idleMilliseconds) + " idle minutes, instance status" + state.toString());
                    EC2AbstractSlave slaveNode = computer.getNode();
                    if (slaveNode != null) {
                        slaveNode.idleTimeout();
                    }
                }
            } else {
                int oneHourSeconds = (int)TimeUnit.SECONDS.convert(1L, TimeUnit.HOURS);
                int freeSecondsLeft = oneHourSeconds - (int)(TimeUnit.SECONDS.convert(uptime, TimeUnit.MILLISECONDS) % (long)oneHourSeconds);
                if ((long)freeSecondsLeft <= TimeUnit.MINUTES.toSeconds(Math.abs(this.idleTerminationMinutes)) && !this.itemsInQueueForThisSlave(computer)) {
                    LOGGER.info("Idle timeout of " + computer.getName() + " after " + TimeUnit.MILLISECONDS.toMinutes(idleMilliseconds) + " idle minutes, with " + TimeUnit.SECONDS.toMinutes(freeSecondsLeft) + " minutes remaining in billing period");
                    EC2AbstractSlave slaveNode = computer.getNode();
                    if (slaveNode != null) {
                        slaveNode.idleTimeout();
                    }
                }
            }
        }
        return CHECK_INTERVAL_MINUTES.intValue();
    }

    private boolean itemsInQueueForThisSlave(EC2Computer c) {
        EC2AbstractSlave selfNode = c.getNode();
        if (selfNode == null) {
            return false;
        }
        LabelAtom selfLabel = selfNode.getSelfLabel();
        Queue.Item[] items = Jenkins.getInstance().getQueue().getItems();
        for (int i = 0; i < items.length; ++i) {
            Queue.Item item = items[i];
            Label assignedLabel = item.getAssignedLabel();
            if (assignedLabel != selfLabel) continue;
            LOGGER.fine("Preventing idle timeout of " + c.getName() + " as there is at least one item in the queue explicitly waiting for this slave");
            return true;
        }
        return false;
    }

    public void start(EC2Computer c) {
        if (Jenkins.get().getInitLevel() != InitMilestone.COMPLETED) {
            InstanceState state = null;
            try {
                state = c.getState();
            }
            catch (AmazonClientException | InterruptedException e) {
                LOGGER.log(Level.FINE, "Error getting EC2 instance state for " + c.getName(), e);
            }
            if (!InstanceState.PENDING.equals((Object)state) && !InstanceState.RUNNING.equals((Object)state)) {
                LOGGER.info("Ignoring start request for " + c.getName() + " during Jenkins startup due to EC2 instance state of " + state);
                return;
            }
        }
        LOGGER.info("Start requested for " + c.getName());
        c.connect(false);
    }

    protected Object readResolve() {
        this.checkLock = new ReentrantLock(false);
        this.clock = Clock.systemUTC();
        return this;
    }

    public void taskAccepted(Executor executor, Queue.Task task) {
        EC2AbstractSlave slaveNode;
        EC2Computer computer = (EC2Computer)executor.getOwner();
        if (computer != null && (slaveNode = computer.getNode()) != null) {
            int maxTotalUses;
            if ((maxTotalUses = slaveNode.maxTotalUses--) <= -1) {
                LOGGER.fine("maxTotalUses set to unlimited (" + slaveNode.maxTotalUses + ") for agent " + slaveNode.instanceId);
                return;
            }
            if (maxTotalUses <= 1) {
                LOGGER.info("maxTotalUses drained - suspending agent " + slaveNode.instanceId);
                computer.setAcceptingTasks(false);
            } else {
                LOGGER.info("Agent " + slaveNode.instanceId + " has " + slaveNode.maxTotalUses + " builds left");
            }
        }
    }

    public void taskCompleted(Executor executor, Queue.Task task, long durationMS) {
        this.postJobAction(executor);
    }

    public void taskCompletedWithProblems(Executor executor, Queue.Task task, long durationMS, Throwable problems) {
        this.postJobAction(executor);
    }

    private void postJobAction(Executor executor) {
        EC2AbstractSlave slaveNode;
        EC2Computer computer = (EC2Computer)executor.getOwner();
        if (computer != null && (slaveNode = computer.getNode()) != null) {
            if (computer.countBusy() <= 1 && !computer.isAcceptingTasks()) {
                LOGGER.info("Agent " + slaveNode.instanceId + " is terminated due to maxTotalUses (" + slaveNode.maxTotalUses + ")");
                slaveNode.terminate();
            } else if (slaveNode.maxTotalUses == 1) {
                LOGGER.info("Agent " + slaveNode.instanceId + " is still in use by more than one (" + computer.countBusy() + ") executers.");
            }
        }
    }

    public static class DescriptorImpl
    extends Descriptor<RetentionStrategy<?>> {
        public String getDisplayName() {
            return "EC2";
        }
    }
}

