/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.wallet;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.collect.PeekingIterator;
import com.google.protobuf.ByteString;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.Nullable;
import org.bitcoinj.core.BloomFilter;
import org.bitcoinj.core.ECKey;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.Utils;
import org.bitcoinj.crypto.ChildNumber;
import org.bitcoinj.crypto.DeterministicHierarchy;
import org.bitcoinj.crypto.DeterministicKey;
import org.bitcoinj.crypto.EncryptedData;
import org.bitcoinj.crypto.HDKeyDerivation;
import org.bitcoinj.crypto.HDUtils;
import org.bitcoinj.crypto.KeyCrypter;
import org.bitcoinj.crypto.KeyCrypterException;
import org.bitcoinj.crypto.KeyCrypterScrypt;
import org.bitcoinj.crypto.LazyECPoint;
import org.bitcoinj.script.Script;
import org.bitcoinj.utils.ListenerRegistration;
import org.bitcoinj.utils.Threading;
import org.bitcoinj.wallet.BasicKeyChain;
import org.bitcoinj.wallet.DefaultKeyChainFactory;
import org.bitcoinj.wallet.DeterministicSeed;
import org.bitcoinj.wallet.EncryptableKeyChain;
import org.bitcoinj.wallet.KeyChain;
import org.bitcoinj.wallet.KeyChainFactory;
import org.bitcoinj.wallet.Protos;
import org.bitcoinj.wallet.RedeemData;
import org.bitcoinj.wallet.UnreadableWalletException;
import org.bitcoinj.wallet.listeners.KeyChainEventListener;
import org.bouncycastle.crypto.params.KeyParameter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DeterministicKeyChain
implements EncryptableKeyChain {
    private static final Logger log = LoggerFactory.getLogger(DeterministicKeyChain.class);
    public static final String DEFAULT_PASSPHRASE_FOR_MNEMONIC = "";
    protected final ReentrantLock lock = Threading.lock("DeterministicKeyChain");
    private DeterministicHierarchy hierarchy;
    @Nullable
    private DeterministicKey rootKey;
    @Nullable
    private DeterministicSeed seed;
    private final Script.ScriptType outputScriptType;
    private final ImmutableList<ChildNumber> accountPath;
    public static final ImmutableList<ChildNumber> ACCOUNT_ZERO_PATH = ImmutableList.of((Object)ChildNumber.ZERO_HARDENED);
    public static final ImmutableList<ChildNumber> ACCOUNT_ONE_PATH = ImmutableList.of((Object)ChildNumber.ONE_HARDENED);
    public static final ImmutableList<ChildNumber> BIP44_ACCOUNT_ZERO_PATH = ImmutableList.of((Object)new ChildNumber(44, true), (Object)ChildNumber.ZERO_HARDENED, (Object)ChildNumber.ZERO_HARDENED);
    public static final ImmutableList<ChildNumber> EXTERNAL_SUBPATH = ImmutableList.of((Object)ChildNumber.ZERO);
    public static final ImmutableList<ChildNumber> INTERNAL_SUBPATH = ImmutableList.of((Object)ChildNumber.ONE);
    private static final int LAZY_CALCULATE_LOOKAHEAD = -1;
    protected int lookaheadSize = 100;
    protected int lookaheadThreshold = this.calcDefaultLookaheadThreshold();
    private DeterministicKey externalParentKey;
    private DeterministicKey internalParentKey;
    private int issuedExternalKeys;
    private int issuedInternalKeys;
    private int keyLookaheadEpoch;
    private final BasicKeyChain basicKeyChain;
    private boolean isFollowing;
    protected int sigsRequiredToSpend = 1;

    private int calcDefaultLookaheadThreshold() {
        return this.lookaheadSize / 3;
    }

    public static Builder<?> builder() {
        return new Builder();
    }

    public DeterministicKeyChain(DeterministicKey key, boolean isFollowing, boolean isWatching, Script.ScriptType outputScriptType) {
        if (isWatching) {
            Preconditions.checkArgument((boolean)key.isPubKeyOnly(), (Object)"Private subtrees not currently supported for watching keys: if you got this key from DKC.getWatchingKey() then use .dropPrivate().dropParent() on it first.");
        } else {
            Preconditions.checkArgument((boolean)key.hasPrivKey(), (Object)"Private subtrees are required.");
        }
        Preconditions.checkArgument((isWatching || !isFollowing ? 1 : 0) != 0, (Object)"Can only follow a key that is watched");
        this.basicKeyChain = new BasicKeyChain();
        this.seed = null;
        this.rootKey = null;
        this.basicKeyChain.importKey(key);
        this.hierarchy = new DeterministicHierarchy(key);
        this.accountPath = key.getPath();
        this.outputScriptType = outputScriptType;
        this.initializeHierarchyUnencrypted(key);
        this.isFollowing = isFollowing;
    }

    protected DeterministicKeyChain(DeterministicSeed seed, @Nullable KeyCrypter crypter, Script.ScriptType outputScriptType, ImmutableList<ChildNumber> accountPath) {
        Preconditions.checkArgument((outputScriptType == null || outputScriptType == Script.ScriptType.P2PKH || outputScriptType == Script.ScriptType.P2WPKH ? 1 : 0) != 0, (Object)"Only P2PKH or P2WPKH allowed.");
        this.outputScriptType = outputScriptType != null ? outputScriptType : Script.ScriptType.P2PKH;
        this.accountPath = accountPath;
        this.seed = seed;
        this.basicKeyChain = new BasicKeyChain(crypter);
        if (!seed.isEncrypted()) {
            this.rootKey = HDKeyDerivation.createMasterPrivateKey((byte[])Preconditions.checkNotNull((Object)seed.getSeedBytes()));
            this.rootKey.setCreationTimeSeconds(seed.getCreationTimeSeconds());
            this.basicKeyChain.importKey(this.rootKey);
            this.hierarchy = new DeterministicHierarchy(this.rootKey);
            for (int i = 1; i <= this.getAccountPath().size(); ++i) {
                this.basicKeyChain.importKey(this.hierarchy.get((List<ChildNumber>)this.getAccountPath().subList(0, i), false, true));
            }
            this.initializeHierarchyUnencrypted(this.rootKey);
        }
    }

    protected DeterministicKeyChain(KeyCrypter crypter, KeyParameter aesKey, DeterministicKeyChain chain) {
        Preconditions.checkNotNull((Object)chain.rootKey);
        Preconditions.checkNotNull((Object)chain.seed);
        Preconditions.checkArgument((!chain.rootKey.isEncrypted() ? 1 : 0) != 0, (Object)"Chain already encrypted");
        this.accountPath = chain.getAccountPath();
        this.outputScriptType = chain.outputScriptType;
        this.issuedExternalKeys = chain.issuedExternalKeys;
        this.issuedInternalKeys = chain.issuedInternalKeys;
        this.lookaheadSize = chain.lookaheadSize;
        this.lookaheadThreshold = chain.lookaheadThreshold;
        this.seed = chain.seed.encrypt(crypter, aesKey);
        this.basicKeyChain = new BasicKeyChain(crypter);
        this.rootKey = chain.rootKey.encrypt(crypter, aesKey, null);
        this.hierarchy = new DeterministicHierarchy(this.rootKey);
        this.basicKeyChain.importKey(this.rootKey);
        for (int i = 1; i < this.getAccountPath().size(); ++i) {
            this.encryptNonLeaf(aesKey, chain, this.rootKey, (ImmutableList<ChildNumber>)this.getAccountPath().subList(0, i));
        }
        DeterministicKey account = this.encryptNonLeaf(aesKey, chain, this.rootKey, this.getAccountPath());
        this.externalParentKey = this.encryptNonLeaf(aesKey, chain, account, HDUtils.concat(this.getAccountPath(), EXTERNAL_SUBPATH));
        this.internalParentKey = this.encryptNonLeaf(aesKey, chain, account, HDUtils.concat(this.getAccountPath(), INTERNAL_SUBPATH));
        for (ECKey eCKey : chain.basicKeyChain.getKeys()) {
            DeterministicKey key = (DeterministicKey)eCKey;
            if (key.getPath().size() != this.getAccountPath().size() + 2) continue;
            DeterministicKey parent = this.hierarchy.get((List<ChildNumber>)((DeterministicKey)Preconditions.checkNotNull((Object)key.getParent())).getPath(), false, false);
            key = new DeterministicKey(key.dropPrivateBytes(), parent);
            this.hierarchy.putKey(key);
            this.basicKeyChain.importKey(key);
        }
        for (ListenerRegistration listenerRegistration : chain.basicKeyChain.getListeners()) {
            this.basicKeyChain.addEventListener(listenerRegistration);
        }
    }

    public ImmutableList<ChildNumber> getAccountPath() {
        return this.accountPath;
    }

    public Script.ScriptType getOutputScriptType() {
        return this.outputScriptType;
    }

    private DeterministicKey encryptNonLeaf(KeyParameter aesKey, DeterministicKeyChain chain, DeterministicKey parent, ImmutableList<ChildNumber> path) {
        DeterministicKey key = chain.hierarchy.get((List<ChildNumber>)path, false, false);
        key = key.encrypt((KeyCrypter)Preconditions.checkNotNull((Object)this.basicKeyChain.getKeyCrypter()), aesKey, parent);
        this.hierarchy.putKey(key);
        this.basicKeyChain.importKey(key);
        return key;
    }

    private void initializeHierarchyUnencrypted(DeterministicKey baseKey) {
        this.externalParentKey = this.hierarchy.deriveChild((List<ChildNumber>)this.getAccountPath(), false, false, ChildNumber.ZERO);
        this.internalParentKey = this.hierarchy.deriveChild((List<ChildNumber>)this.getAccountPath(), false, false, ChildNumber.ONE);
        this.basicKeyChain.importKey(this.externalParentKey);
        this.basicKeyChain.importKey(this.internalParentKey);
    }

    @Override
    public DeterministicKey getKey(KeyChain.KeyPurpose purpose) {
        return this.getKeys(purpose, 1).get(0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<DeterministicKey> getKeys(KeyChain.KeyPurpose purpose, int numberOfKeys) {
        Preconditions.checkArgument((numberOfKeys > 0 ? 1 : 0) != 0);
        this.lock.lock();
        try {
            DeterministicKey parentKey;
            int index;
            switch (purpose) {
                case RECEIVE_FUNDS: 
                case REFUND: {
                    this.issuedExternalKeys += numberOfKeys;
                    index = this.issuedExternalKeys;
                    parentKey = this.externalParentKey;
                    break;
                }
                case AUTHENTICATION: 
                case CHANGE: {
                    this.issuedInternalKeys += numberOfKeys;
                    index = this.issuedInternalKeys;
                    parentKey = this.internalParentKey;
                    break;
                }
                default: {
                    throw new UnsupportedOperationException();
                }
            }
            List<DeterministicKey> lookahead = this.maybeLookAhead(parentKey, index, 0, 0);
            this.basicKeyChain.importKeys(lookahead);
            ArrayList<DeterministicKey> keys = new ArrayList<DeterministicKey>(numberOfKeys);
            for (int i = 0; i < numberOfKeys; ++i) {
                ImmutableList<ChildNumber> path = HDUtils.append(parentKey.getPath(), new ChildNumber(index - numberOfKeys + i, false));
                DeterministicKey k = this.hierarchy.get((List<ChildNumber>)path, false, false);
                this.checkForBitFlip(k);
                keys.add(k);
            }
            ArrayList<DeterministicKey> arrayList = keys;
            return arrayList;
        }
        finally {
            this.lock.unlock();
        }
    }

    private void checkForBitFlip(DeterministicKey k) {
        DeterministicKey parent = (DeterministicKey)Preconditions.checkNotNull((Object)k.getParent());
        byte[] rederived = HDKeyDerivation.deriveChildKeyBytesFromPublic((DeterministicKey)parent, (ChildNumber)k.getChildNumber(), (HDKeyDerivation.PublicDeriveMode)HDKeyDerivation.PublicDeriveMode.WITH_INVERSION).keyBytes;
        byte[] actual = k.getPubKey();
        if (!Arrays.equals(rederived, actual)) {
            throw new IllegalStateException(String.format(Locale.US, "Bit-flip check failed: %s vs %s", Arrays.toString(rederived), Arrays.toString(actual)));
        }
    }

    public DeterministicKey markKeyAsUsed(DeterministicKey k) {
        int numChildren = k.getChildNumber().i() + 1;
        if (k.getParent() == this.internalParentKey) {
            if (this.issuedInternalKeys < numChildren) {
                this.issuedInternalKeys = numChildren;
                this.maybeLookAhead();
            }
        } else if (k.getParent() == this.externalParentKey && this.issuedExternalKeys < numChildren) {
            this.issuedExternalKeys = numChildren;
            this.maybeLookAhead();
        }
        return k;
    }

    public DeterministicKey findKeyFromPubHash(byte[] pubkeyHash) {
        this.lock.lock();
        try {
            DeterministicKey deterministicKey = (DeterministicKey)this.basicKeyChain.findKeyFromPubHash(pubkeyHash);
            return deterministicKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    public DeterministicKey findKeyFromPubKey(byte[] pubkey) {
        this.lock.lock();
        try {
            DeterministicKey deterministicKey = (DeterministicKey)this.basicKeyChain.findKeyFromPubKey(pubkey);
            return deterministicKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public DeterministicKey markPubHashAsUsed(byte[] pubkeyHash) {
        this.lock.lock();
        try {
            DeterministicKey k = (DeterministicKey)this.basicKeyChain.findKeyFromPubHash(pubkeyHash);
            if (k != null) {
                this.markKeyAsUsed(k);
            }
            DeterministicKey deterministicKey = k;
            return deterministicKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public DeterministicKey markPubKeyAsUsed(byte[] pubkey) {
        this.lock.lock();
        try {
            DeterministicKey k = (DeterministicKey)this.basicKeyChain.findKeyFromPubKey(pubkey);
            if (k != null) {
                this.markKeyAsUsed(k);
            }
            DeterministicKey deterministicKey = k;
            return deterministicKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public boolean hasKey(ECKey key) {
        this.lock.lock();
        try {
            boolean bl = this.basicKeyChain.hasKey(key);
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    protected DeterministicKey getKeyByPath(ChildNumber ... path) {
        return this.getKeyByPath((List<ChildNumber>)ImmutableList.copyOf((Object[])path));
    }

    protected DeterministicKey getKeyByPath(List<ChildNumber> path) {
        return this.getKeyByPath(path, false);
    }

    public DeterministicKey getKeyByPath(List<ChildNumber> path, boolean create) {
        return this.hierarchy.get(path, false, create);
    }

    public DeterministicKey getWatchingKey() {
        return this.getKeyByPath((List<ChildNumber>)this.getAccountPath());
    }

    public boolean isWatching() {
        return this.getWatchingKey().isWatching();
    }

    @Override
    public int numKeys() {
        this.lock.lock();
        try {
            this.maybeLookAhead();
            int n = this.basicKeyChain.numKeys();
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public int numLeafKeysIssued() {
        this.lock.lock();
        try {
            int n = this.issuedExternalKeys + this.issuedInternalKeys;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public long getEarliestKeyCreationTime() {
        if (this.seed != null) {
            return this.seed.getCreationTimeSeconds();
        }
        return this.getWatchingKey().getCreationTimeSeconds();
    }

    @Override
    public void addEventListener(KeyChainEventListener listener) {
        this.basicKeyChain.addEventListener(listener);
    }

    @Override
    public void addEventListener(KeyChainEventListener listener, Executor executor) {
        this.basicKeyChain.addEventListener(listener, executor);
    }

    @Override
    public boolean removeEventListener(KeyChainEventListener listener) {
        return this.basicKeyChain.removeEventListener(listener);
    }

    @Nullable
    public List<String> getMnemonicCode() {
        if (this.seed == null) {
            return null;
        }
        this.lock.lock();
        try {
            List<String> list = this.seed.getMnemonicCode();
            return list;
        }
        finally {
            this.lock.unlock();
        }
    }

    public boolean isFollowing() {
        return this.isFollowing;
    }

    @Override
    public List<Protos.Key> serializeToProtobuf() {
        ArrayList result = Lists.newArrayList();
        this.lock.lock();
        try {
            result.addAll(this.serializeMyselfToProtobuf());
        }
        finally {
            this.lock.unlock();
        }
        return result;
    }

    protected List<Protos.Key> serializeMyselfToProtobuf() {
        LinkedList entries = Lists.newLinkedList();
        if (this.seed != null) {
            Protos.Key.Builder mnemonicEntry = BasicKeyChain.serializeEncryptableItem(this.seed);
            mnemonicEntry.setType(Protos.Key.Type.DETERMINISTIC_MNEMONIC);
            DeterministicKeyChain.serializeSeedEncryptableItem(this.seed, mnemonicEntry);
            for (ChildNumber childNumber : this.getAccountPath()) {
                mnemonicEntry.addAccountPath(childNumber.i());
            }
            entries.add(mnemonicEntry.build());
        }
        Map<ECKey, Protos.Key.Builder> keys = this.basicKeyChain.serializeToEditableProtobufs();
        for (Map.Entry<ECKey, Protos.Key.Builder> entry : keys.entrySet()) {
            DeterministicKey key = (DeterministicKey)entry.getKey();
            Protos.Key.Builder proto = entry.getValue();
            proto.setType(Protos.Key.Type.DETERMINISTIC_KEY);
            Protos.DeterministicKey.Builder detKey = proto.getDeterministicKeyBuilder();
            detKey.setChainCode(ByteString.copyFrom((byte[])key.getChainCode()));
            for (ChildNumber num : key.getPath()) {
                detKey.addPath(num.i());
            }
            if (key.equals(this.externalParentKey)) {
                detKey.setIssuedSubkeys(this.issuedExternalKeys);
                detKey.setLookaheadSize(this.lookaheadSize);
                detKey.setSigsRequiredToSpend(this.getSigsRequiredToSpend());
            } else if (key.equals(this.internalParentKey)) {
                detKey.setIssuedSubkeys(this.issuedInternalKeys);
                detKey.setLookaheadSize(this.lookaheadSize);
                detKey.setSigsRequiredToSpend(this.getSigsRequiredToSpend());
            }
            if (entries.isEmpty() && this.isFollowing()) {
                detKey.setIsFollowing(true);
            }
            if (key.getParent() != null) {
                proto.clearCreationTimestamp();
            } else {
                proto.setOutputScriptType(Protos.Key.OutputScriptType.valueOf(this.outputScriptType.name()));
            }
            entries.add(proto.build());
        }
        return entries;
    }

    static List<DeterministicKeyChain> fromProtobuf(List<Protos.Key> keys, @Nullable KeyCrypter crypter) throws UnreadableWalletException {
        return DeterministicKeyChain.fromProtobuf(keys, crypter, new DefaultKeyChainFactory());
    }

    public static List<DeterministicKeyChain> fromProtobuf(List<Protos.Key> keys, @Nullable KeyCrypter crypter, KeyChainFactory factory) throws UnreadableWalletException {
        LinkedList chains = Lists.newLinkedList();
        DeterministicSeed seed = null;
        DeterministicKeyChain chain = null;
        int lookaheadSize = -1;
        int sigsRequiredToSpend = 1;
        ImmutableList<ChildNumber> accountPath = Lists.newArrayList();
        Script.ScriptType outputScriptType = Script.ScriptType.P2PKH;
        PeekingIterator iter = Iterators.peekingIterator(keys.iterator());
        while (iter.hasNext()) {
            DeterministicKey detkey;
            Object passphrase;
            Protos.Key key = (Protos.Key)iter.next();
            Protos.Key.Type t = key.getType();
            if (t == Protos.Key.Type.DETERMINISTIC_MNEMONIC) {
                accountPath = Lists.newArrayList();
                for (int i : key.getAccountPathList()) {
                    accountPath.add(new ChildNumber(i));
                }
                if (accountPath.isEmpty()) {
                    accountPath = ACCOUNT_ZERO_PATH;
                }
                if (chain != null) {
                    Preconditions.checkState((lookaheadSize >= 0 ? 1 : 0) != 0);
                    chain.setLookaheadSize(lookaheadSize);
                    chain.setSigsRequiredToSpend(sigsRequiredToSpend);
                    chain.maybeLookAhead();
                    chains.add(chain);
                    chain = null;
                }
                long timestamp = key.getCreationTimestamp() / 1000L;
                passphrase = DEFAULT_PASSPHRASE_FOR_MNEMONIC;
                if (key.hasSecretBytes()) {
                    if (key.hasEncryptedDeterministicSeed()) {
                        throw new UnreadableWalletException("Malformed key proto: " + key.toString());
                    }
                    byte[] seedBytes = null;
                    if (key.hasDeterministicSeed()) {
                        seedBytes = key.getDeterministicSeed().toByteArray();
                    }
                    seed = new DeterministicSeed(key.getSecretBytes().toStringUtf8(), seedBytes, (String)passphrase, timestamp);
                } else if (key.hasEncryptedData()) {
                    if (key.hasDeterministicSeed()) {
                        throw new UnreadableWalletException("Malformed key proto: " + key.toString());
                    }
                    EncryptedData data = new EncryptedData(key.getEncryptedData().getInitialisationVector().toByteArray(), key.getEncryptedData().getEncryptedPrivateKey().toByteArray());
                    EncryptedData encryptedSeedBytes = null;
                    if (key.hasEncryptedDeterministicSeed()) {
                        Protos.EncryptedData encryptedSeed = key.getEncryptedDeterministicSeed();
                        encryptedSeedBytes = new EncryptedData(encryptedSeed.getInitialisationVector().toByteArray(), encryptedSeed.getEncryptedPrivateKey().toByteArray());
                    }
                    seed = new DeterministicSeed(data, encryptedSeedBytes, timestamp);
                } else {
                    throw new UnreadableWalletException("Malformed key proto: " + key.toString());
                }
                if (!log.isDebugEnabled()) continue;
                log.debug("Deserializing: DETERMINISTIC_MNEMONIC: {}", (Object)seed);
                continue;
            }
            if (t != Protos.Key.Type.DETERMINISTIC_KEY) continue;
            if (!key.hasDeterministicKey()) {
                throw new UnreadableWalletException("Deterministic key missing extra data: " + key.toString());
            }
            byte[] chainCode = key.getDeterministicKey().getChainCode().toByteArray();
            LinkedList path = Lists.newLinkedList();
            passphrase = key.getDeterministicKey().getPathList().iterator();
            while (passphrase.hasNext()) {
                int i = (Integer)passphrase.next();
                path.add(new ChildNumber(i));
            }
            LazyECPoint pubkey = new LazyECPoint(ECKey.CURVE.getCurve(), key.getPublicKey().toByteArray());
            ImmutableList immutablePath = ImmutableList.copyOf((Collection)path);
            if (key.hasOutputScriptType()) {
                outputScriptType = Script.ScriptType.valueOf(key.getOutputScriptType().name());
            }
            boolean isWatchingAccountKey = false;
            boolean isFollowingKey = false;
            boolean isSpendingKey = false;
            if (key.getDeterministicKey().getIsFollowing()) {
                if (chain != null) {
                    Preconditions.checkState((lookaheadSize >= 0 ? 1 : 0) != 0);
                    chain.setLookaheadSize(lookaheadSize);
                    chain.setSigsRequiredToSpend(sigsRequiredToSpend);
                    chain.maybeLookAhead();
                    chains.add(chain);
                    chain = null;
                    seed = null;
                }
                isFollowingKey = true;
            }
            if (chain == null) {
                DeterministicKey accountKey;
                boolean isMarried;
                boolean bl = isMarried = !isFollowingKey && !chains.isEmpty() && ((DeterministicKeyChain)chains.get(chains.size() - 1)).isFollowing();
                if (seed == null && key.hasSecretBytes()) {
                    accountKey = new DeterministicKey((ImmutableList<ChildNumber>)immutablePath, chainCode, pubkey, new BigInteger(1, key.getSecretBytes().toByteArray()), null);
                    accountKey.setCreationTimeSeconds(key.getCreationTimestamp() / 1000L);
                    chain = factory.makeSpendingKeyChain(key, (Protos.Key)iter.peek(), accountKey, isMarried, outputScriptType);
                    isSpendingKey = true;
                } else if (seed == null) {
                    accountKey = new DeterministicKey((ImmutableList<ChildNumber>)immutablePath, chainCode, pubkey, null, null);
                    accountKey.setCreationTimeSeconds(key.getCreationTimestamp() / 1000L);
                    chain = factory.makeWatchingKeyChain(key, (Protos.Key)iter.peek(), accountKey, isFollowingKey, isMarried, outputScriptType);
                    isWatchingAccountKey = true;
                } else {
                    chain = factory.makeKeyChain(key, (Protos.Key)iter.peek(), seed, crypter, isMarried, outputScriptType, (ImmutableList<ChildNumber>)ImmutableList.builder().addAll(accountPath).build());
                    chain.lookaheadSize = -1;
                }
            }
            DeterministicKey parent = null;
            if (!(path.isEmpty() || isWatchingAccountKey || isSpendingKey)) {
                ChildNumber index = (ChildNumber)path.removeLast();
                parent = chain.hierarchy.get(path, false, false);
                path.add(index);
            }
            if (key.hasSecretBytes()) {
                BigInteger priv = new BigInteger(1, key.getSecretBytes().toByteArray());
                detkey = new DeterministicKey((ImmutableList<ChildNumber>)immutablePath, chainCode, pubkey, priv, parent);
            } else if (key.hasEncryptedData()) {
                Protos.EncryptedData proto = key.getEncryptedData();
                EncryptedData data = new EncryptedData(proto.getInitialisationVector().toByteArray(), proto.getEncryptedPrivateKey().toByteArray());
                Preconditions.checkNotNull((Object)crypter, (Object)"Encountered an encrypted key but no key crypter provided");
                detkey = new DeterministicKey((ImmutableList<ChildNumber>)immutablePath, chainCode, crypter, pubkey, data, parent);
            } else {
                detkey = new DeterministicKey((ImmutableList<ChildNumber>)immutablePath, chainCode, pubkey, null, parent);
            }
            if (key.hasCreationTimestamp()) {
                detkey.setCreationTimeSeconds(key.getCreationTimestamp() / 1000L);
            }
            if (log.isDebugEnabled()) {
                log.debug("Deserializing: DETERMINISTIC_KEY: {}", (Object)detkey);
            }
            if (!isWatchingAccountKey) {
                if (path.isEmpty()) {
                    if (chain.rootKey == null) {
                        chain.rootKey = detkey;
                        chain.hierarchy = new DeterministicHierarchy(detkey);
                    }
                } else if (path.size() == chain.getAccountPath().size() + 1 || isSpendingKey) {
                    if (detkey.getChildNumber().num() == 0) {
                        chain.externalParentKey = detkey;
                        chain.issuedExternalKeys = key.getDeterministicKey().getIssuedSubkeys();
                        lookaheadSize = Math.max(lookaheadSize, key.getDeterministicKey().getLookaheadSize());
                        sigsRequiredToSpend = key.getDeterministicKey().getSigsRequiredToSpend();
                    } else if (detkey.getChildNumber().num() == 1) {
                        chain.internalParentKey = detkey;
                        chain.issuedInternalKeys = key.getDeterministicKey().getIssuedSubkeys();
                    }
                }
            }
            chain.hierarchy.putKey(detkey);
            chain.basicKeyChain.importKey(detkey);
        }
        if (chain != null) {
            Preconditions.checkState((lookaheadSize >= 0 ? 1 : 0) != 0);
            chain.setLookaheadSize(lookaheadSize);
            chain.setSigsRequiredToSpend(sigsRequiredToSpend);
            chain.maybeLookAhead();
            chains.add(chain);
        }
        return chains;
    }

    @Override
    public DeterministicKeyChain toEncrypted(CharSequence password) {
        Preconditions.checkNotNull((Object)password);
        Preconditions.checkArgument((password.length() > 0 ? 1 : 0) != 0);
        Preconditions.checkState((this.seed != null ? 1 : 0) != 0, (Object)"Attempt to encrypt a watching chain.");
        Preconditions.checkState((!this.seed.isEncrypted() ? 1 : 0) != 0);
        KeyCrypterScrypt scrypt = new KeyCrypterScrypt();
        KeyParameter derivedKey = scrypt.deriveKey(password);
        return this.toEncrypted(scrypt, derivedKey);
    }

    @Override
    public DeterministicKeyChain toEncrypted(KeyCrypter keyCrypter, KeyParameter aesKey) {
        return new DeterministicKeyChain(keyCrypter, aesKey, this);
    }

    @Override
    public DeterministicKeyChain toDecrypted(CharSequence password) {
        Preconditions.checkNotNull((Object)password);
        Preconditions.checkArgument((password.length() > 0 ? 1 : 0) != 0);
        KeyCrypter crypter = this.getKeyCrypter();
        Preconditions.checkState((crypter != null ? 1 : 0) != 0, (Object)"Chain not encrypted");
        KeyParameter derivedKey = crypter.deriveKey(password);
        return this.toDecrypted(derivedKey);
    }

    @Override
    public DeterministicKeyChain toDecrypted(KeyParameter aesKey) {
        Preconditions.checkState((this.getKeyCrypter() != null ? 1 : 0) != 0, (Object)"Key chain not encrypted");
        Preconditions.checkState((this.seed != null ? 1 : 0) != 0, (Object)"Can't decrypt a watching chain");
        Preconditions.checkState((boolean)this.seed.isEncrypted());
        String passphrase = DEFAULT_PASSPHRASE_FOR_MNEMONIC;
        DeterministicSeed decSeed = this.seed.decrypt(this.getKeyCrypter(), passphrase, aesKey);
        DeterministicKeyChain chain = this.makeKeyChainFromSeed(decSeed, this.getAccountPath(), this.outputScriptType);
        if (!chain.getWatchingKey().getPubKeyPoint().equals(this.getWatchingKey().getPubKeyPoint())) {
            throw new KeyCrypterException.PublicPrivateMismatch("Provided AES key is wrong");
        }
        chain.lookaheadSize = this.lookaheadSize;
        for (ECKey eCKey : this.basicKeyChain.getKeys()) {
            DeterministicKey key = (DeterministicKey)eCKey;
            if (key.getPath().size() != this.getAccountPath().size() + 2) continue;
            Preconditions.checkState((boolean)key.isEncrypted());
            DeterministicKey parent = chain.hierarchy.get((List<ChildNumber>)((DeterministicKey)Preconditions.checkNotNull((Object)key.getParent())).getPath(), false, false);
            key = new DeterministicKey(key.dropPrivateBytes(), parent);
            chain.hierarchy.putKey(key);
            chain.basicKeyChain.importKey(key);
        }
        chain.issuedExternalKeys = this.issuedExternalKeys;
        chain.issuedInternalKeys = this.issuedInternalKeys;
        for (ListenerRegistration listenerRegistration : this.basicKeyChain.getListeners()) {
            chain.basicKeyChain.addEventListener(listenerRegistration);
        }
        return chain;
    }

    protected DeterministicKeyChain makeKeyChainFromSeed(DeterministicSeed seed, ImmutableList<ChildNumber> accountPath, Script.ScriptType outputScriptType) {
        return new DeterministicKeyChain(seed, null, outputScriptType, accountPath);
    }

    @Override
    public boolean checkPassword(CharSequence password) {
        Preconditions.checkNotNull((Object)password);
        Preconditions.checkState((this.getKeyCrypter() != null ? 1 : 0) != 0, (Object)"Key chain not encrypted");
        return this.checkAESKey(this.getKeyCrypter().deriveKey(password));
    }

    @Override
    public boolean checkAESKey(KeyParameter aesKey) {
        Preconditions.checkState((this.rootKey != null ? 1 : 0) != 0, (Object)"Can't check password for a watching chain");
        Preconditions.checkNotNull((Object)aesKey);
        Preconditions.checkState((this.getKeyCrypter() != null ? 1 : 0) != 0, (Object)"Key chain not encrypted");
        try {
            return this.rootKey.decrypt(aesKey).getPubKeyPoint().equals(this.rootKey.getPubKeyPoint());
        }
        catch (KeyCrypterException e) {
            return false;
        }
    }

    @Override
    @Nullable
    public KeyCrypter getKeyCrypter() {
        return this.basicKeyChain.getKeyCrypter();
    }

    @Override
    public int numBloomFilterEntries() {
        return this.numKeys() * 2;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public BloomFilter getFilter(int size, double falsePositiveRate, long tweak) {
        this.lock.lock();
        try {
            Preconditions.checkArgument((size >= this.numBloomFilterEntries() ? 1 : 0) != 0);
            this.maybeLookAhead();
            BloomFilter bloomFilter = this.basicKeyChain.getFilter(size, falsePositiveRate, tweak);
            return bloomFilter;
        }
        finally {
            this.lock.unlock();
        }
    }

    public int getLookaheadSize() {
        this.lock.lock();
        try {
            int n = this.lookaheadSize;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public void setLookaheadSize(int lookaheadSize) {
        this.lock.lock();
        try {
            boolean readjustThreshold = this.lookaheadThreshold == this.calcDefaultLookaheadThreshold();
            this.lookaheadSize = lookaheadSize;
            if (readjustThreshold) {
                this.lookaheadThreshold = this.calcDefaultLookaheadThreshold();
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public void setLookaheadThreshold(int num) {
        this.lock.lock();
        try {
            if (num >= this.lookaheadSize) {
                throw new IllegalArgumentException("Threshold larger or equal to the lookaheadSize");
            }
            this.lookaheadThreshold = num;
        }
        finally {
            this.lock.unlock();
        }
    }

    public int getLookaheadThreshold() {
        this.lock.lock();
        try {
            if (this.lookaheadThreshold >= this.lookaheadSize) {
                int n = 0;
                return n;
            }
            int n = this.lookaheadThreshold;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public void maybeLookAhead() {
        this.lock.lock();
        try {
            List<DeterministicKey> keys = this.maybeLookAhead(this.externalParentKey, this.issuedExternalKeys);
            keys.addAll(this.maybeLookAhead(this.internalParentKey, this.issuedInternalKeys));
            if (keys.isEmpty()) {
                return;
            }
            ++this.keyLookaheadEpoch;
            this.basicKeyChain.importKeys(keys);
        }
        finally {
            this.lock.unlock();
        }
    }

    private List<DeterministicKey> maybeLookAhead(DeterministicKey parent, int issued) {
        Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread());
        return this.maybeLookAhead(parent, issued, this.getLookaheadSize(), this.getLookaheadThreshold());
    }

    private List<DeterministicKey> maybeLookAhead(DeterministicKey parent, int issued, int lookaheadSize, int lookaheadThreshold) {
        Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread());
        int numChildren = this.hierarchy.getNumChildren(parent.getPath());
        int needed = issued + lookaheadSize + lookaheadThreshold - numChildren;
        if (needed <= lookaheadThreshold) {
            return new ArrayList<DeterministicKey>();
        }
        log.info("{} keys needed for {} = {} issued + {} lookahead size + {} lookahead threshold - {} num children", new Object[]{needed, parent.getPathAsString(), issued, lookaheadSize, lookaheadThreshold, numChildren});
        ArrayList<DeterministicKey> result = new ArrayList<DeterministicKey>(needed);
        Stopwatch watch = Stopwatch.createStarted();
        int nextChild = numChildren;
        for (int i = 0; i < needed; ++i) {
            DeterministicKey key = HDKeyDerivation.deriveThisOrNextChildKey(parent, nextChild);
            key = key.dropPrivateBytes();
            this.hierarchy.putKey(key);
            result.add(key);
            nextChild = key.getChildNumber().num() + 1;
        }
        watch.stop();
        log.info("Took {}", (Object)watch);
        return result;
    }

    public void maybeLookAheadScripts() {
    }

    public int getIssuedExternalKeys() {
        this.lock.lock();
        try {
            int n = this.issuedExternalKeys;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public int getIssuedInternalKeys() {
        this.lock.lock();
        try {
            int n = this.issuedInternalKeys;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Nullable
    public DeterministicSeed getSeed() {
        this.lock.lock();
        try {
            DeterministicSeed deterministicSeed = this.seed;
            return deterministicSeed;
        }
        finally {
            this.lock.unlock();
        }
    }

    List<DeterministicKey> getKeys(boolean includeLookahead, boolean includeParents) {
        List<ECKey> keys = this.basicKeyChain.getKeys();
        LinkedList<DeterministicKey> result = new LinkedList<DeterministicKey>();
        if (!includeLookahead) {
            int treeSize = this.internalParentKey.getPath().size();
            for (ECKey key : keys) {
                DeterministicKey detkey = (DeterministicKey)key;
                DeterministicKey parent = detkey.getParent();
                if (!includeParents && parent == null || !includeParents && detkey.getPath().size() <= treeSize || this.internalParentKey.equals(parent) && detkey.getChildNumber().i() >= this.issuedInternalKeys || this.externalParentKey.equals(parent) && detkey.getChildNumber().i() >= this.issuedExternalKeys) continue;
                result.add(detkey);
            }
        } else {
            for (ECKey key : keys) {
                result.add((DeterministicKey)key);
            }
        }
        return result;
    }

    public List<DeterministicKey> getIssuedReceiveKeys() {
        ArrayList<DeterministicKey> keys = new ArrayList<DeterministicKey>(this.getKeys(false, false));
        Iterator i = keys.iterator();
        while (i.hasNext()) {
            DeterministicKey parent = ((DeterministicKey)i.next()).getParent();
            if (parent != null && this.externalParentKey.equals(parent)) continue;
            i.remove();
        }
        return keys;
    }

    public List<DeterministicKey> getLeafKeys() {
        ImmutableList.Builder keys = ImmutableList.builder();
        for (ECKey eCKey : this.getKeys(true, false)) {
            DeterministicKey dKey = (DeterministicKey)eCKey;
            if (dKey.getPath().size() != this.getAccountPath().size() + 2) continue;
            keys.add((Object)dKey);
        }
        return keys.build();
    }

    static void serializeSeedEncryptableItem(DeterministicSeed seed, Protos.Key.Builder proto) {
        if (seed.isEncrypted() && seed.getEncryptedSeedData() != null) {
            EncryptedData data = seed.getEncryptedSeedData();
            proto.getEncryptedDeterministicSeedBuilder().setEncryptedPrivateKey(ByteString.copyFrom((byte[])data.encryptedBytes)).setInitialisationVector(ByteString.copyFrom((byte[])data.initialisationVector));
            Preconditions.checkState((seed.getEncryptionType() == Protos.Wallet.EncryptionType.ENCRYPTED_SCRYPT_AES ? 1 : 0) != 0);
        } else {
            byte[] secret = seed.getSeedBytes();
            if (secret != null) {
                proto.setDeterministicSeed(ByteString.copyFrom((byte[])secret));
            }
        }
    }

    public int getKeyLookaheadEpoch() {
        this.lock.lock();
        try {
            int n = this.keyLookaheadEpoch;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public boolean isMarried() {
        return false;
    }

    public RedeemData getRedeemData(DeterministicKey followedKey) {
        throw new UnsupportedOperationException();
    }

    public Script freshOutputScript(KeyChain.KeyPurpose purpose) {
        throw new UnsupportedOperationException();
    }

    public String toString() {
        MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper((Object)this).omitNullValues();
        helper.addValue((Object)this.outputScriptType);
        helper.add("accountPath", this.accountPath);
        helper.add("lookaheadSize", this.lookaheadSize);
        helper.add("lookaheadThreshold", this.lookaheadThreshold);
        if (this.isFollowing) {
            helper.addValue((Object)"following");
        }
        return helper.toString();
    }

    public String toString(boolean includeLookahead, boolean includePrivateKeys, @Nullable KeyParameter aesKey, NetworkParameters params) {
        DeterministicKey watchingKey = this.getWatchingKey();
        StringBuilder builder = new StringBuilder();
        if (this.seed != null) {
            if (includePrivateKeys) {
                DeterministicSeed decryptedSeed = this.seed.isEncrypted() ? this.seed.decrypt(this.getKeyCrypter(), DEFAULT_PASSPHRASE_FOR_MNEMONIC, aesKey) : this.seed;
                List<String> words = decryptedSeed.getMnemonicCode();
                builder.append("Seed as words:     ").append(Utils.SPACE_JOINER.join(words)).append('\n');
                builder.append("Seed as hex:       ").append(decryptedSeed.toHexString()).append('\n');
            } else if (this.seed.isEncrypted()) {
                builder.append("Seed is encrypted\n");
            }
            builder.append("Seed birthday:     ").append(this.seed.getCreationTimeSeconds()).append("  [").append(Utils.dateTimeFormat(this.seed.getCreationTimeSeconds() * 1000L)).append("]\n");
        } else {
            builder.append("Key birthday:      ").append(watchingKey.getCreationTimeSeconds()).append("  [").append(Utils.dateTimeFormat(watchingKey.getCreationTimeSeconds() * 1000L)).append("]\n");
        }
        builder.append("Ouput script type: ").append((Object)this.outputScriptType).append('\n');
        builder.append("Key to watch:      ").append(watchingKey.serializePubB58(params, this.outputScriptType)).append('\n');
        builder.append("Lookahead siz/thr: ").append(this.lookaheadSize).append('/').append(this.lookaheadThreshold).append('\n');
        this.formatAddresses(includeLookahead, includePrivateKeys, aesKey, params, builder);
        return builder.toString();
    }

    protected void formatAddresses(boolean includeLookahead, boolean includePrivateKeys, @Nullable KeyParameter aesKey, NetworkParameters params, StringBuilder builder) {
        for (DeterministicKey key : this.getKeys(includeLookahead, true)) {
            String comment = null;
            if (key.equals(this.rootKey)) {
                comment = "root";
            } else if (key.equals(this.getWatchingKey())) {
                comment = "account";
            } else if (key.equals(this.internalParentKey)) {
                comment = "internal";
            } else if (key.equals(this.externalParentKey)) {
                comment = "external";
            } else if (this.internalParentKey.equals(key.getParent()) && key.getChildNumber().i() >= this.issuedInternalKeys) {
                comment = "*";
            } else if (this.externalParentKey.equals(key.getParent()) && key.getChildNumber().i() >= this.issuedExternalKeys) {
                comment = "*";
            }
            key.formatKeyWithAddress(includePrivateKeys, aesKey, builder, params, this.outputScriptType, comment);
        }
    }

    public void setSigsRequiredToSpend(int sigsRequiredToSpend) {
        this.sigsRequiredToSpend = sigsRequiredToSpend;
    }

    public int getSigsRequiredToSpend() {
        return this.sigsRequiredToSpend;
    }

    @Nullable
    public RedeemData findRedeemDataByScriptHash(ByteString bytes) {
        return null;
    }

    public static class Builder<T extends Builder<T>> {
        protected SecureRandom random;
        protected int bits = 128;
        protected String passphrase;
        protected long creationTimeSecs = 0L;
        protected byte[] entropy;
        protected DeterministicSeed seed;
        protected Script.ScriptType outputScriptType = Script.ScriptType.P2PKH;
        protected DeterministicKey watchingKey = null;
        protected boolean isFollowing = false;
        protected DeterministicKey spendingKey = null;
        protected ImmutableList<ChildNumber> accountPath = null;

        protected Builder() {
        }

        protected T self() {
            return (T)this;
        }

        public T entropy(byte[] entropy, long creationTimeSecs) {
            this.entropy = entropy;
            this.creationTimeSecs = creationTimeSecs;
            return this.self();
        }

        public T seed(DeterministicSeed seed) {
            this.seed = seed;
            return this.self();
        }

        public T random(SecureRandom random, int bits) {
            this.random = random;
            this.bits = bits;
            return this.self();
        }

        public T random(SecureRandom random) {
            this.random = random;
            return this.self();
        }

        public T watch(DeterministicKey accountKey) {
            Preconditions.checkState((this.accountPath == null ? 1 : 0) != 0, (Object)"either watch or accountPath");
            this.watchingKey = accountKey;
            this.isFollowing = false;
            return this.self();
        }

        public T watchAndFollow(DeterministicKey accountKey) {
            Preconditions.checkState((this.accountPath == null ? 1 : 0) != 0, (Object)"either watchAndFollow or accountPath");
            this.watchingKey = accountKey;
            this.isFollowing = true;
            return this.self();
        }

        public T spend(DeterministicKey accountKey) {
            Preconditions.checkState((this.accountPath == null ? 1 : 0) != 0, (Object)"either spend or accountPath");
            this.spendingKey = accountKey;
            this.isFollowing = false;
            return this.self();
        }

        public T outputScriptType(Script.ScriptType outputScriptType) {
            this.outputScriptType = outputScriptType;
            return this.self();
        }

        public T passphrase(String passphrase) {
            this.passphrase = passphrase;
            return this.self();
        }

        public T accountPath(ImmutableList<ChildNumber> accountPath) {
            Preconditions.checkState((this.watchingKey == null ? 1 : 0) != 0, (Object)"either watch or accountPath");
            this.accountPath = (ImmutableList)Preconditions.checkNotNull(accountPath);
            return this.self();
        }

        public DeterministicKeyChain build() {
            Preconditions.checkState((this.passphrase == null || this.seed == null ? 1 : 0) != 0, (Object)"Passphrase must not be specified with seed");
            if (this.accountPath == null) {
                this.accountPath = ACCOUNT_ZERO_PATH;
            }
            if (this.random != null) {
                return new DeterministicKeyChain(new DeterministicSeed(this.random, this.bits, this.getPassphrase()), null, this.outputScriptType, this.accountPath);
            }
            if (this.entropy != null) {
                return new DeterministicKeyChain(new DeterministicSeed(this.entropy, this.getPassphrase(), this.creationTimeSecs), null, this.outputScriptType, this.accountPath);
            }
            if (this.seed != null) {
                return new DeterministicKeyChain(this.seed, null, this.outputScriptType, this.accountPath);
            }
            if (this.watchingKey != null) {
                return new DeterministicKeyChain(this.watchingKey, this.isFollowing, true, this.outputScriptType);
            }
            if (this.spendingKey != null) {
                return new DeterministicKeyChain(this.spendingKey, false, false, this.outputScriptType);
            }
            throw new IllegalStateException();
        }

        protected String getPassphrase() {
            return this.passphrase != null ? this.passphrase : DeterministicKeyChain.DEFAULT_PASSPHRASE_FOR_MNEMONIC;
        }
    }
}

