/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.blockmanagement;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocol.BlockListAsLongs;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoStriped;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.Namesystem;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.Phase;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.StartupProgress;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.Status;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.Step;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.StepType;
import org.apache.hadoop.net.NetworkTopology;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
class BlockManagerSafeMode {
    static final Logger LOG = LoggerFactory.getLogger(BlockManagerSafeMode.class);
    static final Step STEP_AWAITING_REPORTED_BLOCKS = new Step(StepType.AWAITING_REPORTED_BLOCKS);
    private final BlockManager blockManager;
    private final Namesystem namesystem;
    private final boolean haEnabled;
    private volatile BMSafeModeStatus status = BMSafeModeStatus.OFF;
    private final double threshold;
    private long blockThreshold;
    private long blockTotal;
    private long blockSafe;
    private final int datanodeThreshold;
    private final int safeReplication;
    private final double replQueueThreshold;
    private long blockReplQueueThreshold;
    @VisibleForTesting
    final long extension;
    private final AtomicLong reachedTime = new AtomicLong();
    private long startTime;
    private final Daemon smmthread = new Daemon((Runnable)new SafeModeMonitor());
    private long lastStatusReport;
    private StartupProgress.Counter awaitingReportedBlocksCounter;
    private final LongAdder bytesInFutureBlocks = new LongAdder();
    private final LongAdder bytesInFutureECBlockGroups = new LongAdder();
    private final boolean inRollBack;

    BlockManagerSafeMode(BlockManager blockManager, Namesystem namesystem, boolean haEnabled, Configuration conf) {
        this.blockManager = blockManager;
        this.namesystem = namesystem;
        this.haEnabled = haEnabled;
        this.threshold = conf.getFloat("dfs.namenode.safemode.threshold-pct", 0.999f);
        if (this.threshold > 1.0) {
            LOG.warn("The threshold value shouldn't be greater than 1, threshold: {}", (Object)this.threshold);
        }
        this.datanodeThreshold = conf.getInt("dfs.namenode.safemode.min.datanodes", 0);
        int minReplication = conf.getInt("dfs.namenode.replication.min", 1);
        this.safeReplication = conf.getInt("dfs.namenode.safemode.replication.min", minReplication);
        this.replQueueThreshold = conf.getFloat("dfs.namenode.replqueue.threshold-pct", (float)this.threshold);
        this.extension = conf.getTimeDuration("dfs.namenode.safemode.extension", 30000L, TimeUnit.MILLISECONDS);
        this.inRollBack = BlockManagerSafeMode.isInRollBackMode(NameNode.getStartupOption(conf));
        LOG.info("{} = {}", (Object)"dfs.namenode.safemode.threshold-pct", (Object)this.threshold);
        LOG.info("{} = {}", (Object)"dfs.namenode.safemode.min.datanodes", (Object)this.datanodeThreshold);
        LOG.info("{} = {}", (Object)"dfs.namenode.safemode.extension", (Object)this.extension);
    }

    void activate(long total) {
        assert (this.namesystem.hasWriteLock());
        assert (this.status == BMSafeModeStatus.OFF);
        this.startTime = Time.monotonicNow();
        this.setBlockTotal(total);
        if (this.areThresholdsMet()) {
            boolean exitResult = this.leaveSafeMode(false);
            Preconditions.checkState((boolean)exitResult, (Object)"Failed to leave safe mode.");
        } else {
            this.status = BMSafeModeStatus.PENDING_THRESHOLD;
            this.initializeReplQueuesIfNecessary();
            this.reportStatus("STATE* Safe mode ON.", true);
            this.lastStatusReport = Time.monotonicNow();
        }
    }

    boolean isInSafeMode() {
        if (this.status != BMSafeModeStatus.OFF) {
            this.doConsistencyCheck();
            return true;
        }
        return false;
    }

    void checkSafeMode() {
        assert (this.namesystem.hasWriteLock());
        if (this.namesystem.inTransitionToActive()) {
            return;
        }
        switch (this.status) {
            case PENDING_THRESHOLD: {
                if (this.areThresholdsMet()) {
                    if (this.extension > 0L) {
                        this.status = BMSafeModeStatus.EXTENSION;
                        this.reachedTime.set(Time.monotonicNow());
                        this.smmthread.start();
                        this.initializeReplQueuesIfNecessary();
                        this.reportStatus("STATE* Safe mode extension entered.", true);
                        break;
                    }
                    this.leaveSafeMode(false);
                    break;
                }
                this.initializeReplQueuesIfNecessary();
                this.reportStatus("STATE* Safe mode ON.", false);
                break;
            }
            case EXTENSION: {
                this.reportStatus("STATE* Safe mode ON.", false);
                break;
            }
            case OFF: {
                break;
            }
            default: {
                assert (false) : "Non-recognized block manager safe mode status: " + (Object)((Object)this.status);
                break;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void adjustBlockTotals(int deltaSafe, int deltaTotal) {
        long newBlockTotal;
        assert (this.namesystem.hasWriteLock());
        if (!this.isSafeModeTrackingBlocks()) {
            return;
        }
        BlockManagerSafeMode blockManagerSafeMode = this;
        synchronized (blockManagerSafeMode) {
            LOG.debug("Adjusting block totals from {}/{} to {}/{}", new Object[]{this.blockSafe, this.blockTotal, this.blockSafe + (long)deltaSafe, this.blockTotal + (long)deltaTotal});
            assert (this.blockSafe + (long)deltaSafe >= 0L) : "Can't reduce blockSafe " + this.blockSafe + " by " + deltaSafe + ": would be negative";
            assert (this.blockTotal + (long)deltaTotal >= 0L) : "Can't reduce blockTotal " + this.blockTotal + " by " + deltaTotal + ": would be negative";
            this.blockSafe += (long)deltaSafe;
            newBlockTotal = this.blockTotal + (long)deltaTotal;
        }
        this.setBlockTotal(newBlockTotal);
        this.checkSafeMode();
    }

    boolean isSafeModeTrackingBlocks() {
        assert (this.namesystem.hasWriteLock());
        return this.haEnabled && this.status != BMSafeModeStatus.OFF;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setBlockTotal(long total) {
        assert (this.namesystem.hasWriteLock());
        BlockManagerSafeMode blockManagerSafeMode = this;
        synchronized (blockManagerSafeMode) {
            this.blockTotal = total;
            this.blockThreshold = (long)((double)total * this.threshold);
        }
        this.blockReplQueueThreshold = (long)((double)total * this.replQueueThreshold);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    String getSafeModeTip() {
        String msg = "";
        BlockManagerSafeMode blockManagerSafeMode = this;
        synchronized (blockManagerSafeMode) {
            msg = this.blockSafe < this.blockThreshold ? msg + String.format("The reported blocks %d needs additional %d blocks to reach the threshold %.4f of total blocks %d.%n", this.blockSafe, this.blockThreshold - this.blockSafe, this.threshold, this.blockTotal) : msg + String.format("The reported blocks %d has reached the threshold %.4f of total blocks %d. ", this.blockSafe, this.threshold, this.blockTotal);
        }
        int numLive = this.blockManager.getDatanodeManager().getNumLiveDataNodes();
        msg = numLive < this.datanodeThreshold ? msg + String.format("The number of live datanodes %d needs an additional %d live datanodes to reach the minimum number %d.%n", numLive, this.datanodeThreshold - numLive, this.datanodeThreshold) : msg + String.format("The number of live datanodes %d has reached the minimum number %d. ", numLive, this.datanodeThreshold);
        if (this.getBytesInFuture() > 0L) {
            msg = msg + "Name node detected blocks with generation stamps in future. This means that Name node metadata is inconsistent. This can happen if Name node metadata files have been manually replaced. Exiting safe mode will cause loss of " + this.getBytesInFuture() + " byte(s). Please restart name node with right metadata or use \"hdfs dfsadmin -safemode forceExit\" if you are certain that the NameNode was started with the correct FsImage and edit logs. If you encountered this during a rollback, it is safe to exit with -safemode forceExit.";
            return msg;
        }
        String turnOffTip = "Safe mode will be turned off automatically ";
        switch (this.status) {
            case PENDING_THRESHOLD: {
                msg = msg + "Safe mode will be turned off automatically once the thresholds have been reached.";
                break;
            }
            case EXTENSION: {
                msg = msg + "In safe mode extension. Safe mode will be turned off automatically in " + this.timeToLeaveExtension() / 1000L + " seconds.";
                break;
            }
            case OFF: {
                msg = msg + "Safe mode will be turned off automatically soon.";
                break;
            }
            default: {
                assert (false) : "Non-recognized block manager safe mode status: " + (Object)((Object)this.status);
                break;
            }
        }
        return msg;
    }

    /*
     * Enabled aggressive block sorting
     */
    boolean leaveSafeMode(boolean force) {
        assert (this.namesystem.hasWriteLock()) : "Leaving safe mode needs write lock!";
        long bytesInFuture = this.getBytesInFuture();
        if (bytesInFuture > 0L) {
            if (!force) {
                LOG.error("Refusing to leave safe mode without a force flag. Exiting safe mode will cause a deletion of {} byte(s). Please use -forceExit flag to exit safe mode forcefully if data loss is acceptable.", (Object)bytesInFuture);
                return false;
            }
            LOG.warn("Leaving safe mode due to forceExit. This will cause a data loss of {} byte(s).", (Object)bytesInFuture);
            this.bytesInFutureBlocks.reset();
            this.bytesInFutureECBlockGroups.reset();
        } else if (force) {
            LOG.warn("forceExit used when normal exist would suffice. Treating force exit as normal safe mode exit.");
        }
        if (!this.blockManager.isPopulatingReplQueues() && this.blockManager.shouldPopulateReplQueues()) {
            this.blockManager.initializeReplQueues();
        }
        if (this.status != BMSafeModeStatus.OFF) {
            NameNode.stateChangeLog.info("STATE* Safe mode is OFF");
        }
        this.status = BMSafeModeStatus.OFF;
        long timeInSafemode = Time.monotonicNow() - this.startTime;
        NameNode.stateChangeLog.info("STATE* Leaving safe mode after {} secs", (Object)(timeInSafemode / 1000L));
        NameNode.getNameNodeMetrics().setSafeModeTime(timeInSafemode);
        NetworkTopology nt = this.blockManager.getDatanodeManager().getNetworkTopology();
        NameNode.stateChangeLog.info("STATE* Network topology has {} racks and {} datanodes", (Object)nt.getNumOfRacks(), (Object)nt.getNumOfLeaves());
        NameNode.stateChangeLog.info("STATE* UnderReplicatedBlocks has {} blocks", (Object)this.blockManager.numOfUnderReplicatedBlocks());
        this.namesystem.startSecretManagerIfNecessary();
        StartupProgress prog = NameNode.getStartupProgress();
        if (prog.getStatus(Phase.SAFEMODE) != Status.COMPLETE) {
            prog.endStep(Phase.SAFEMODE, STEP_AWAITING_REPORTED_BLOCKS);
            prog.endPhase(Phase.SAFEMODE);
        }
        return true;
    }

    synchronized void incrementSafeBlockCount(int storageNum, BlockInfo storedBlock) {
        int safe;
        assert (this.namesystem.hasWriteLock());
        if (this.status == BMSafeModeStatus.OFF) {
            return;
        }
        int n = safe = storedBlock.isStriped() ? (int)((BlockInfoStriped)storedBlock).getRealDataBlockNum() : this.safeReplication;
        if (storageNum == safe) {
            ++this.blockSafe;
            StartupProgress prog = NameNode.getStartupProgress();
            if (prog.getStatus(Phase.SAFEMODE) != Status.COMPLETE) {
                if (this.awaitingReportedBlocksCounter == null) {
                    this.awaitingReportedBlocksCounter = prog.getCounter(Phase.SAFEMODE, STEP_AWAITING_REPORTED_BLOCKS);
                }
                this.awaitingReportedBlocksCounter.increment();
            }
            this.checkSafeMode();
        }
    }

    synchronized void decrementSafeBlockCount(BlockInfo b) {
        assert (this.namesystem.hasWriteLock());
        if (this.status == BMSafeModeStatus.OFF) {
            return;
        }
        BlockInfo storedBlock = this.blockManager.getStoredBlock(b);
        if (storedBlock.isComplete() && this.blockManager.countNodes(b).liveReplicas() == this.safeReplication - 1) {
            --this.blockSafe;
            assert (this.blockSafe >= 0L);
            this.checkSafeMode();
        }
    }

    void checkBlocksWithFutureGS(BlockListAsLongs.BlockReportReplica brr) {
        assert (this.namesystem.hasWriteLock());
        if (this.status == BMSafeModeStatus.OFF) {
            return;
        }
        if (!this.blockManager.getShouldPostponeBlocksFromFuture() && !this.inRollBack && this.blockManager.isGenStampInFuture(brr)) {
            if (this.blockManager.getBlockIdManager().isStripedBlock(brr)) {
                this.bytesInFutureECBlockGroups.add(brr.getBytesOnDisk());
            } else {
                this.bytesInFutureBlocks.add(brr.getBytesOnDisk());
            }
        }
    }

    long getBytesInFuture() {
        return this.getBytesInFutureBlocks() + this.getBytesInFutureECBlockGroups();
    }

    long getBytesInFutureBlocks() {
        return this.bytesInFutureBlocks.longValue();
    }

    long getBytesInFutureECBlockGroups() {
        return this.bytesInFutureECBlockGroups.longValue();
    }

    void close() {
        assert (this.namesystem.hasWriteLock()) : "Closing bmSafeMode needs write lock!";
        try {
            this.smmthread.interrupt();
            this.smmthread.join(3000L);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    private long timeToLeaveExtension() {
        return this.reachedTime.get() + this.extension - Time.monotonicNow();
    }

    private static boolean isInRollBackMode(HdfsServerConstants.StartupOption option) {
        return option == HdfsServerConstants.StartupOption.ROLLBACK || option == HdfsServerConstants.StartupOption.ROLLINGUPGRADE && option.getRollingUpgradeStartupOption() == HdfsServerConstants.RollingUpgradeStartupOption.ROLLBACK;
    }

    private void initializeReplQueuesIfNecessary() {
        boolean canInitializeReplQueues;
        assert (this.namesystem.hasWriteLock());
        boolean bl = canInitializeReplQueues = this.blockManager.shouldPopulateReplQueues() && this.blockSafe >= this.blockReplQueueThreshold;
        if (canInitializeReplQueues && !this.blockManager.isPopulatingReplQueues() && !this.haEnabled) {
            this.blockManager.initializeReplQueues();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean areThresholdsMet() {
        assert (this.namesystem.hasWriteLock());
        int datanodeNum = this.blockManager.getDatanodeManager().getNumLiveDataNodes();
        BlockManagerSafeMode blockManagerSafeMode = this;
        synchronized (blockManagerSafeMode) {
            return this.blockSafe >= this.blockThreshold && datanodeNum >= this.datanodeThreshold;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doConsistencyCheck() {
        boolean assertsOn = false;
        if (!$assertionsDisabled) {
            assertsOn = true;
            if (!true) {
                throw new AssertionError();
            }
        }
        if (!assertsOn) {
            return;
        }
        int activeBlocks = this.blockManager.getActiveBlockCount();
        BlockManagerSafeMode blockManagerSafeMode = this;
        synchronized (blockManagerSafeMode) {
            if (this.blockTotal != (long)activeBlocks && (this.blockSafe < 0L || this.blockSafe > this.blockTotal)) {
                LOG.warn("SafeMode is in inconsistent filesystem state. BlockManagerSafeMode data: blockTotal={}, blockSafe={}; BlockManager data: activeBlocks={}", new Object[]{this.blockTotal, this.blockSafe, activeBlocks});
            }
        }
    }

    private void reportStatus(String msg, boolean rightNow) {
        assert (this.namesystem.hasWriteLock());
        long curTime = Time.monotonicNow();
        if (!rightNow && curTime - this.lastStatusReport < 20000L) {
            return;
        }
        NameNode.stateChangeLog.info(msg + " \n" + this.getSafeModeTip());
        this.lastStatusReport = curTime;
    }

    private class SafeModeMonitor
    implements Runnable {
        private static final long RECHECK_INTERVAL = 1000L;

        private SafeModeMonitor() {
        }

        @Override
        public void run() {
            while (BlockManagerSafeMode.this.namesystem.isRunning()) {
                try {
                    BlockManagerSafeMode.this.namesystem.writeLock();
                    if (BlockManagerSafeMode.this.status == BMSafeModeStatus.OFF) break;
                    if (this.canLeave()) {
                        BlockManagerSafeMode.this.leaveSafeMode(false);
                        break;
                    }
                }
                finally {
                    BlockManagerSafeMode.this.namesystem.writeUnlock();
                }
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
            if (!BlockManagerSafeMode.this.namesystem.isRunning()) {
                LOG.info("NameNode is being shutdown, exit SafeModeMonitor thread");
            }
        }

        private boolean canLeave() {
            if (BlockManagerSafeMode.this.namesystem.inTransitionToActive()) {
                return false;
            }
            if (BlockManagerSafeMode.this.timeToLeaveExtension() > 0L) {
                BlockManagerSafeMode.this.reportStatus("STATE* Safe mode ON, in safe mode extension.", false);
                return false;
            }
            if (!BlockManagerSafeMode.this.areThresholdsMet()) {
                BlockManagerSafeMode.this.reportStatus("STATE* Safe mode ON, thresholds not met.", false);
                return false;
            }
            return true;
        }
    }

    static enum BMSafeModeStatus {
        PENDING_THRESHOLD,
        EXTENSION,
        OFF;

    }
}

