/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.connector.mongodb.table.converter;

import java.io.Serializable;
import java.math.BigDecimal;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.HashMap;
import org.apache.flink.annotation.Internal;
import org.apache.flink.connector.mongodb.common.utils.MongoConstants;
import org.apache.flink.table.data.DecimalData;
import org.apache.flink.table.data.GenericArrayData;
import org.apache.flink.table.data.GenericMapData;
import org.apache.flink.table.data.GenericRowData;
import org.apache.flink.table.data.RowData;
import org.apache.flink.table.data.StringData;
import org.apache.flink.table.data.TimestampData;
import org.apache.flink.table.types.logical.ArrayType;
import org.apache.flink.table.types.logical.DecimalType;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.table.types.logical.MapType;
import org.apache.flink.table.types.logical.RowType;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.function.SerializableFunction;
import org.bson.BsonArray;
import org.bson.BsonDocument;
import org.bson.BsonType;
import org.bson.BsonValue;
import org.bson.types.Decimal128;

@Internal
public class BsonToRowDataConverters {
    public static BsonToRowDataConverter createConverter(RowType rowType) {
        final SerializableFunction<BsonValue, Object> internalRowConverter = BsonToRowDataConverters.createNullSafeInternalConverter((LogicalType)rowType);
        return new BsonToRowDataConverter(){
            private static final long serialVersionUID = 1L;

            @Override
            public RowData convert(BsonDocument bsonDocument) {
                return (RowData)internalRowConverter.apply((Object)bsonDocument);
            }
        };
    }

    private static SerializableFunction<BsonValue, Object> createNullSafeInternalConverter(LogicalType type) {
        return BsonToRowDataConverters.wrapIntoNullSafeInternalConverter(BsonToRowDataConverters.createInternalConverter(type), type);
    }

    private static SerializableFunction<BsonValue, Object> wrapIntoNullSafeInternalConverter(final SerializableFunction<BsonValue, Object> internalConverter, final LogicalType type) {
        return new SerializableFunction<BsonValue, Object>(){
            private static final long serialVersionUID = 1L;

            public Object apply(BsonValue bsonValue) {
                if (BsonToRowDataConverters.isBsonValueNull(bsonValue) || BsonToRowDataConverters.isBsonDecimalNaN(bsonValue)) {
                    if (type.isNullable()) {
                        return null;
                    }
                    throw new IllegalArgumentException("Unable to convert to <" + type + "> from nullable value " + bsonValue);
                }
                return internalConverter.apply((Object)bsonValue);
            }
        };
    }

    private static boolean isBsonValueNull(BsonValue bsonValue) {
        return bsonValue == null || bsonValue.isNull() || bsonValue.getBsonType() == BsonType.UNDEFINED;
    }

    private static boolean isBsonDecimalNaN(BsonValue bsonValue) {
        return bsonValue.isDecimal128() && bsonValue.asDecimal128().getValue().isNaN();
    }

    private static SerializableFunction<BsonValue, Object> createInternalConverter(final LogicalType type) {
        switch (type.getTypeRoot()) {
            case NULL: {
                return new SerializableFunction<BsonValue, Object>(){
                    private static final long serialVersionUID = 1L;

                    public Object apply(BsonValue bsonValue) {
                        return null;
                    }
                };
            }
            case BOOLEAN: {
                return new SerializableFunction<BsonValue, Object>(){
                    private static final long serialVersionUID = 1L;

                    public Object apply(BsonValue bsonValue) {
                        return BsonToRowDataConverters.convertToBoolean(bsonValue);
                    }
                };
            }
            case INTEGER: 
            case INTERVAL_YEAR_MONTH: {
                return new SerializableFunction<BsonValue, Object>(){
                    private static final long serialVersionUID = 1L;

                    public Object apply(BsonValue bsonValue) {
                        return BsonToRowDataConverters.convertToInt(bsonValue);
                    }
                };
            }
            case BIGINT: 
            case INTERVAL_DAY_TIME: {
                return new SerializableFunction<BsonValue, Object>(){
                    private static final long serialVersionUID = 1L;

                    public Object apply(BsonValue bsonValue) {
                        return BsonToRowDataConverters.convertToLong(bsonValue);
                    }
                };
            }
            case TIMESTAMP_WITHOUT_TIME_ZONE: {
                return new SerializableFunction<BsonValue, Object>(){
                    private static final long serialVersionUID = 1L;

                    public Object apply(BsonValue bsonValue) {
                        return TimestampData.fromLocalDateTime((LocalDateTime)BsonToRowDataConverters.convertToLocalDateTime(bsonValue));
                    }
                };
            }
            case TIMESTAMP_WITH_LOCAL_TIME_ZONE: {
                return new SerializableFunction<BsonValue, Object>(){
                    private static final long serialVersionUID = 1L;

                    public Object apply(BsonValue bsonValue) {
                        return TimestampData.fromInstant((Instant)BsonToRowDataConverters.convertToInstant(bsonValue));
                    }
                };
            }
            case DOUBLE: {
                return new SerializableFunction<BsonValue, Object>(){
                    private static final long serialVersionUID = 1L;

                    public Object apply(BsonValue bsonValue) {
                        return BsonToRowDataConverters.convertToDouble(bsonValue);
                    }
                };
            }
            case CHAR: 
            case VARCHAR: {
                return new SerializableFunction<BsonValue, Object>(){
                    private static final long serialVersionUID = 1L;

                    public Object apply(BsonValue bsonValue) {
                        return StringData.fromString((String)BsonToRowDataConverters.convertToString(bsonValue));
                    }
                };
            }
            case BINARY: 
            case VARBINARY: {
                return new SerializableFunction<BsonValue, Object>(){
                    private static final long serialVersionUID = 1L;

                    public Object apply(BsonValue bsonValue) {
                        return BsonToRowDataConverters.convertToBinary(bsonValue);
                    }
                };
            }
            case DECIMAL: {
                return new SerializableFunction<BsonValue, Object>(){
                    private static final long serialVersionUID = 1L;

                    public Object apply(BsonValue bsonValue) {
                        DecimalType decimalType = (DecimalType)type;
                        BigDecimal decimalValue = BsonToRowDataConverters.convertToBigDecimal(bsonValue);
                        return DecimalData.fromBigDecimal((BigDecimal)decimalValue, (int)decimalType.getPrecision(), (int)decimalType.getScale());
                    }
                };
            }
            case ROW: {
                return BsonToRowDataConverters.createRowConverter((RowType)type);
            }
            case ARRAY: {
                return BsonToRowDataConverters.createArrayConverter((ArrayType)type);
            }
            case MAP: {
                return BsonToRowDataConverters.createMapConverter((MapType)type);
            }
        }
        throw new UnsupportedOperationException("Unsupported type: " + type);
    }

    private static SerializableFunction<BsonValue, Object> createRowConverter(RowType rowType) {
        final SerializableFunction[] fieldConverters = (SerializableFunction[])rowType.getFields().stream().map(RowType.RowField::getType).map(BsonToRowDataConverters::createNullSafeInternalConverter).toArray(SerializableFunction[]::new);
        final int arity = rowType.getFieldCount();
        final String[] fieldNames = rowType.getFieldNames().toArray(new String[0]);
        return new SerializableFunction<BsonValue, Object>(){
            private static final long serialVersionUID = 1L;

            public Object apply(BsonValue bsonValue) {
                if (!bsonValue.isDocument()) {
                    throw new IllegalArgumentException("Unable to convert to rowType from unexpected value '" + bsonValue + "' of type " + bsonValue.getBsonType());
                }
                BsonDocument document = bsonValue.asDocument();
                GenericRowData row = new GenericRowData(arity);
                for (int i = 0; i < arity; ++i) {
                    String fieldName = fieldNames[i];
                    BsonValue fieldValue = document.get((Object)fieldName);
                    Object convertedField = fieldConverters[i].apply((Object)fieldValue);
                    row.setField(i, convertedField);
                }
                return row;
            }
        };
    }

    private static SerializableFunction<BsonValue, Object> createArrayConverter(ArrayType arrayType) {
        final SerializableFunction<BsonValue, Object> elementConverter = BsonToRowDataConverters.createNullSafeInternalConverter(arrayType.getElementType());
        return new SerializableFunction<BsonValue, Object>(){
            private static final long serialVersionUID = 1L;

            public Object apply(BsonValue bsonValue) {
                if (!bsonValue.isArray()) {
                    throw new IllegalArgumentException("Unable to convert to arrayType from unexpected value '" + bsonValue + "' of type " + bsonValue.getBsonType());
                }
                BsonArray in = bsonValue.asArray();
                Object[] elementArray = new Object[in.size()];
                for (int i = 0; i < in.size(); ++i) {
                    elementArray[i] = elementConverter.apply(in.get(i));
                }
                return new GenericArrayData(elementArray);
            }
        };
    }

    private static SerializableFunction<BsonValue, Object> createMapConverter(MapType mapType) {
        LogicalType keyType = mapType.getKeyType();
        Preconditions.checkArgument((boolean)keyType.supportsInputConversion(String.class));
        LogicalType valueType = mapType.getValueType();
        final SerializableFunction<BsonValue, Object> valueConverter = BsonToRowDataConverters.createNullSafeInternalConverter(valueType);
        return new SerializableFunction<BsonValue, Object>(){
            private static final long serialVersionUID = 1L;

            public Object apply(BsonValue bsonValue) {
                if (!bsonValue.isDocument()) {
                    throw new IllegalArgumentException("Unable to convert to rowType from unexpected value '" + bsonValue + "' of type " + bsonValue.getBsonType());
                }
                BsonDocument document = bsonValue.asDocument();
                HashMap<StringData, Object> map = new HashMap<StringData, Object>();
                for (String key : document.keySet()) {
                    map.put(StringData.fromString((String)key), valueConverter.apply((Object)document.get((Object)key)));
                }
                return new GenericMapData(map);
            }
        };
    }

    private static boolean convertToBoolean(BsonValue bsonValue) {
        if (bsonValue.isBoolean()) {
            return bsonValue.asBoolean().getValue();
        }
        throw new IllegalArgumentException("Unable to convert to boolean from unexpected value '" + bsonValue + "' of type " + bsonValue.getBsonType());
    }

    private static int convertToInt(BsonValue bsonValue) {
        if (bsonValue.isInt32()) {
            return bsonValue.asNumber().intValue();
        }
        throw new IllegalArgumentException("Unable to convert to integer from unexpected value '" + bsonValue + "' of type " + bsonValue.getBsonType());
    }

    private static long convertToLong(BsonValue bsonValue) {
        if (bsonValue.isInt64()) {
            return bsonValue.asNumber().longValue();
        }
        throw new IllegalArgumentException("Unable to convert to long from unexpected value '" + bsonValue + "' of type " + bsonValue.getBsonType());
    }

    private static double convertToDouble(BsonValue bsonValue) {
        if (bsonValue.isDouble()) {
            return bsonValue.asNumber().doubleValue();
        }
        throw new IllegalArgumentException("Unable to convert to double from unexpected value '" + bsonValue + "' of type " + bsonValue.getBsonType());
    }

    private static Instant convertToInstant(BsonValue bsonValue) {
        if (bsonValue.isTimestamp()) {
            return Instant.ofEpochSecond(bsonValue.asTimestamp().getTime());
        }
        if (bsonValue.isDateTime()) {
            return Instant.ofEpochMilli(bsonValue.asDateTime().getValue());
        }
        throw new IllegalArgumentException("Unable to convert to Instant from unexpected value '" + bsonValue + "' of type " + bsonValue.getBsonType());
    }

    private static LocalDateTime convertToLocalDateTime(BsonValue bsonValue) {
        Instant instant;
        if (bsonValue.isTimestamp()) {
            instant = Instant.ofEpochSecond(bsonValue.asTimestamp().getTime());
        } else if (bsonValue.isDateTime()) {
            instant = Instant.ofEpochMilli(bsonValue.asDateTime().getValue());
        } else {
            throw new IllegalArgumentException("Unable to convert to LocalDateTime from unexpected value '" + bsonValue + "' of type " + bsonValue.getBsonType());
        }
        return Timestamp.from(instant).toLocalDateTime();
    }

    private static byte[] convertToBinary(BsonValue bsonValue) {
        if (bsonValue.isBinary()) {
            return bsonValue.asBinary().getData();
        }
        throw new IllegalArgumentException("Unsupported BYTES value type: " + bsonValue.getClass().getSimpleName());
    }

    private static String convertToString(BsonValue bsonValue) {
        if (bsonValue.isString()) {
            return bsonValue.asString().getValue();
        }
        if (bsonValue.isObjectId()) {
            return bsonValue.asObjectId().getValue().toHexString();
        }
        return new BsonDocument("_value", bsonValue).toJson(MongoConstants.DEFAULT_JSON_WRITER_SETTINGS);
    }

    private static BigDecimal convertToBigDecimal(BsonValue bsonValue) {
        if (bsonValue.isDecimal128()) {
            Decimal128 decimal128Value = bsonValue.asDecimal128().decimal128Value();
            if (decimal128Value.isFinite()) {
                return bsonValue.asDecimal128().decimal128Value().bigDecimalValue();
            }
            throw new IllegalArgumentException("Unable to convert infinite bson decimal to Decimal type.");
        }
        throw new IllegalArgumentException("Unable to convert to decimal from unexpected value '" + bsonValue + "' of type " + bsonValue.getBsonType());
    }

    @FunctionalInterface
    public static interface BsonToRowDataConverter
    extends Serializable {
        public RowData convert(BsonDocument var1);
    }
}

