package com.aliyun.sdk.gateway.pop.interceptor.httpRequest;

import com.aliyun.auth.signature.SignerParams;
import com.aliyun.core.http.*;
import com.aliyun.sdk.gateway.pop.auth.SignatureVersion;
import com.aliyun.sdk.gateway.pop.auth.V3SignatureComposer;
import com.aliyun.auth.credentials.ICredential;
import com.aliyun.auth.credentials.provider.AnonymousCredentialProvider;
import com.aliyun.sdk.gateway.pop.auth.signer.PopSigner;
import com.aliyun.core.logging.ClientLogger;
import com.aliyun.core.utils.AttributeMap;
import com.aliyun.core.utils.ParseUtil;
import com.aliyun.core.utils.StringUtils;
import darabonba.core.TeaConfiguration;
import darabonba.core.TeaPair;
import darabonba.core.TeaRequest;
import darabonba.core.client.ClientConfiguration;
import darabonba.core.client.ClientOption;
import darabonba.core.interceptor.HttpRequestInterceptor;
import darabonba.core.interceptor.InterceptorContext;
import darabonba.core.utils.CommonUtil;
import darabonba.core.utils.ModelUtil;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static com.aliyun.core.utils.EncodeUtil.hexEncode;

public class V3HttpReqInterceptor implements HttpRequestInterceptor {
    private final ClientLogger logger = new ClientLogger(V3HttpReqInterceptor.class);

    @Override
    public HttpRequest modifyHttpRequest(InterceptorContext context, AttributeMap attributes) {
        logger.info("Request pre-process begin.");
        TeaRequest request = context.teaRequest();
        HttpRequest httpRequest = context.httpRequest();
        TeaConfiguration configuration = context.configuration();
        ClientConfiguration clientConfiguration = configuration.clientConfiguration();
        PopSigner signer = (PopSigner) clientConfiguration.option(ClientOption.SIGNER);
        if (signer.getSignerVersion() != SignatureVersion.V3) {
            return httpRequest;
        }
        HttpMethod method = Optional.ofNullable(configuration.method()).orElseGet(request::method);
        Map<String, String> headers = CommonUtil.merge(String.class, CommonUtil.buildMap(
                new TeaPair("x-acs-signature-nonce", CommonUtil.getNonce()),
                new TeaPair("x-acs-date", CommonUtil.getTimestamp())
                ),
                request.headers().toMap()
        );
        Map<String, String> query = request.query();
        byte[] body = null;
        String hashedRequestPayload = hexEncode(signer.hash(StringUtils.toBytes("")));
        if (!CommonUtil.isUnset(request.stream())) {
            body = ParseUtil.readAsBytes(request.stream());
            hashedRequestPayload = hexEncode(signer.hash(body));
            headers.put("content-type", FormatType.RAW);
        } else if (!CommonUtil.isUnset(request.body())) {
            if (request.reqBodyType().equals(BodyType.JSON)) {
                if (request.body() instanceof byte[]) {
                    body = (byte[]) request.body();
                } else {
                    body = StringUtils.toBytes(ParseUtil.toJSONString(request.body()));
                }
                hashedRequestPayload = hexEncode(signer.hash(body));
                headers.put("content-type", FormatType.JSON);
            } else {
                if (request.body() instanceof byte[]) {
                    body = (byte[]) request.body();
                } else {
                    body = StringUtils.toBytes(Objects.requireNonNull(ModelUtil.toFormString(ModelUtil.query(CommonUtil.assertAsMap(request.body())))));
                }
                hashedRequestPayload = hexEncode(signer.hash(body));
                headers.put("content-type", FormatType.FORM);
            }
        }
        headers.put(signer.getContent(), hashedRequestPayload);
        if (!(configuration.credentialProvider() instanceof AnonymousCredentialProvider)) {
            ICredential credential = configuration.credentialProvider().getCredentials();
            String accessKeyId = credential.accessKeyId();
            String securityToken = credential.securityToken();
            if (!StringUtils.isEmpty(securityToken)) {
                headers.put("x-acs-accesskey-id", accessKeyId);
                headers.put("x-acs-security-token", securityToken);
            }
            String strToSign = V3SignatureComposer.composeStringToSign(method, query, headers, request.pathname(), hashedRequestPayload);
            strToSign = signer.getSignerName().algorithmName() + "\n" + hexEncode(signer.hash(StringUtils.toBytes(strToSign)));
            SignerParams params = SignerParams.create();
            params.setCredentials(credential);
            headers.put("Authorization", signer.getSignerName().algorithmName() + " Credential=" + accessKeyId
                    + ",SignedHeaders=" + V3SignatureComposer.buildSignedHeaders(headers)
                    + ",Signature=" + signer.signString(strToSign, params));
        }
        HttpHeaders httpHeaders = new HttpHeaders(headers);
        httpRequest = new HttpRequest(method,
                ModelUtil.composeUrl(configuration.endpoint(), query, configuration.protocol(), request.pathname()));
        httpRequest.setHeaders(httpHeaders);
        if (!CommonUtil.isUnset(body)) {
            httpRequest.setBody(body);
        }
        return httpRequest;
    }
}
